(ns ^:figwheel-always com.kaicode.infamous
  (:require [com.kaicode.Famous]
            [cljs.core.async :refer [put! chan]]
            [datascript :as d]))

(enable-console-print!)

(defonce famous js/famous)
(defonce FamousEngine (.. famous -core -FamousEngine))
(defonce Node (.. famous -core -Node))
(defonce GestureHandler (.. famous -components -GestureHandler))
(defonce PhysicsEngine (.. famous -physics -PhysicsEngine))

(defonce physics (.. famous -physics))
(defonce math (.. famous -math))

(defonce FamousBox (.. physics -Box))
(defonce Spring (.. physics -Spring))
(defonce RotationalSpring (.. physics -RotationalSpring))
(defonce Quaternion (.. math -Quaternion))
(defonce Vec3 (.. math -Vec3))

(defn get-node-by-id [id]
      (ffirst (d/q '[:find (pull ?node [*]) :in $ ?id :where [?node :node/id ?id]] @conn id)))

(defn get-famous-components [node]
      (.. (:node/famous-node node) getComponents))

(defn get-famous-component-by-type-name [node component-type-name]
      (let [famous-components (get-famous-components node)]
           (first (filter (fn [component]
                              (let [cn (.. component -constructor -name)]
                                   (= component-type-name cn)))
                          famous-components))))

(defonce famous-components {:DOMElement (.. famous -domRenderables -DOMElement)
                            :Align      (.. famous -components -Align)})

(defn- famous-compare [x y]
       "Compare famous javascript objects. See https://github.com/tonsky/datascript/issues/69"
       (let [str-x (.. js/JSON (stringify x))
             str-y (.. js/JSON (stringify y))]
            (compare str-x str-y)))

(extend-protocol IComparable
                 FamousBox (^number -compare [x y]
                                             (famous-compare x y))
                 Vec3 (^number -compare [x y]
                                        (famous-compare x y))
                 Quaternion (^number -compare [x y]
                                              (famous-compare x y))
                 Spring (^number -compare [x y]
                                          (famous-compare x y))
                 RotationalSpring (^number -compare [x y]
                                                    (famous-compare x y))
                 Node (^number -compare [x y]
                                        (famous-compare x y)))

(def schema {:node/id         {:db/unique :db.unique/identity}
             :node/children   {:db/cardinality :db.cardinality/many
                               :db/isComponent true
                               :db/valueType   :db.type/ref}
             :node/components {:db/cardinality :db.cardinality/many
                               :db/isComponent true
                               :db/valueType   :db.type/ref}
             :node/physics    {:db/cardinality :db.cardinality/one
                               :db/isComponent true
                               :db/valueType   :db.type/ref}})
(def conn (d/create-conn schema))

(defn save [scene-graph]
      (d/transact! conn [scene-graph]))

(defn events->chan
      "Given a node and event type return a channel of
      observed events. Can supply the channel to receive events as third
      optional argument."
      ([node event payload-function] (events->chan node event (chan) payload-function))
      ([node event c payload-function]
        (let [node (if (map? node)
                     (:node/famous-node node)
                     node)]
             (.. (GestureHandler. node) (on event (fn []
                                                      (put! c (or (payload-function) event))))))
        c))

(defn- create-component [component-descriptor famous-node]
       (let [component-type (:component/type component-descriptor)]
            (if (keyword? component-type)
              (let [component-constructor (famous-components component-type)
                    component (component-constructor. famous-node)
                    properties (dissoc component-descriptor :component/type :db/id)]
                   (doseq [p properties
                           :let [name (name (first p))
                                 value (second p)]]
                          (cond
                            (= name "content") (.. component (setContent value))
                            (= name "classes") (doseq [clz value]
                                                      (.. component (addClass clz)))
                            :else (do
                                    (.. component (setProperty name value)))))
                   component)
              (let [component (clj->js component-descriptor)]
                   (.. famous-node (addComponent component))
                   component))))




(defn- attach-famous-node-to-scene-graph [node]
       (let [famous-node (Node.)
             size-mode (clj->js (:node/size-mode node))
             absolute-size (clj->js (:node/absolute-size node))
             align (clj->js (:node/align node))
             position (clj->js (:node/position node))
             mount-point (clj->js (:node/mount-point node))
             origin (clj->js (:node/origin node))
             proportional-size (clj->js (:node/proportional-size node))
             differential-size (clj->js (:node/differential-size node))
             ]
            (.apply (.-setSizeMode famous-node) famous-node size-mode)
            (.apply (.-setAbsoluteSize famous-node) famous-node absolute-size)
            (.apply (.-setAlign famous-node) famous-node align)
            (.apply (.-setPosition famous-node) famous-node position)
            (.apply (.-setMountPoint famous-node) famous-node mount-point)
            (.apply (.-setOrigin famous-node) famous-node origin)
            (.apply (.-setProportionalSize famous-node) famous-node proportional-size)
            (.apply (.-setDifferentialSize famous-node) famous-node differential-size)

            (doseq [child-node (:node/children node)
                    :let [a-child-node (attach-famous-node-to-scene-graph child-node)
                          a-famous-child-node (:node/famous-node a-child-node)]]
                   (.. famous-node (addChild a-famous-child-node)))

            (d/transact! conn [{:db/id            (:db/id node)
                                :node/famous-node famous-node}])

            (update-in node [:node/famous-node] #(identity famous-node))))


(defn- find-nodes-with-physics []
       (map #(first %) (d/q '[:find (pull ?node [*]) :where [?node :node/physics _]] @conn)))


(defn- find-nodes-with-components []
       (map #(first %) (d/q '[:find (pull ?node [*]) :where [?node :node/components _]] @conn)))

(defn render-scene-graph [root-id]
      (let [simulation (PhysicsEngine.)
            scene-graph (attach-famous-node-to-scene-graph (get-node-by-id root-id))
            root-node (:node/famous-node scene-graph)
            physics-nodes (find-nodes-with-physics)
            nodes-with-components (find-nodes-with-components)
            context (.. FamousEngine (createScene "body"))]
           (.. context (addChild root-node))

           (doseq [{physics :node/physics} physics-nodes]
                  (.. simulation (add (:box physics) (:spring physics) (:rotational-spring physics))))

           (doseq [{components :node/components :as node} nodes-with-components
                   :let [famous-node (:node/famous-node node)]]
                  (doseq [component-descriptor components]
                         (create-component component-descriptor famous-node)
                         )
                  )

           (.. FamousEngine (requestUpdate (clj->js {:onUpdate (fn [time]
                                                                   (.. simulation (update time))
                                                                   (doseq [pn physics-nodes
                                                                           :let [famous-node (:node/famous-node pn)
                                                                                 physics (:node/physics pn)
                                                                                 physics-transform (.. simulation (getTransform (:box physics)))
                                                                                 p (.. physics-transform -position)
                                                                                 r (.. physics-transform -rotation)]]
                                                                          (.. famous-node
                                                                              (setPosition (* 0 1446) 0 0)
                                                                              (setRotation (nth r 0) (nth r 1) (nth r 2) (nth r 3))))

                                                                   (this-as this
                                                                            (.. FamousEngine (requestUpdateOnNextTick this)))
                                                                   )})))

           (.. FamousEngine init)))
