goog.provide('Cl.SimplexSolver');

goog.require('Cl');
goog.require('Cl.CL');
goog.require('Cl.DummyVariable');
goog.require('Cl.HashSet');
goog.require('Cl.HashTable');
goog.require('Cl.LinearExpression');
goog.require('Cl.ObjectiveVariable');
goog.require('Cl.SlackVariable');
goog.require('Cl.StayConstraint');
goog.require('Cl.Tableau');
goog.require('Cl.Tableau');
goog.require('Cl.errors');

goog.scope(function() {
  var CL = Cl.CL;
  var HashSet = Cl.HashSet;
  var HashTable = Cl.HashTable;
;
;
;
;
;
;
;
;
;
;
;
/**
 * @constructor
 * @extends {Cl.Tableau}
 */
Cl.SimplexSolver = function() {
  Cl.Tableau.call(this);
  this._stayMinusErrorVars = new Array();
  this._stayPlusErrorVars = new Array();
  this._errorVars = new HashTable();
  this._markerVars = new HashTable();
  this._resolve_pair = new Array(0, 0);
  this._objective = new Cl.ObjectiveVariable("Z");
  this._editVarMap = new HashTable();
  this._slackCounter = 0;
  this._artificialCounter = 0;
  this._dummyCounter = 0;
  this._epsilon = 1e-8;
  this._fOptimizeAutomatically = true;
  this._fNeedsSolving = false;
  this._rows = new HashTable();
  this._rows.put(this._objective, new Cl.LinearExpression());
  this._stkCedcns = new Array();
  this._stkCedcns.push(0);
};
goog.inherits(Cl.SimplexSolver, Cl.Tableau);

Cl.SimplexSolver.prototype.addLowerBound = function(v, lower) {
  var cn;
  cn = new Cl.LinearInequality(v, CL.GEQ, new Cl.LinearExpression(lower));
  return this.addConstraint(cn);
};
Cl.SimplexSolver.prototype.addUpperBound = function(v, upper) {
  var cn;
  cn = new Cl.LinearInequality(v, CL.LEQ, new Cl.LinearExpression(upper));
  return this.addConstraint(cn);
};
Cl.SimplexSolver.prototype.addBounds = function(v, lower, upper) {
  this.addLowerBound(v, lower);
  this.addUpperBound(v, upper);
  return this;
};
Cl.SimplexSolver.prototype.addConstraint = function(cn) {
  var clvEminus, clvEplus, eplus_eminus, expr, i, prevEConstant;
  eplus_eminus = new Array(2);
  prevEConstant = new Array(1);
  expr = this.newExpression(cn, eplus_eminus, prevEConstant);
  prevEConstant = prevEConstant[0];
  if (!this.tryAddingDirectly(expr)) {
    this.addWithArtificialVariable(expr);
  }
  this._fNeedsSolving = true;
  if (cn.isEditConstraint()) {
    i = this._editVarMap.size();
    clvEplus = eplus_eminus[0], clvEminus = eplus_eminus[1];
    this._editVarMap.put(cn.variable(), new Cl.EditInfo(cn, clvEplus, clvEminus, prevEConstant, i));
  }
  if (this._fOptimizeAutomatically) {
    this.optimize(this._objective);
    this.setExternalVariables();
  }
  cn.addedTo(this);
  return this;
};
Cl.SimplexSolver.prototype.addConstraintNoException = function(cn) {
  try {
    this.addConstraint(cn);
    return true;
  } catch (e) {
    return false;
  }
};
Cl.SimplexSolver.prototype.addEditVar = function(v, strength) {
  var cnEdit;
  strength = strength || Cl.Strength.strong;
  cnEdit = new Cl.EditConstraint(v, strength);
  this.addConstraint(cnEdit);
  return this;
};
Cl.SimplexSolver.prototype.removeEditVar = function(v) {
  var cei, cn;
  cei = this._editVarMap.get(v);
  cn = cei.Constraint();
  this.removeConstraint(cn);
  return this;
};
Cl.SimplexSolver.prototype.beginEdit = function() {
  CL.Assert(this._editVarMap.size() > 0, "_editVarMap.size() > 0");
  this._infeasibleRows.clear();
  this.resetStayConstants();
  this._stkCedcns.push(this._editVarMap.size());
  return this;
};
Cl.SimplexSolver.prototype.endEdit = function() {
  var n;
  CL.Assert(this._editVarMap.size() > 0, "_editVarMap.size() > 0");
  this.resolve();
  this._stkCedcns.pop();
  n = this._stkCedcns[this._stkCedcns.length - 1];
  this.removeEditVarsTo(n);
  return this;
};
Cl.SimplexSolver.prototype.removeAllEditVars = function() {
  return this.removeEditVarsTo(0);
};
Cl.SimplexSolver.prototype.removeEditVarsTo = function(n) {
  try {
    this._editVarMap.each(goog.bind(function(v, cei) {
      if (cei.Index() >= n) {
        return this.removeEditVar(v);
      }
    }, this));
    CL.Assert(this._editVarMap.size() === n, "_editVarMap.size() == n");
    return this;
  } catch (e) {
    throw new Cl.errors.InternalError("Constraint not found in removeEditVarsTo");
  }
};
Cl.SimplexSolver.prototype.addPointStays = function(listOfPoints) {
  var i, multiplier, weight;
  weight = 1.0;
  multiplier = 2.0;
  i = 0;
  while (i < listOfPoints.length) {
    this.addPointStay(listOfPoints[i], weight);
    weight *= multiplier;
    i++;
  }
  return this;
};
Cl.SimplexSolver.prototype.addPointStay = function(a1, a2, a3) {
  var clp, vx, vy, weight;
  if (a1 instanceof Cl.Point) {
    clp = a1;
    weight = a2;
    this.addStay(clp.X(), Cl.Strength.weak, weight || 1.0);
    this.addStay(clp.Y(), Cl.Strength.weak, weight || 1.0);
  } else {
    vx = a1;
    vy = a2;
    weight = a3;
    this.addStay(vx, Cl.Strength.weak, weight || 1.0);
    this.addStay(vy, Cl.Strength.weak, weight || 1.0);
  }
  return this;
};
Cl.SimplexSolver.prototype.addStay = function(v, strength, weight) {
  var cn;
  cn = new Cl.StayConstraint(v, strength || Cl.Strength.weak, weight || 1.0);
  return this.addConstraint(cn);
};
Cl.SimplexSolver.prototype.removeConstraint = function(cn) {
  this.removeConstraintInternal(cn);
  cn.removedFrom(this);
  return this;
};
Cl.SimplexSolver.prototype.removeConstraintInternal = function(cn) {
  var cei, clv, clvEditMinus, cnEdit, col, eVars, exitVar, expr, i, marker, minRatio, zRow;
  this._fNeedsSolving = true;
  this.resetStayConstants();
  zRow = this.rowExpression(this._objective);
  eVars = this._errorVars.get(cn);
  if (eVars != null) {
    eVars.each(goog.bind(function(clv) {
      var expr;
      expr = this.rowExpression(clv);
      if (expr == null) {
        return zRow.addVariable(clv, -cn.weight() * cn.strength().symbolicWeight().toDouble(), this._objective, this);
      } else {
        return zRow.addExpression(expr, -cn.weight() * cn.strength().symbolicWeight().toDouble(), this._objective, this);
      }
    }, this));
  }
  marker = this._markerVars.remove(cn);
  if (marker == null) {
    throw new Cl.errors.ConstraintNotFound();
  }
  if (this.rowExpression(marker) == null) {
    col = this._columns.get(marker);
    exitVar = null;
    minRatio = 0.0;
    col.each(goog.bind(function(v) {
      var coeff, expr, r;
      if (v.isRestricted()) {
        expr = this.rowExpression(v);
        coeff = expr.coefficientFor(marker);
        if (this.fTraceOn) {
          this.traceprint("Marker " + marker + "'s coefficient in " + expr + " is " + coeff);
        }
        if (coeff < 0.0) {
          r = -expr.constant() / coeff;
          if (!(exitVar != null) || r < minRatio || (Cl.approx(r, minRatio) && v.hashCode() < exitVar.hashCode())) {
            minRatio = r;
            return exitVar = v;
          }
        }
      }
    }, this));
    if (exitVar == null) {
      if (CL.fTraceOn) {
        CL.traceprint("exitVar is still null");
      }
      col.each(goog.bind(function(v) {
        var coeff, expr, r;
        if (v.isRestricted()) {
          expr = this.rowExpression(v);
          coeff = expr.coefficientFor(marker);
          r = expr.constant() / coeff;
          if (!(exitVar != null) || r < minRatio) {
            minRatio = r;
            return exitVar = v;
          }
        }
      }, this));
    }
    if (exitVar == null) {
      if (col.size() === 0) {
        this.removeColumn(marker);
      } else {
        col.each(goog.bind(function(v) {
          if (v !== this._objective) {
            return exitVar = v;
          }
        }, this));
      }
    }
    if (exitVar != null) {
      this.pivot(marker, exitVar);
    }
  }
  if (this.rowExpression(marker) != null) {
    expr = this.removeRow(marker);
    expr = null;
  }
  if (eVars != null) {
    eVars.each(goog.bind(function(v) {
      if (v !== marker) {
        return this.removeColumn(v);
      }
    }, this));
  }
  if (cn.isStayConstraint()) {
    if (eVars != null) {
      i = 0;
      while (i < this._stayPlusErrorVars.length) {
        eVars.remove(this._stayPlusErrorVars[i]);
        eVars.remove(this._stayMinusErrorVars[i]);
        i++;
      }
    }
  } else if (cn.isEditConstraint()) {
    CL.Assert(eVars != null, "eVars != null");
    cnEdit = cn;
    clv = cnEdit.variable();
    cei = this._editVarMap.get(clv);
    clvEditMinus = cei.ClvEditMinus();
    this.removeColumn(clvEditMinus);
    this._editVarMap.remove(clv);
  }
  if (eVars != null) {
    this._errorVars.remove(eVars);
  }
  if (this._fOptimizeAutomatically) {
    this.optimize(this._objective);
    this.setExternalVariables();
  }
  return this;
};
Cl.SimplexSolver.prototype.reset = function() {
  throw new Cl.errors.InternalError("reset not implemented");
};
Cl.SimplexSolver.prototype.resolveArray = function(newEditConstants) {
  this._editVarMap.each(goog.bind(function(v, cei) {
    var i;
    i = cei.Index();
    if (i < newEditConstants.length) {
      return this.suggestValue(v, newEditConstants[i]);
    }
  }, this));
  return this.resolve();
};
Cl.SimplexSolver.prototype.resolvePair = function(x, y) {
  this._resolve_pair[0] = x;
  this._resolve_pair[1] = y;
  return this.resolveArray(this._resolve_pair);
};
Cl.SimplexSolver.prototype.resolve = function() {
  this.dualOptimize();
  this.setExternalVariables();
  this._infeasibleRows.clear();
  this.resetStayConstants();
  return this;
};
Cl.SimplexSolver.prototype.suggestValue = function(v, x) {
  var cei, clvEditMinus, clvEditPlus, delta, i;
  cei = this._editVarMap.get(v);
  if (cei == null) {
    throw new Error("suggestValue for variable " + v + ", but var is not an edit variable\n");
    throw new Cl.errors.Error();
  }
  i = cei.Index();
  clvEditPlus = cei.ClvEditPlus();
  clvEditMinus = cei.ClvEditMinus();
  delta = x - cei.PrevEditConstant();
  cei.SetPrevEditConstant(x);
  this.deltaEditConstant(delta, clvEditPlus, clvEditMinus);
  return this;
};
Cl.SimplexSolver.prototype.setAutosolve = function(f) {
  this._fOptimizeAutomatically = f;
  return this;
};
Cl.SimplexSolver.prototype.FIsAutosolving = function() {
  return this._fOptimizeAutomatically;
};
Cl.SimplexSolver.prototype.solve = function() {
  if (this._fNeedsSolving) {
    this.optimize(this._objective);
    this.setExternalVariables();
  }
  return this;
};
Cl.SimplexSolver.prototype.setEditedValue = function(v, n) {
  if (!this.FContainsVariable(v)) {
    v.change_value(n);
    return this;
  }
  if (!Cl.approx(n, v.value())) {
    this.addEditVar(v);
    this.beginEdit();
    try {
      this.suggestValue(v, n);
    } catch (e) {
      throw new Cl.errors.InternalError("Error in setEditedValue");
    }
    this.endEdit();
  }
  return this;
};
Cl.SimplexSolver.prototype.FContainsVariable = function(v) {
  return this.columnsHasKey(v) || (this.rowExpression(v) != null);
};
Cl.SimplexSolver.prototype.addVar = function(v) {
  if (!this.FContainsVariable(v)) {
    try {
      this.addStay(v);
    } catch (e) {
      throw new Cl.errorsInternalError("Error in addVar -- required failure is impossible");
    }
  }
  return this;
};
Cl.SimplexSolver.prototype.getInternalInfo = function() {
  retstr += "\nSolver info:\n";
  retstr += "Stay Error Variables: ";
  retstr += this._stayPlusErrorVars.length + this._stayMinusErrorVars.length;
  retstr += " (" + this._stayPlusErrorVars.length + " +, ";
  retstr += this._stayMinusErrorVars.length + " -)\n";
  retstr += "Edit Variables: " + this._editVarMap.size();
  retstr += "\n";
  return retstr;
};
Cl.SimplexSolver.prototype.getDebugInfo = function() {
  var bstr;
  bstr = this.toString();
  bstr += this.getInternalInfo();
  bstr += "\n";
  return bstr;
};
Cl.SimplexSolver.prototype.toString = function() {
  var bstr;
  bstr = Cl.SimplexSolver.superClass_.toString.call(this);
  bstr += "\n_stayPlusErrorVars: ";
  bstr += "[" + this._stayPlusErrorVars + "]";
  bstr += "\n_stayMinusErrorVars: ";
  bstr += "[" + this._stayMinusErrorVars + "]";
  bstr += "\n";
  bstr += "_editVarMap:\n" + CL.hashToString(this._editVarMap);
  bstr += "\n";
  return bstr;
};
Cl.SimplexSolver.prototype.getConstraintMap = function() {
  return this._markerVars;
};
Cl.SimplexSolver.prototype.addWithArtificialVariable = function(expr) {
  var av, az, azRow, azTableauRow, e, entryVar;
  av = new Cl.SlackVariable(++this._artificialCounter, "a");
  az = new Cl.ObjectiveVariable("az");
  azRow = expr.clone();
  this.addRow(az, azRow);
  this.addRow(av, expr);
  this.optimize(az);
  azTableauRow = this.rowExpression(az);
  if (!Cl.approx(azTableauRow.constant(), 0.0)) {
    this.removeRow(az);
    this.removeColumn(av);
    throw new Cl.errors.RequiredFailure();
  }
  e = this.rowExpression(av);
  if (e != null) {
    if (e.isConstant()) {
      this.removeRow(av);
      this.removeRow(az);
      return;
    }
    entryVar = e.anyPivotableVariable();
    this.pivot(entryVar, av);
  }
  CL.Assert(!(this.rowExpression(av) != null), "rowExpression(av) == null");
  this.removeColumn(av);
  return this.removeRow(az);
};
Cl.SimplexSolver.prototype.tryAddingDirectly = function(expr) {
  var subject;
  subject = this.chooseSubject(expr);
  if (subject == null) {
    return false;
  }
  expr.newSubject(subject);
  if (this.columnsHasKey(subject)) {
    this.substituteOut(subject, expr);
  }
  this.addRow(subject, expr);
  return true;
};
Cl.SimplexSolver.prototype.chooseSubject = function(expr) {
  var coeff, foundNewRestricted, foundUnrestricted, retval, subject, terms;
  subject = null;
  foundUnrestricted = false;
  foundNewRestricted = false;
  terms = expr.terms();
  retval = null;
  terms.each(goog.bind(function(v, c) {
    var col;
    if (foundUnrestricted) {
      if (!v.isRestricted() && !this.columnsHasKey(v)) {
        retval = v;
      }
    } else if (v.isRestricted()) {
      if (!foundNewRestricted && !v.isDummy() && c < 0.0) {
        col = this._columns.get(v);
        if (!(col != null) || (col.size() === 1 && this.columnsHasKey(this._objective))) {
          subject = v;
          return foundNewRestricted = true;
        }
      }
    } else {
      subject = v;
      return foundUnrestricted = true;
    }
  }, this));
  if (retval != null) {
    return retval;
  }
  if (subject != null) {
    return subject;
  }
  coeff = 0;
  if (terms.keys().some(function(v) {
    return !v.isDummy();
  })) {
    return null;
  }
  terms.each(goog.bind(function(v, c) {
    if (!this.columnsHasKey(v)) {
      subject = v;
      return coeff = c;
    }
  }, this));
  if (!Cl.approx(expr.constant(), 0)) {
    throw new Cl.errors.RequiredFailure();
  }
  if (coeff > 0) {
    expr.multiplyMe(-1);
  }
  return subject;
};
Cl.SimplexSolver.prototype.deltaEditConstant = function(delta, plusErrorVar, minusErrorVar) {
  var columnVars, exprMinus, exprPlus;
  if (CL.fTraceOn) {
    CL.fnenterprint("deltaEditConstant :" + delta + ", " + plusErrorVar + ", " + minusErrorVar);
  }
  exprPlus = this.rowExpression(plusErrorVar);
  if (exprPlus != null) {
    exprPlus.incrementConstant(delta);
    if (exprPlus.constant() < 0.0) {
      this._infeasibleRows.add(plusErrorVar);
    }
    return;
  }
  exprMinus = this.rowExpression(minusErrorVar);
  if (exprMinus != null) {
    exprMinus.incrementConstant(-delta);
    if (exprMinus.constant() < 0.0) {
      this._infeasibleRows.add(minusErrorVar);
    }
    return;
  }
  columnVars = this._columns.get(minusErrorVar);
  if (!columnVars) {
    throw new Error("columnVars is null -- tableau is:\n" + this);
  }
  return columnVars.each(goog.bind(function(basicVar) {
    var c, expr;
    expr = this.rowExpression(basicVar);
    c = expr.coefficientFor(minusErrorVar);
    expr.incrementConstant(c * delta);
    if (basicVar.isRestricted() && expr.constant() < 0.0) {
      return this._infeasibleRows.add(basicVar);
    }
  }, this));
};
Cl.SimplexSolver.prototype.dualOptimize = function() {
  var entryVar, exitVar, expr, ratio, terms, zRow, _results;
  if (CL.fTraceOn) {
    CL.fnenterprint("dualOptimize:");
  }
  zRow = this.rowExpression(this._objective);
  _results = [];
  while (!this._infeasibleRows.isEmpty()) {
    exitVar = this._infeasibleRows.values()[0];
    this._infeasibleRows.remove(exitVar);
    entryVar = null;
    expr = this.rowExpression(exitVar);
    _results.push((function() {
      if (expr != null) {
        if (expr.constant() < 0.0) {
          ratio = Number.MAX_VALUE;
          terms = expr.terms();
          terms.each(function(v, c) {
            var r, zc;
            if (c > 0.0 && v.isPivotable()) {
              zc = zRow.coefficientFor(v);
              r = zc / c;
              if (r < ratio || (Cl.approx(r, ratio) && v.hashCode() < entryVar.hashCode())) {
                entryVar = v;
                return ratio = r;
              }
            }
          });
          if (ratio === Number.MAX_VALUE) {
            throw new Cl.errors.InternalError("ratio == nil (MAX_VALUE) in dualOptimize");
          }
          return this.pivot(entryVar, exitVar);
        }
      }
    }).call(this));
  }
  return _results;
};
Cl.SimplexSolver.prototype.newExpression = function(cn, eplus_eminus, prevEConstant) {
  var cnExpr, cnTerms, dummyVar, eminus, eplus, expr, slackVar, sw, swCoeff, zRow;
  cnExpr = cn.expression();
  expr = new Cl.LinearExpression(cnExpr.constant());
  slackVar = new Cl.SlackVariable();
  dummyVar = new Cl.DummyVariable();
  eminus = new Cl.SlackVariable();
  eplus = new Cl.SlackVariable();
  cnTerms = cnExpr.terms();
  cnTerms.each(goog.bind(function(v, c) {
    var e;
    e = this.rowExpression(v);
    if (e == null) {
      return expr.addVariable(v, c);
    } else {
      return expr.addExpression(e, c);
    }
  }, this));
  if (cn.isInequality()) {
    ++this._slackCounter;
    slackVar = new Cl.SlackVariable(this._slackCounter, "s");
    expr.setVariable(slackVar, -1);
    this._markerVars.put(cn, slackVar);
    if (!cn.isRequired()) {
      ++this._slackCounter;
      eminus = new Cl.SlackVariable(this._slackCounter, "em");
      expr.setVariable(eminus, 1.0);
      zRow = this.rowExpression(this._objective);
      sw = cn.strength().symbolicWeight().times(cn.weight());
      zRow.setVariable(eminus, sw.toDouble());
      this.insertErrorVar(cn, eminus);
      this.noteAddedVariable(eminus, this._objective);
    }
  } else {
    if (cn.isRequired()) {
      ++this._dummyCounter;
      dummyVar = new Cl.DummyVariable(this._dummyCounter, "d");
      expr.setVariable(dummyVar, 1.0);
      this._markerVars.put(cn, dummyVar);
    } else {
      ++this._slackCounter;
      eplus = new Cl.SlackVariable(this._slackCounter, "ep");
      eminus = new Cl.SlackVariable(this._slackCounter, "em");
      expr.setVariable(eplus, -1.0);
      expr.setVariable(eminus, 1.0);
      this._markerVars.put(cn, eplus);
      zRow = this.rowExpression(this._objective);
      sw = cn.strength().symbolicWeight().times(cn.weight());
      swCoeff = sw.toDouble();
      zRow.setVariable(eplus, swCoeff);
      this.noteAddedVariable(eplus, this._objective);
      zRow.setVariable(eminus, swCoeff);
      this.noteAddedVariable(eminus, this._objective);
      this.insertErrorVar(cn, eminus);
      this.insertErrorVar(cn, eplus);
      if (cn.isStayConstraint()) {
        this._stayPlusErrorVars.push(eplus);
        this._stayMinusErrorVars.push(eminus);
      } else if (cn.isEditConstraint()) {
        eplus_eminus[0] = eplus;
        eplus_eminus[1] = eminus;
        prevEConstant[0] = cnExpr.constant();
      }
    }
  }
  if (expr.constant() < 0) {
    expr.multiplyMe(-1);
  }
  return expr;
};
Cl.SimplexSolver.prototype.optimize = function(zVar) {
  var columnVars, entryVar, exitVar, minRatio, objectiveCoeff, r, terms, zRow, _results;
  zRow = this.rowExpression(zVar);
  CL.Assert(zRow != null, "zRow != null");
  entryVar = null;
  exitVar = null;
  _results = [];
  while (true) {
    objectiveCoeff = 0;
    terms = zRow.terms();
    terms.escapingEach(function(v, c) {
      if (v.isPivotable() && c < objectiveCoeff) {
        objectiveCoeff = c;
        entryVar = v;
        return {
          brk: true
        };
      }
    });
    if (objectiveCoeff >= -this._epsilon) {
      return;
    }
    minRatio = Number.MAX_VALUE;
    columnVars = this._columns.get(entryVar);
    r = 0.0;
    columnVars.each(goog.bind(function(v) {
      var coeff, expr;
      if (this.fTraceOn) {
        this.traceprint("Checking " + v);
      }
      if (v.isPivotable()) {
        expr = this.rowExpression(v);
        coeff = expr.coefficientFor(entryVar);
        if (coeff < 0.0) {
          r = -expr.constant() / coeff;
          if (r < minRatio || (Cl.approx(r, minRatio) && v.hashCode() < exitVar.hashCode())) {
            minRatio = r;
            return exitVar = v;
          }
        }
      }
    }, this));
    if (minRatio === Number.MAX_VALUE) {
      throw new Cl.errors.InternalError("Objective function is unbounded in optimize");
    }
    _results.push(this.pivot(entryVar, exitVar));
  }
  return _results;
};
Cl.SimplexSolver.prototype.pivot = function(entryVar, exitVar) {
  var pexpr;
  pexpr = this.removeRow(exitVar);
  pexpr.changeSubject(exitVar, entryVar);
  this.substituteOut(entryVar, pexpr);
  return this.addRow(entryVar, pexpr);
};
Cl.SimplexSolver.prototype.resetStayConstants = function() {
  var expr, i, _results;
  i = 0;
  _results = [];
  while (i < this._stayPlusErrorVars.length) {
    expr = this.rowExpression(this._stayPlusErrorVars[i]);
    if (expr == null) {
      expr = this.rowExpression(this._stayMinusErrorVars[i]);
    }
    if (expr != null) {
      expr.set_constant(0.0);
    }
    _results.push(i++);
  }
  return _results;
};
Cl.SimplexSolver.prototype.setExternalVariables = function() {
  this._externalParametricVars.each(goog.bind(function(v) {
    if (this.rowExpression(v) != null) {
      throw new Error("Error: variable" + v + " in _externalParametricVars is basic");
    } else {
      return v.change_value(0.0);
    }
  }, this));
  this._externalRows.each(goog.bind(function(v) {
    var expr;
    expr = this.rowExpression(v);
    return v.change_value(expr.constant());
  }, this));
  return this._fNeedsSolving = false;
};
Cl.SimplexSolver.prototype.insertErrorVar = function(cn, aVar) {
  var cnset;
  cnset = this._errorVars.get(aVar);
  if (cnset == null) {
    this._errorVars.put(cn, cnset = new HashSet());
  }
  return cnset.add(aVar);
};;

}); // close goog.scope()
