goog.provide('Cl.Tableau');

goog.require('Cl');
goog.require('Cl.CL');
goog.require('Cl.HashSet');
goog.require('Cl.HashTable');

goog.scope(function() {
  var CL = Cl.CL;
  var HashSet = Cl.HashSet;
  var HashTable = Cl.HashTable;
;
;
;
;
/**
 * @constructor
 */
Cl.Tableau = function() {
  this._columns = new HashTable();
  this._rows = new HashTable();
  this._infeasibleRows = new HashSet();
  this._externalRows = new HashSet();
  this._externalParametricVars = new HashSet();
};

Cl.Tableau.prototype.columns = function() {
  return this._columns;
};
Cl.Tableau.prototype.rows = function() {
  return this._rows;
};
Cl.Tableau.prototype.columnsHasKey = function(subject) {
  return this._columns.get(subject) != null;
};
Cl.Tableau.prototype.rowExpression = function(v) {
  return this._rows.get(v);
};
Cl.Tableau.prototype.noteRemovedVariable = function(v, subject) {
  if (subject != null) {
    return this._columns.get(v).remove(subject);
  }
};
Cl.Tableau.prototype.noteAddedVariable = function(v, subject) {
  if (subject) {
    return this.insertColVar(v, subject);
  }
};
Cl.Tableau.prototype.getInternalInfo = function() {
  var retstr;
  retstr = "Tableau Information:\n";
  retstr += "Rows: " + this._rows.size();
  retstr += " (= " + (this._rows.size() - 1) + " constraints)";
  retstr += "\nColumns: " + this._columns.size();
  retstr += "\nInfeasible Rows: " + this._infeasibleRows.size();
  retstr += "\nExternal basic variables: " + this._externalRows.size();
  retstr += "\nExternal parametric variables: ";
  retstr += this._externalParametricVars.size();
  retstr += "\n";
  return retstr;
};
Cl.Tableau.prototype.toString = function() {
  var bstr;
  bstr = "Tableau:\n";
  this._rows.each(function(clv, expr) {
    bstr += clv;
    bstr += " <==> ";
    bstr += expr;
    return bstr += "\n";
  });
  bstr += "\nColumns:\n";
  bstr += CL.hashToString(this._columns);
  bstr += "\nInfeasible rows: ";
  bstr += CL.setToString(this._infeasibleRows);
  bstr += "External basic variables: ";
  bstr += CL.setToString(this._externalRows);
  bstr += "External parametric variables: ";
  bstr += CL.setToString(this._externalParametricVars);
  return bstr;
};
Cl.Tableau.prototype.insertColVar = function(param_var, rowvar) {
  var rowset;
  rowset = this._columns.get(param_var);
  if (!rowset) {
    this._columns.put(param_var, rowset = new HashSet());
  }
  return rowset.add(rowvar);
};
Cl.Tableau.prototype.addRow = function(aVar, expr) {
  this._rows.put(aVar, expr);
  expr.terms().each(goog.bind(function(clv, coeff) {
    this.insertColVar(clv, aVar);
    if (clv.isExternal()) {
      return this._externalParametricVars.add(clv);
    }
  }, this));
  if (aVar.isExternal()) {
    return this._externalRows.add(aVar);
  }
};
Cl.Tableau.prototype.removeColumn = function(aVar) {
  var rows;
  rows = this._columns.remove(aVar);
  if (rows != null) {
    rows.each(goog.bind(function(clv) {
      var expr;
      expr = this._rows.get(clv);
      return expr.terms().remove(aVar);
    }, this));
  }
  if (aVar.isExternal()) {
    this._externalRows.remove(aVar);
    return this._externalParametricVars.remove(aVar);
  }
};
Cl.Tableau.prototype.removeRow = function(aVar) {
  var expr;
  expr = this._rows.get(aVar);
  CL.Assert(expr != null);
  expr.terms().each(goog.bind(function(clv, coeff) {
    var varset;
    varset = this._columns.get(clv);
    if (varset != null) {
      return varset.remove(aVar);
    }
  }, this));
  this._infeasibleRows.remove(aVar);
  if (aVar.isExternal()) {
    this._externalRows.remove(aVar);
  }
  this._rows.remove(aVar);
  return expr;
};
Cl.Tableau.prototype.substituteOut = function(oldVar, expr) {
  var varset;
  varset = this._columns.get(oldVar);
  varset.each(goog.bind(function(v) {
    var row;
    row = this._rows.get(v);
    row.substituteOut(oldVar, expr, v, this);
    if (v.isRestricted() && row.constant() < 0.0) {
      return this._infeasibleRows.add(v);
    }
  }, this));
  if (oldVar.isExternal()) {
    this._externalRows.add(oldVar);
    this._externalParametricVars.remove(oldVar);
  }
  return this._columns.remove(oldVar);
};;

}); // close goog.scope()
