(ns dubstep.macros
  (:use [clojure.core.match :only [match]]
        [dubstep.core :only [in?]]
        [dubstep.util :only [error]]))


(defn- parse-state-body
  "Transforms state macro body into map of fns."
  [l]
  (reduce (fn [m form]
            (match [form]
                   [(docstring :when string?)]
                   (assoc m :doc docstring)
                   
                   [(wubs :when map?)]
                   (assoc m :wubs wubs)
                   
                   [([(k :when #{'enter 'update 'exit})
                      (bindings :when vector?)
                      & body] :seq)]
                   (assoc m (keyword k) `(fn ~bindings ~@body))

                   :else (error (str "Unsupported in defstate form: " (prn-str form)))))
          {} l))

(comment
  (parse-state-body '("Docstring describing state"
                      (enter [this wubs] (x))
                      (update [this wubs] (y))
                      (exit [this wubs] (print 1) 2)))

  (parse-state-body '()))


(defmacro defstate
  "Creates and installs a new state. Optionally takes docstring, map of wubs, and/or enter/update/exit functions.
   Wubs must implement IWatchable."
  [sm name & rest]
  `(let [sm# ~sm
         s# ~(parse-state-body rest)]
     
     (when (dubstep.core/has-state? sm# ~name)
       (error (str "State machine already has state " ~name " defined.")))
     ;;Add state to SM
     (swap! sm# #(assoc-in % [:states ~name] s#))

     ;;Add wub watchers
     (doseq [[k# watchable#] (:wubs s#)]
       (add-watch watchable# (keyword (str "dubstep-" (name k#) "-" (clojure.core/name ~name) "-watcher"))
                  (fn [& _#]
                    (when (in? sm# ~name)
                      ;;notify state machine watchers, ghetto-style.
                      (swap! sm# identity)
                      
                      ;;run update-fn
                      (when-let [update# (:update s#)]
                        (update# sm# (:wubs s#)))))))
     
     sm#))

(defn- parse-event-body [l]
  (if (string? (first l))
    (merge {:docstring (first l)}
           (parse-event-body (rest l)))
    (match [l]
           [([(bindings :when vector?)
              & body] :seq)]
           {:event `(fn ~bindings ~@body)}

           :else (error (str "Unsupported event form: " (prn-str l))))))

(comment
  (parse-event-body '([this] (println 1) 2))
  (parse-event-body '("docstring" [this] x y)))


(defmacro defevent
  "Creates and installs a new event.
   Events are passed the state machine as first arg and any additional context from the trigger! call.
   Events take optional docstring first argument."
  [sm name & rest]
  `(let [sm# ~sm
         e# ~(parse-event-body rest)]
     (if (dubstep.core/has-event? sm# ~name)
       (error (str "State machine already has event '" ~name "' defined."))
       (swap! sm# #(assoc-in % [:events ~name] e#)))
     sm#))
