(ns shim.json
  (:require [goog.date.DateTime :as DateTime]))

(defn parse-tagged-string
  "Parse a tagged string within JSON.
   Assumes string begins with a throwaway first character and there is a single space between the tag and its data."
  [tagged-string]
  (let [split-idx (.indexOf tagged-string " ")
        tag (.substring tagged-string 1 split-idx)
        d (.substring tagged-string (inc split-idx))
        handler (case tag
                  ;;prefer plain JavaScript dates over goog.DateTimes.
                  "inst" #(js/Date. (DateTime/fromRfc822String %)))]

    (handler d)))

(defn generate-tagged-string
  "Replaces certain JavaScript objects with tagged string representations."
  [key x]
  (cond
    (instance? goog.date.DateTime x)
    (str "#inst " (.toXmlDateTime x true))
    :else x))

(defn parse-json
  "Parse JSON string using native handler with 'reviver' function that hooks into parse-tagged-string.
   See: https://developer.mozilla.org/en-US/docs/JavaScript/Reference/Global_Objects/JSON/parse"
  [json-str]
  (.parse js/JSON json-str
          (fn [k v]
            (if (and (string? v) (= "#" (aget v 0)))
              (parse-tagged-string v)
              v))))

(defn generate-json
  "Generate a JSON string using native handler with a 'replacer' that uses tagged string representations."
  [obj]
  (.stringify js/JSON obj generate-tagged-string))