(ns com.kurogitsune.avm-clj
  (:require [clojure.core.match :refer [match]]
            [clojure.core.logic :as logic]
            [clojure.set :as cs]))

(defn is-holder-map [x]
  (and (map? x) (= (count x) 1) (integer? (first (keys x)))))

(defn deep-merge [a b]
  (if (= a b) a
    (match [a b]
      [(_ :guard is-holder-map) (_ :guard map?)] #{a b} 
      [(_ :guard map?) (_ :guard is-holder-map)] #{a b}
      [(_ :guard map?) (_ :guard map?)] (merge-with deep-merge a b)
      [_ nil] a
      [nil _] b
      [(ea :guard set?) (eb :guard set?)] (clojure.set/union ea eb)  
      [(ea :guard set?) _] (clojure.set/union ea #{b}) 
      [_ (eb :guard set?)] (clojure.set/union #{a} eb)
      [(_ :guard sequential?) (_ :guard sequential?)]
        ;; [head rest] for holders. ex.[a b c] [{1 :_} {2 :_}] => [#{a {1 :_}} #{[b c] {2 :_}}] 
        (if (some? (not-empty (rest b)))
          [#{(first a) (first b)} (into #{} (map (fn [x] (if (= (count x) 1) (first x) (vec x))) #{(rest a) (rest b)}))]
          [#{(first a) (first b)} (vec (rest a))])
      [a b] #{a b}
  )))

(defn deep-replace [a b]
  (if (= a b) a
    (match [a b]
      [(_ :guard map?) (_ :guard map?)] (merge-with deep-replace a b)
      [_ nil] a
      [a b] b
  )))

(defn extract-holders 
  ([a] (extract-holders :_ a))
  ([parent a]
    (match [a]
      [[(i :guard integer?) body]] {parent {i body}}
      [[k (s :guard set?)]] (apply merge-with merge  (map (partial extract-holders {parent k}) s))
      [[k (v :guard map?)]] (apply merge-with merge  (map (partial extract-holders {parent k}) v))
      [(_ :guard map?)] (apply merge-with merge (map (partial extract-holders parent) a))
      [[k (v :guard sequential?)]] (apply merge (map-indexed (fn [i x] (extract-holders {{parent k} i} x)) v))
      [_] {}
    )))

(defn executed [x] (binding [*ns* (find-ns 'com.kurogitsune.avm-clj)] (load-string x)))
(defn unified-holders [extracted]
  (let [nums
    (->> extracted
      (map (fn [kv] (map  (fn [num_] (first num_)) (second kv)))))
    nums-available (filter (fn [x] (>= (count x) 2)) nums)
    num-vector (vec (map (fn [x] (str "_" x)) (sort (distinct (flatten nums)))))
    ;; (logic/run* [_1 _2 _3 _4 _5 _6]  (== _1 _4) (== _2 _4) (== _2 _5) (== _3 _5) (== _3 _6)))
    run-string (str "(logic/run* [" (clojure.string/join " " (vec num-vector)) "] " 
      (->> nums-available
        (map (fn [xs] (map (fn [x] (str "_" x)) xs)))
        (map (partial clojure.string/join " "))
        (map (fn [x] (str "(logic/== " x ")")))
        (clojure.string/join " ")
      ) ")")]
    (if (> (count nums-available) 0) 
      (into {} (map vector num-vector (first (executed run-string)))) 
      (into {} (map vector num-vector num-vector)))))

(defn extract-number [x]
  (read-string (second (re-matches #"_(.+)" (str x)))))

(defn get-unified-holder-pairs [a]
  (into {} (map (fn [x] [(extract-number (first x)) (extract-number (second x))]) (unified-holders (extract-holders a)))))

(defn deep-replace-holders [pairs a]
  (match [a]
    [[(i :guard integer?) :_]] [(get pairs i) :_]
    [[(i :guard integer?) body]] [(get pairs i) body]
    [[k (s :guard set?)]] 
      ;; #{1 :_, 3 :_} => 1 = 3 => #{0 :_} => {0 :_}
      (let [res [k (into #{} (map (partial deep-replace-holders pairs) s))]] (if (= 1 (count (second res))) [(first res) (first (second res))] res))
    [[k (v :guard map?)]] [k (into {} (map (partial deep-replace-holders pairs) v))]
    [(_ :guard map?)] (into {} (map (partial deep-replace-holders pairs) a))
    [[k (v :guard sequential?)]] [k (vec (map (partial deep-replace-holders pairs) v))]
    [_] a
))

(defn deep-input-holders [pairs a]
  (match [a]
    [[(i :guard integer?) :_]] (get pairs i)
    [[k (h :guard is-holder-map)]] [k (get pairs (first (keys h)))]
    [[k (v :guard map?)]] [k (into {} (map (partial deep-input-holders pairs) v))]
    [(_ :guard map?)] (into {} (map (partial deep-input-holders pairs) a))
    [[k (v :guard set?)]] [k (into #{} (map (partial deep-input-holders pairs) v))]
    [[k (v :guard sequential?)]] [k (vec (map (partial deep-input-holders pairs) v))]
    [_] a
))

(defn refresh-holders [a]
  (let [holder-pairs (get-unified-holder-pairs a)]
    (deep-replace-holders holder-pairs a)))

(defn unify [a b]
  (refresh-holders (deep-merge a b)))

(defn is-compatible [sub super] 
  (or (= sub super)
    (match [sub super]
      [_ :_] true ;; place holder ex.{:PHONE :_}
      [_ (_ :guard (fn [x] (and (map? x) (= 1 (count x)) (integer? (first (keys x))))))] true ;; place holder ex.{:SUBJ {1 :_}}
      [(_ :guard map?) (_ :guard map?)] (every? (fn [sb] (is-compatible (second sb) (get super (first sb)))) sub)
      [(ea :guard set?) (eb :guard set?)] (some? (not-empty (clojure.set/intersection ea eb)))
      [_ nil] true  
      [nil _] false
      [ea (eb :guard set?)] (some? (some (fn [x] (is-compatible ea x)) eb)) 
      [(ea :guard set?) eb] (some? (some (fn [x] (is-compatible x eb)) ea)) 
      [(v1 :guard sequential?) (v2 :guard sequential?)] (every? (fn [x] (is-compatible (first x) (second x))) (map vector v1 v2))
      [_ _] false
    )
  ))

(defn holder-set-to-map [s]
  (let [holder (some (fn [x] (if (is-holder-map x) x)) s) i (first (keys holder))] 
    (if (some? holder) {i (first (cs/difference s #{holder}))} {})))

(defn extract-replace-map [merged]
  ;; {:SUBJ #{{1 :_} [{:HEAD "NP"}]}} => {1 [{:HEAD "NP"}]}
  (match [merged]
    ;; [#{"foo" {1:_}} #{["bar" "bazz"] {2:_}}]
    [(s :guard set?)] (holder-set-to-map s)
    [[(k :guard keyword?) (s :guard set?)]] 
      (holder-set-to-map s)
    [[k (v :guard map?)]] 
      (apply merge (map extract-replace-map v))
    [(v :guard map?)] 
      (apply merge (map extract-replace-map v))
    [(v :guard sequential?)] 
      (apply merge (map extract-replace-map v))
    [_] {}
  ))

(defn extract-replace-map-with [holdered target]
  (let [merged (deep-merge holdered target)] (extract-replace-map merged)))
