(ns com.kurogitsune.logicadb.core
  (:require [clojure.java.jdbc :as j]
            [honeysql.core :as sql]
            [honeysql.helpers :refer :all]
            [pandect.algo.sha1 :as pd]
            [com.kurogitsune.logica.core :as lg]
            [clojurewerkz.serialism.core :as s]
            [clojure.core.match :refer [match]]))

(defn connected-db [path]
  {:classname   "org.sqlite.JDBC"
   :subprotocol "sqlite"
   :subname    	path
   })
   
(defn create-logic-ref [db table-name]
  (try (j/db-do-commands db
    (j/create-table-ddl table-name
      [:hash :binary :unique "PRIMARY KEY" "ON CONFLICT IGNORE"]))
    (catch Exception e (println e))))
    
(defn create-logic-base [db table-name]
  (try (j/db-do-commands db
    (j/create-table-ddl table-name
      [:hash :binary :unique "PRIMARY KEY" "ON CONFLICT IGNORE"]
      [:orderLogic :integer]
      [:type :text]
      [:name :text]
      [:param :text]
			[:eval :text]
      [:selfestablished :text]
      [:x0 :binary]
      [:x1 :binary]
      [:x2 :binary]
      [:x3 :binary]
      [:x4 :binary]))
    (catch Exception e (println e))))
   
(defn insert-unique! [db table data] 
  (let [dup (j/query db (sql/format (sql/build :select :hash :from table :where [:= :hash (:hash data)])))]
    (if (empty? dup) (j/insert! db table data))))
    
(defn serialized [l] (s/serialize l :json))
(defn deserialized [n] (s/deserialize n :json))

(defn safe-nth [x n] (try (nth x n) (catch Exception e nil)))
(defn nth-pval [p n] (safe-nth (vec (:args p)) (* 2 n)))

(defn dbhash [x] (pd/sha1-bytes (serialized (into #{} x))))

(defn db-styled [l] 
  (defn predicateOrFunc-db-styled [l]
    (let [xs (filter some? [(nth-pval l 0) (nth-pval l 1) (nth-pval l 2) (nth-pval l 3) (nth-pval l 4)])]
      (let [xhs (map (fn [x] (dbhash (second x))) xs)]
        (let [xh0 (safe-nth xhs 0) xh1 (safe-nth xhs 1) xh2 (safe-nth xhs 2) xh3 (safe-nth xhs 3) xh4 (safe-nth xhs 4)]
          (let [p (dissoc (assoc (into {} l) :orderLogic (:order l) :hash (dbhash l) :x0 xh0 :x1 xh1 :x2 xh2 :x3 xh3 :x4 xh4) :args :order )] 
            (flatten [[p] (flatten (map (fn [x] (db-styled (second x))) xs))]))
						))))
  (match [l] 
    [{:type "predicate"}] (predicateOrFunc-db-styled l)
    [{:type "variable"}]
      (let [v (dissoc (assoc (into {} l) :hash (dbhash l) :orderLogic (:order l)) :order)]
				v)
    [{:type "object"}]
      (let [o (assoc (into {} l) :hash (dbhash l))]
        o)   
    [{:type "func"}] (predicateOrFunc-db-styled l)
    [{:type "and" :a a :b b}]
      (let [an (dissoc (assoc (into {} l) :hash (dbhash l) :x0 (dbhash a) :x1 (dbhash b)) :a :b)]
        (flatten [[an] (db-styled a) (db-styled b)]))
    [{:type "or" :a a :b b}]
      (let [o (dissoc (assoc (into {} l) :hash (dbhash l) :x0 (dbhash a) :x1 (dbhash b)) :a :b)]
        (flatten [[o] (db-styled a) (db-styled b)]))
    [{:type "not" :a a}]
      (let [n (dissoc (assoc (into {} l) :hash (dbhash l) :x0 (dbhash a)) :a)]
        (flatten [[n] (db-styled a)]))
    [{:type "imp" :a a :b b}]
      (let [i (dissoc (assoc (into {} l) :hash (dbhash l) :x0 (dbhash a) :x1 (dbhash b)) :a :b)]
        (flatten [[i] (db-styled a) (db-styled b)]))
    [{:type "all" :x x :px px}]
      (let [a (dissoc (assoc (into {} l) :hash (dbhash l) :param x :x0 (dbhash px)) :x :px)]
        (flatten [[a] (db-styled px)]))
    [{:type "exists" :x x :px px}]
      (let [e (dissoc (assoc (into {} l) :hash (dbhash l) :param x :x0 (dbhash px)) :x :px)]
        (flatten [[e] (db-styled px)]))
    [_] []
    ))
	
(defn insert-values! [db table values]
	(let [values-string (second (reduce (fn [l r] [(+ 1 (first l)) (str (second l) "(nth values " (first l) ") ")]) [0 ""] values)) 
				inserts (str "(fn [db table values] (j/insert! db table " values-string " ))")]
		((binding [*ns* (find-ns 'com.kurogitsune.logicadb.core)] (load-string inserts)) db table values)))

(defn insertToDB [db table ls] 
	(let [values (flatten (map (fn [l] (db-styled l)) ls))]
		(insert-values! db table values)))
    
(defn selectFromHash 
  [logics h]
  (filter (fn [x] (and (some? h) (some? (:hash x)) (= (String. h) (String. (:hash x))))) logics))

(defn selectOneFromHash [logics h] (first (selectFromHash logics h)))

(defn recoverFromLogics [logics ldb]
  (defn recoverArgsFromLogics [logics ldb hashes]
    (let [xs (filter some? (not-empty (map (fn [h] (selectOneFromHash logics h)) hashes)))]
      (let [values (map (fn [x] (recoverFromLogics logics x)) xs)]
        (let [args (into {} (map-indexed (fn [i x] [(keyword (str "x" i)) x]) values))] args))))
  (match [ldb]
    [{:type "predicate" :orderlogic order :name name :x0 xh0 :x1 xh1 :x2 xh2 :x3 xh3 :x4 xh4}]
      (lg/predicateOf order name (recoverArgsFromLogics logics ldb [xh0 xh1 xh2 xh3 xh4]))
    [{:type "variable" :orderlogic order :name name}] (lg/variableOf order name)
    [{:type "object" :name name}] (lg/objectOf name) 
    [{:type "predicate" :orderlogic order :name name :eval e :x0 xh0 :x1 xh1 :x2 xh2 :x3 xh3 :x4 xh4 :selfestablished f}]
      (lg/funcOf order name e (assoc (recoverArgsFromLogics logics ldb [xh0 xh1 xh2 xh3 xh4]) :selfestablished f))
    [{:type "and" :x0 a :x1 b}] 
      (lg/andOf (recoverFromLogics logics (selectOneFromHash logics a)) (recoverFromLogics logics (selectOneFromHash logics b)))
    [{:type "or" :x0 a :x1 b}] 
      (lg/impOf (recoverFromLogics logics (selectOneFromHash logics a)) (recoverFromLogics logics (selectOneFromHash logics b)))
    [{:type "not" :x0 a}] 
      (lg/notOf (recoverFromLogics logics (selectOneFromHash logics a)))
    [{:type "imp" :x0 a :x1 b}] 
      (lg/impOf (recoverFromLogics logics (selectOneFromHash logics a)) (recoverFromLogics logics (selectOneFromHash logics b)))
    [{:type "all" :param x :x0 px}] 
      (lg/allOf x (recoverFromLogics logics (selectOneFromHash logics px)))
    [{:type "exists" :param x :x0 px}] 
      (lg/existsOf x (recoverFromLogics logics (selectOneFromHash logics px)))
    [_] nil
  ))


(defn removeLogicDB [db table l] 
  (let [q (sql/build :delete-from table :where [:= :hash (dbhash l)])]
    (j/execute! db (sql/format q))))
(defn addLogicsDB [db table ls] 
	(if (some? (not-empty ls))
		(let []
			(insertToDB db :logic ls)
			(insert-values! db table (map (fn [x] {:hash (dbhash x)}) ls)))))
(defn removeLogicsDB [db table ls] (doseq [l ls] (removeLogicDB db table l)))

(defn getLogicsDB [db table] 
	(let [ql (sql/build :select :* :from :logic) qh (sql/build :select :* :from table)]
		(let [logics (j/query db (sql/format ql)) hashes (j/query db (sql/format qh))]
			(let [xs (map (fn [h] (first (filter (fn [l] (= (String. (:hash l)) (String. (:hash h)))) logics))) hashes)]
				(into #{} (map (fn [x] (recoverFromLogics logics x)) xs))))))

(defn executed [request if-error] 
	(binding [*ns* (find-ns 'com.kurogitsune.logicadb.core)] 
		(try (eval (read-string request)) (catch Exception e (let [] (println e) if-error)))))

(defn xkv-folded [xkvs]
  (first (flatten (flatten xkvs))))

(defn xkv-replaced-with [logic lcs xkv]
  (match [logic]
    ;; {:x0 "x0"} && {:x0 (variableOf "x0")} && {:x0 foo} -> {:x0 [foo]} 
    [{:args args}] (xkv-folded (map (fn [a] (map (fn [kv] (if (= (keyword (:name (second a))) (first kv)) {(first kv) (map (fn [lc] ((first kv) (:args lc))) lcs)}) ) xkv ))args))
    [_] xkv
  ))
(defn xkv-input [logic xkv]
  (reduce (fn [l kv] 
    (match [l (first (second kv))] ;; @todo fix hack
      [{:args args} {:type t}] (assoc l :args (assoc args (first kv) (first (second kv))))  
      [_ _] l
    )) logic xkv))
(defn perm [f as bs] (flatten (map (fn [a] (map (fn [b] (f a b)) bs)) as)))

(defn self-established [logic]
  (and (some? (:selfestablished logic)) ((executed (:selfestablished logic) (fn [x] false)) logic)))

(defn findConstVerOf [logics logic xkv]
  ;; 前後関係を考慮したい。p(x)かつ[x<10](x)ならば、など、pのxが後ろにかかる。
  (match [(xkv-input logic xkv)]
    [{:type "and" :a a :b b}] (perm (fn [a1 b1] (lg/andOf a1 b1)) (findConstVerOf logics a xkv) (findConstVerOf logics b (xkv-replaced-with a (findConstVerOf logics a xkv) xkv)))
    [l1] (filter (fn [l] (or (lg/isConstVerOf l l1) (let [se (and (self-established l1) (= (:args l) (:args l1)))] se))) logics)
  ))

(defn substituted1 [l x lc]
	(match [l lc]
		[{:type "predicate"} {:type "predicate"}]
		(let [arg ((keyword x) (:args lc))] (if (some? arg) (assoc l :args (clojure.set/union (:args l ) (assoc (:args l) (keyword x) arg) )) l))
		[_ {:type "and" :a a :b b}] (substituted1 (substituted1 l x a) x b)
		[_ {:type "or" :a a :b b}] (substituted1 (substituted1 l x a) x b)
		[_ {:type "not" :a a}] (substituted1 l x a)
		[_ _] l
	))

(defn deductions [logics]
  (let [result
    (map
      (fn [l] 
        (match [l] 
          [{:type "all" :x x :px {:type "imp" :a a :b b}}] (map (fn [x] (lg/substituted b x)) (findConstVerOf logics a {(keyword x) x})) 
          [_] nil
        ))
      logics)]
    (into #{} (filter some? (flatten result)))))

(defn inductions [logics]
  (let [result
    (map (fn [l1] (map (fn [l2] (if (not (= l1 l2)) (lg/induction (lg/andOf l1 l2)) nil)) logics)) logics)]
    (into #{} (filter some? (flatten result)))))

(defn abductions [logics]
  (let [result
    (map
      (fn [l] 
        (match [l] 
          [{:type "all" :x x :px {:type "imp" :a a :b b}}] 
          (flatten (map (fn [bx] (map (fn [ax] (if (= (:args ax) (:args bx)) ax)) (findConstVerOf logics a {(keyword x) x}))) (findConstVerOf logics b {(keyword x) x}))) 
          [_] nil
        ))
      logics)]
    (into #{} (filter some? (flatten result)))))

(defn introduce [logics]
  (let [result
    (map
      (fn [l] 
        (match [l]
          [{:type "all" :x x1 :px {:type "imp" :a {:type "exists" :x x2 :px a } :b b}}] 
						(if (= x1 x2)
							(let [acs (findConstVerOf logics a {(keyword x1) x1})] 
							 (map (fn [ac] (substituted1 b x1 ac)) acs))
							nil) 
          [_] nil
        ))
      logics)]
    (into #{} (filter some? (flatten result)))))

(defrecord brain-db [db facts candidates hiddenfacts temp])
(defn create-brain-db [db] (->brain-db db #{} #{} #{} #{}))
(defn addLogics [brain table ls]
  (addLogicsDB (:db @brain) table ls)
  (reset! brain (update-in @brain [(keyword table)] clojure.set/union (into #{} ls))))
(defn addLogic [brain table l] (addLogics brain table [l]))
(defn removeLogic [brain table l]
  (removeLogicDB (:db @brain) table l)
  (reset! brain (update-in @brain [(keyword table)] clojure.set/difference #{l})))
(defn removeLogics [brain table ls] (doseq [l ls] (removeLogic brain table l)))

(defn safeDropTable [db table] (try (j/db-do-commands db (j/drop-table-ddl table)) (catch Exception e nil)))

(defn substituted [p pc] 
  (let 
    [newArgs (map 
      (fn [x]
        (match [(second x)]
          [{:type "variable" :order o :name n}] (first (filter (fn [x2] (= (keyword n) (first x2))) (:args pc)))
          [{:type "func" :eval f}] ((lg/executed f) (second (first (filter (fn [x2] (= (first x) (first x2))) (:args pc)))))
          [_] [(:first x) (:second x)]
          ))
      (:args p))] 
    (lg/predicateOf (:order p) (:name p) (into {} newArgs))))

(defn eval-functions-applied [brain logics]
  (let [fs (filter (fn [l] (some? (:eval l))) logics)]
    (reduce (fn [l r] ((executed (:eval r) (fn [brain self] brain)) l r)) brain fs)))

(defn think [brain] 
  (eval-functions-applied brain (:facts @brain)))
