(ns com.kurogitsune.logicadb.core
  (:require [clojure.java.jdbc :as j]
            [honeysql.core :as sql]
            [honeysql.helpers :refer :all]
            [pandect.algo.sha1 :as pd]
            [com.kurogitsune.logica.core :as lg]
            [clojurewerkz.serialism.core :as s]
            [clojure.core.match :refer [match]]))

(defn connected-db [path]
  {:classname   "org.sqlite.JDBC"
   :subprotocol "sqlite"
   :subname    	path
   })
   
(defn create-logic-ref [db table-name]
  (try (j/db-do-commands db
    (j/create-table-ddl table-name
      [:hash :binary :unique "PRIMARY KEY" "ON CONFLICT IGNORE"]))
    (catch Exception e (println e))))
    
(defn create-logic-base [db table-name]
  (try (j/db-do-commands db
    (j/create-table-ddl table-name
      [:hash :binary :unique "PRIMARY KEY" "ON CONFLICT IGNORE"]
      [:clojure :text]))
    (catch Exception e (println e))))
   
(defn insert-unique! [db table data] 
  (let [dup (j/query db (sql/format (sql/build :select :hash :from table :where [:= :hash (:hash data)])))]
    (if (empty? dup) (j/insert! db table data))))
    
(defn serialized-json [l] (s/serialize l :json))
(defn deserialized-json [n] (s/deserialize n :json))

(defn serialized [l] (s/serialize l :clojure))
(defn deserialized [b] (s/deserialize b s/clojure-content-type))

(defn safe-nth [x n] (try (nth x n) (catch Exception e nil)))
(defn nth-pval [p n] (safe-nth (vec (:args p)) (* 2 n)))

(defn dbhash [x] (pd/sha1-bytes (serialized-json (into #{} x))))

(defn db-styled [l] {:hash (dbhash l) :clojure (serialized l)})
	
(defn insert-values! [db table values]
	(let [values-string (second (reduce (fn [l r] [(+ 1 (first l)) (str (second l) "(nth values " (first l) ") ")]) [0 ""] values)) 
				inserts (str "(fn [db table values] (j/insert! db table " values-string " ))")]
		((binding [*ns* (find-ns 'com.kurogitsune.logicadb.core)] (load-string inserts)) db table values)))

(defn insertToDB [db table ls] 
	(let [values (map (fn [l] (db-styled l)) ls)]
		(insert-values! db table values)))
    
(defn selectFromHash 
  [logics h]
  (filter (fn [x] (and (some? h) (some? (:hash x)) (= (String. h) (String. (:hash x))))) logics))

(defn selectOneFromHash [logics h] (first (selectFromHash logics h)))

(defn recover-from-db-style [ldb] (deserialized (:clojure ldb)))

(defn removeLogicDB [db table l] 
  (let [q (sql/build :delete-from table :where [:= :hash (dbhash l)])]
    (j/execute! db (sql/format q))))
(defn addLogicsDB [db table ls] 
	(if (some? (not-empty ls))
		(let []
			(insertToDB db :logic ls)
			(insert-values! db table (map (fn [x] {:hash (dbhash x)}) ls)))))
(defn removeLogicsDB [db table ls] (doseq [l ls] (removeLogicDB db table l)))

(defn getLogicsDB [db table] 
	(let [q (sql/build :select :* :from table :join [:logic [:= (keyword (str (name table) ".hash")) :logic.hash]])]
		(let [xs (j/query db (sql/format q))]
				(into #{} (map (fn [x] (recover-from-db-style x)) xs)))))

(defn get-logics-db-include [db table l]
	(let [clojure (serialized l)]
		(let [q (sql/build :select :* :from table :join [:logic [:= (keyword (str (name table) ".hash")) :logic.hash]] :where [:like :logic.clojure (str "%" clojure "%")])]
			(let [xs (j/query db (sql/format q))]
					(into #{} (map (fn [x] (recover-from-db-style x)) xs))))))

(defn executed [request if-error] 
	(binding [*ns* (find-ns 'com.kurogitsune.logicadb.core)] 
		(try (eval (read-string request)) (catch Exception e (let [] (println e) if-error)))))

(defn xkv-folded [xkvs]
  (first (flatten (flatten xkvs))))

(defn xkv-replaced-with [logic lcs xkv]
  (match [logic]
    ;; {:x0 "x0"} && {:x0 (variableOf "x0")} && {:x0 foo} -> {:x0 [foo]} 
    [{:args args}] (xkv-folded (map (fn [a] (map (fn [kv] (if (= (keyword (:name (second a))) (first kv)) {(first kv) (map (fn [lc] ((first kv) (:args lc))) lcs)}) ) xkv ))args))
    [_] xkv
  ))
(defn xkv-input [logic xkv]
  (reduce (fn [l kv] 
    (match [l (first (second kv))] ;; @todo fix hack
      [{:args args} {:type t}] (assoc l :args (assoc args (first kv) (first (second kv))))  
      [_ _] l
    )) logic xkv))
(defn perm [f as bs] (flatten (map (fn [a] (map (fn [b] (f a b)) bs)) as)))

(defn self-established [logic]
  (and (some? (:selfestablished logic)) ((executed (:selfestablished logic) (fn [x] false)) logic)))

(defn findConstVerOf [logics logic xkv]
  ;; 前後関係を考慮したい。p(x)かつ[x<10](x)ならば、など、pのxが後ろにかかる。
  (match [(xkv-input logic xkv)]
    [{:type "and" :a a :b b}] (perm (fn [a1 b1] (lg/andOf a1 b1)) (findConstVerOf logics a xkv) (findConstVerOf logics b (xkv-replaced-with a (findConstVerOf logics a xkv) xkv)))
    [l1] (filter (fn [l] (or (lg/isConstVerOf l l1) (let [se (and (self-established l1) (= (:args l) (:args l1)))] se))) logics)
  ))

(defn substituted1 [l x lc]
	(match [l lc]
		[{:type "predicate"} {:type "predicate"}]
		(let [arg ((keyword x) (:args lc))] (if (some? arg) (assoc l :args (clojure.set/union (:args l ) (assoc (:args l) (keyword x) arg) )) l))
		[_ {:type "and" :a a :b b}] (substituted1 (substituted1 l x a) x b)
		[_ {:type "or" :a a :b b}] (substituted1 (substituted1 l x a) x b)
		[_ {:type "not" :a a}] (substituted1 l x a)
		[_ _] l
	))

(defn deductions [logics]
  (let [result
    (map
      (fn [l] 
        (match [l] 
          [{:type "all" :x x :px {:type "imp" :a a :b b}}] (map (fn [x] (lg/substituted b x)) (findConstVerOf logics a {(keyword x) x})) 
          [_] nil
        ))
      logics)]
    (into #{} (filter some? (flatten result)))))

(defn inductions [logics]
  (let [result
    (map (fn [l1] (map (fn [l2] (if (not (= l1 l2)) (lg/induction (lg/andOf l1 l2)) nil)) logics)) logics)]
    (into #{} (filter some? (flatten result)))))

(defn abductions [logics]
  (let [result
    (map
      (fn [l] 
        (match [l] 
          [{:type "all" :x x :px {:type "imp" :a a :b b}}] 
          (flatten (map (fn [bx] (map (fn [ax] (if (= (:args ax) (:args bx)) ax)) (findConstVerOf logics a {(keyword x) x}))) (findConstVerOf logics b {(keyword x) x}))) 
          [_] nil
        ))
      logics)]
    (into #{} (filter some? (flatten result)))))

(defn introduce [logics]
  (let [result
    (map
      (fn [l] 
        (match [l]
          [{:type "all" :x x1 :px {:type "imp" :a {:type "exists" :x x2 :px a } :b b}}] 
						(if (= x1 x2)
							(let [acs (findConstVerOf logics a {(keyword x1) x1})] 
							 (map (fn [ac] (substituted1 b x1 ac)) acs))
							nil) 
          [_] nil
        ))
      logics)]
    (into #{} (filter some? (flatten result)))))

(defrecord brain-db [db facts candidates hiddenfacts temp])
(defn create-brain-db [db] (->brain-db db #{} #{} #{} #{}))
(defn addLogics [brain table ls]
  (addLogicsDB (:db @brain) table ls)
  (reset! brain (update-in @brain [(keyword table)] clojure.set/union (into #{} ls))))
(defn addLogic [brain table l] (addLogics brain table [l]))
(defn removeLogic [brain table l]
  (removeLogicDB (:db @brain) table l)
  (reset! brain (update-in @brain [(keyword table)] clojure.set/difference #{l})))
(defn removeLogics [brain table ls] (doseq [l ls] (removeLogic brain table l)))

(defn safeDropTable [db table] (try (j/db-do-commands db (j/drop-table-ddl table)) (catch Exception e nil)))

(defn substituted [p pc] 
  (let 
    [newArgs (map 
      (fn [x]
        (match [(second x)]
          [{:type "variable" :order o :name n}] (first (filter (fn [x2] (= (keyword n) (first x2))) (:args pc)))
          [{:type "func" :eval f}] ((lg/executed f) (second (first (filter (fn [x2] (= (first x) (first x2))) (:args pc)))))
          [_] [(:first x) (:second x)]
          ))
      (:args p))] 
    (lg/predicateOf (:order p) (:name p) (into {} newArgs))))

(defn eval-functions-applied [brain logics]
  (let [fs (filter (fn [l] (some? (:eval l))) logics)]
    (reduce (fn [l r] ((executed (:eval r) (fn [brain self] brain)) l r)) brain fs)))

(defn think [brain] 
  (eval-functions-applied brain (:facts @brain)))
