(ns com.kurogitsune.logicadb.core
  (:require [clojure.java.jdbc :as j]
            [honeysql.core :as sql]
            [honeysql.helpers :refer :all]
            [pandect.algo.sha1 :as pd]
            [com.kurogitsune.logica.core :as lg]
						[com.kurogitsune.unbrace :as u]
            [clojurewerkz.serialism.core :as s]
            [clojure.core.match :refer [match]]))

(defn connected-db [path]
  {:classname   "org.sqlite.JDBC"
   :subprotocol "sqlite"
   :subname    	path
   })
   
(defn create-logic-ref [db table-name]
  (try (j/db-do-commands db
    (j/create-table-ddl table-name
      [:hash :binary :unique "PRIMARY KEY" "ON CONFLICT IGNORE"]))
    (catch Exception e (println e))))
    
(defn create-logic-base [db table-name]
  (try (j/db-do-commands db
    (j/create-table-ddl table-name
      [:hash :binary :unique "PRIMARY KEY" "ON CONFLICT IGNORE"]
      [:clojure :text]))
    (catch Exception e (println e))))
   
(defn insert-unique! [db table data] 
  (let [dup (j/query db (sql/format (sql/build :select :hash :from table :where [:= :hash (:hash data)])))]
    (if (empty? dup) (j/insert! db table data))))
    
(defn serialized-json [l] (s/serialize l :json))
(defn deserialized-json [n] (s/deserialize n :json))

(defn serialized [l] (s/serialize l :clojure))
(defn deserialized [b] (s/deserialize b s/clojure-content-type))

(defn safe-nth [x n] (try (nth x n) (catch Exception e nil)))
(defn nth-pval [p n] (safe-nth (vec (:args p)) (* 2 n)))

(defn dbhash [x] (pd/sha1-bytes (serialized-json (into #{} x))))

(defn db-styled [l] {:hash (dbhash l) :clojure (serialized l)})
	
(defn insert-values! [db table values]
	(let [values-string (second (reduce (fn [l r] [(+ 1 (first l)) (str (second l) "(nth values " (first l) ") ")]) [0 ""] values)) 
				inserts (str "(fn [db table values] (j/insert! db table " values-string " ))")]
		((binding [*ns* (find-ns 'com.kurogitsune.logicadb.core)] (load-string inserts)) db table values)))

(defn insertToDB [db table ls] 
	(let [values (map (fn [l] (db-styled l)) ls)]
		(insert-values! db table values)))
    
(defn selectFromHash 
  [logics h]
  (filter (fn [x] (and (some? h) (some? (:hash x)) (= (String. h) (String. (:hash x))))) logics))

(defn selectOneFromHash [logics h] (first (selectFromHash logics h)))

(defn recover-from-db-style [ldb] (deserialized (:clojure ldb)))

(defn removeLogicDB [db table l] 
  (let [q (sql/build :delete-from table :where [:= :hash (dbhash l)])]
    (j/execute! db (sql/format q))))
(defn addLogicsDB [db table ls] 
	(if (some? (not-empty ls))
		(let []
			(insertToDB db :logic ls)
			(insert-values! db table (map (fn [x] {:hash (dbhash x)}) ls)))))
(defn removeLogicsDB [db table ls] (doseq [l ls] (removeLogicDB db table l)))

(defn getLogicsDB [db table] 
	(let [q (sql/build :select :* :from table :join [:logic [:= (keyword (str (name table) ".hash")) :logic.hash]])]
		(let [xs (j/query db (sql/format q))]
				(into #{} (map (fn [x] (recover-from-db-style x)) xs)))))

(defn get-logics-db-include [db table l]
	(let [clojure (serialized l)]
		(let [q (sql/build :select :* :from table :join [:logic [:= (keyword (str (name table) ".hash")) :logic.hash]] :where [:like :logic.clojure (str "%" clojure "%")])]
			(let [xs (j/query db (sql/format q))]
					(into #{} (map (fn [x] (recover-from-db-style x)) xs))))))

(defrecord brain-db [db facts candidates hiddenfacts temp])
(defn create-brain-db [db] (->brain-db db #{} #{} #{} #{}))
(defn addLogics [brain table ls]
	(let [news (clojure.set/difference (into #{} ls) (table @brain))]
  	(addLogicsDB (:db @brain) table news)
  	(reset! brain (update-in @brain [(keyword table)] clojure.set/union (into #{} news)))))
(defn addLogic [brain table l] (addLogics brain table [l]))
(defn removeLogic [brain table l]
  (removeLogicDB (:db @brain) table l)
  (reset! brain (update-in @brain [(keyword table)] clojure.set/difference #{l})))
(defn removeLogics [brain table ls] (doseq [l ls] (removeLogic brain table l)))

(defn safeDropTable [db table] (try (j/db-do-commands db (j/drop-table-ddl table)) (catch Exception e nil)))

(defn substituted [p pc] 
  (let 
    [newArgs (map 
      (fn [x]
        (match [(second x)]
          [{:type "variable" :order o :name n}] (first (filter (fn [x2] (= (keyword n) (first x2))) (:args pc)))
          [{:type "func" :eval f}] ((lg/executed f) (second (first (filter (fn [x2] (= (first x) (first x2))) (:args pc)))))
          [_] [(:first x) (:second x)]
          ))
      (:args p))] 
    (lg/predicateOf (:order p) (:name p) (into {} newArgs))))
