(ns com.madeye.auth.v1.auth
  (:use [compojure.core :only (GET PUT POST DELETE defroutes)])
  (:use [slingshot.slingshot :only [try+ throw+]])
  (:gen-class)
  (:require (compojure handler route)
            [clojure.string :as str]
            [ring.util.response :as response]
            [ring.middleware.json :as rjson] 
            [ring.adapter.jetty :as jetty :only (run-jetty)]
            [clojure.data.json :as json]
            [com.madeye.clojure.common.common :as c]
            [com.madeye.clojure.common.config :as cfg]
            [com.madeye.auth.v1.crypt :as crypt]
            [clj-time.core :as tm]
            [clj-time.local :as tloc]
            [clj-time.format :as tfmt] 
            [taoensso.timbre :as timbre]
            [clojure.data.codec.base64 :as b64]
            [com.madeye.auth.v1.user :as u]
            [com.madeye.auth.v1.client :as client]
            [com.madeye.auth.v1.mongo.user :as mu]
  )
)


(timbre/refer-timbre)
(timbre/set-config! [:appenders :standard-out :enabled?] false)
(timbre/set-config! [:appenders :spit :enabled?] true)
(timbre/set-config! [:shared-appender-config :spit-filename] "auth.log")
(timbre/set-config! [:fmt-output-fn] (fn [{:keys [level throwable message timestamp hostname ns]}
                                      & [{:keys [nofonts?] :as appender-fmt-output-opts}]]
                                      (format "%s %s %s [%s] - %s%s\n"
                                      timestamp hostname (-> level name str/upper-case) ns (or message "")
                                      (or (timbre/stacktrace throwable "" (when nofonts? {})) ""))))


(defn- reload [] (use :reload-all 'com.madeye.auth.v1.auth))

(def ^:const ctx-login "/oauth2/token")

(def ^:const registration-grant-type "urn:madeye:oauth:grant-type:registration")

(defn base64-encode [original]
    (String. (b64/encode original) "UTF-8"))

(defn build-response
  [user]
  (let [token-expiry (cfg/get-obj-property :auth.token_expiry)
        auth-key (cfg/get-property :auth.key)
        expiry (tfmt/unparse (tfmt/formatters :basic-date-time) (tm/plus (tm/now) (tm/seconds token-expiry)))
        obj-id (str (:id user))
        to-encrypt (json/write-str (-> user (assoc :expiry expiry) (dissoc :_id) (assoc :userid obj-id)))]
    { :access_token (base64-encode (crypt/encrypt (crypt/get-bytes (str to-encrypt)) (crypt/public auth-key)))
      :token_type "bearer"
      :expires_in token-expiry
      :user_id (str (:_id user))
      :username (:username user)
    }
  )
)

(defn- handle-login
  [params]
  (let [grant-type (:grant_type params)
        username (:username params)
        password (:password params)
        auth-obj (cfg/get-property :auth.fn)]
    (debug "grant-type: " grant-type ", username: " username)
    (case grant-type 
      "password"
      (do
        (debug "Logging in") 
        (if-let [user-info (u/verify-user auth-obj username password)]
          (response/response (build-response user-info))
          (-> (response/response { :error "Invalid grant" :error_description "The username and/or password is incorrect" })
              (response/status 400))
        )
      )
      "urn:madeye:oauth:grant-type:registration"
      (do
         (debug "Registering")
         (let [accept-terms (read-string (:accept_terms params))
               allow-marketing (read-string (:allow_marketing params))]
           (debug "allow-marketing: " allow-marketing ", accept_terms: " accept-terms)
           (if accept-terms
              (do
                (debug "terms accepted")
                (try 
                  (let [new-user (u/create-user auth-obj username password allow-marketing)]
                    (-> (response/response (build-response new-user))
                        (response/status 200))
                  )
                  (catch clojure.lang.ExceptionInfo e
                    (error e)
                    (-> (response/response { :error "invalid_request" :error_reason (:code (ex-data e)) :error_description (.getMessage e) })
                        (response/status 400))
                  ) 
                )
              )             
              (do
                (debug "terms not accepted")
                (-> (response/response { :error "Invalid request" :error_description "You must accept the terms and conditions" })
                    (response/status 400))
              )
           )
         )
      )
      { :status 400 :body (str "The grant type '" grant-type "' is not supported.") }
    )
  )
)

(defn- test-response
  [request]
  (debug "test-response called")
  (response/response "Test OK!")
)

(defroutes app*
    (GET "/" request "com.madeye.auth.v1: API for authentication")
    (POST ctx-login request (handle-login (:params request)))
    (POST "/test" request (client/authorise-request request test-response))
)

; (def app (rjson/wrap-json-response (compojure.handler/api app*)))

(def app
    (-> (compojure.handler/api app*)
        (rjson/wrap-json-body )
        ; (rjson/wrap-json-params)
        (rjson/wrap-json-response)))

(defn start-server 
  "Initialisation function - starts Jetty server"
  [config-file]
  (info "Starting auth server")
  (cfg/load-config config-file)
  (let [auth-key (client/load-auth-key (cfg/get-property :auth.keyfile))
        auth-fn (eval (cfg/get-obj-property :auth.verify))
        server (jetty/run-jetty #'app {:port (cfg/get-obj-property :auth.port) :join? false})]
    (cfg/set-property :auth.fn auth-fn)
    (cfg/set-property :auth.server server)
  )
)

