(ns com.madeye.auth.v1.client
	(:require
            [clojure.data.codec.base64 :as b64]
            [com.madeye.clojure.common.common :as c]
            [com.madeye.clojure.common.config :as cfg]
            [clojure.data.json :as json]
            [com.madeye.auth.v1.crypt :as crypt]
            [ring.util.response :as response]
            [clj-time.core :as tm]
            [clj-time.format :as tfmt] 
            [taoensso.timbre :as timbre]
	)
)

(timbre/refer-timbre)

(defn- expired?
  [user-map]
  (let [expiry (tfmt/parse (:expiry user-map))
        now (tm/now)]
    (debug "Expiry: " expiry)
    (debug "Now: " now)
    (tm/after? now expiry)
  )
)

(defn authorise-token
  ([access-token]
    (try 
    (let [bytes (b64/decode (.getBytes access-token))
    	  auth-key (cfg/get-property :auth.key)
          decrypted (c/convert-string-keys-to-keywords (json/read-str (String. (crypt/decrypt bytes (crypt/private auth-key)))))
          expired (expired? decrypted)
          ]
      (debug "Decrypted: " decrypted)
      (debug "expired? " expired)
      (if expired
        nil
        decrypted
      )
    )
    (catch Exception e 
      (error "Failed to decrypt token" e)
      nil
    ))
  )
  ([user-id access-token]
    (if-let [decrypted (authorise-token access-token)]
      (if (= user-id (:id decrypted))
        decrypted
        nil
      )
    )
  )
)

(defn- get-bearer-token-from-header
  [header]
  (if (nil? header)
    nil
    (let [matches (re-find #"^Bearer (.*)" header)]
      (if (not (empty? matches))
        (matches 1)
        nil
      )
    )
  )
)

(defn- get-bearer-token-from-request
  [request]
  (debug "request:" request)
  (debug "headers:" (:headers request))
  (-> request 
    :headers 
    (get "authorization")
    (get-bearer-token-from-header)
  )
)

(defn- authorise-request-internal
  [request handler-fn authorisation-fn]
  (let [bearer-token (get-bearer-token-from-request request)]
    (debug "bearer-token: " bearer-token)
    (if-let [authed (authorisation-fn bearer-token)]
      (handler-fn request)
      (-> (response/response "") 
          (response/header "WWW-Authenticate" "Bearer error=\"invalid_token\", error_description=\"The username and/or password is incorrect or the token has expired\"" )
          (response/status 401))
    )
  )
)

(defn authorise-request
  ([request handler-fn]
    (authorise-request-internal request handler-fn authorise-token)
  )
  ([request user-id handler-fn]
    (authorise-request-internal request (partial handler-fn user-id) (partial authorise-token user-id))
  )
)

(defn load-auth-key
  [auth-key-file]
  (let [auth-key (crypt/read-keys auth-key-file)]
  	(cfg/set-property :auth.key auth-key)
  	auth-key
  )  
)
