(ns com.madeye.auth.v1.mongo.user
  (:gen-class)
  (:require [clojure.string :as str]
            [clojure.data.json :as json]
            [com.madeye.clojure.common.common :as c]
            [taoensso.timbre :as timbre]
            [monger.core :as mg]
            [monger.collection :as mc]
            [clojurewerkz.scrypt.core :as sc]
  )
  (:import [com.mongodb MongoOptions ServerAddress MongoException$DuplicateKey])
  (:import [org.bson.types ObjectId])
  (:use [com.madeye.auth.v1.user :only [User verify-user create-user get-user delete-user]])
  (:use [slingshot.slingshot :only [try+ throw+]])
)


(timbre/refer-timbre)

(def ^:const users-collection "users")

(defn- reload [] (use :reload-all 'com.madeye.auth.v1.mongo.user))

(defn get-connection
  [uri]
  (mg/connect-via-uri uri) 
)

(defn- scrypt
  [str]
  (sc/encrypt str 16384 8 1)
)

(extend-protocol User
  com.mongodb.DBApiLayer
 (get-user
    [db username]
    ; Uniqueness constraing on username so should only be one 
    (first (mc/find-maps db users-collection {:username username}))
 )
 (verify-user
  	[db username password]
    (debug "db: " db)
    (debug "verify-user: username - " username ", password - \"" password "\"")
    (if-let [user (get-user db username)]
       (if (sc/verify password (:password user))
         (do 
           (debug "User verified")
           user
         )
         (do 
           (debug "Password mismatch")
           nil
         )
       )
       (do 
         (debug "User " username " not found")
         nil
       )
    )
  )

  (create-user
    ([db username password allow-marketing]
      (try+
        (mc/insert-and-return db users-collection { :_id (ObjectId.) :username username :password (scrypt password) :allow-marketing allow-marketing})
      (catch com.mongodb.MongoException$DuplicateKey e
        (error e)
        (throw+ (ex-info "Another user with the same email address is already registered" {:code "user-already-exists"}))
      ))
    )
    ([db username password]
      (create-user db username password false)
    )
  )

  (delete-user
    [db username]
    (mc/remove db users-collection { :username username })
  )
)

