(ns clj-audio.wav-audio-recorder
  (:gen-class)
  (:use [clj-audio.low-level]
        [clj-audio.pcm-audio-recorder]
        [clj-audio.common])
  ;;logging features
  (:require [clojure.tools.logging :as log]
            [clojurewerkz.buffy.core :refer :all])
  (:use [clojure.java.io :only [output-stream]])
  (:import java.nio.ByteBuffer [java.nio ByteOrder]))

(defn get-wave-header-template
  "This function return array-map which represents WAV header template."
  []
  (array-map
    :chunkId (.getBytes "RIFF")
    :chunkSize (int 0) ;; (data size + header size(44) ) - 8 = data size + 36
    :format (.getBytes "WAVE")
    :subchunk1Id (.getBytes "fmt ") ;; space at the end is intentional
    :subchunk1Size (int 16) ;; 16 for PCM - constant value
    :audioFormat (short 1) ;; short 1 a constant value - means PCM format acoording to http://audiocoding.ru/wav_formats.txt
    :numChannels (short 2) ;; this value should be set according to audio parameters duiring the record
    :sampleRate (int 8000) ;; this value should be set according to audio parameters duiring the record
    :byteRate (int 0) ;; this value should be set according to audio parameters duiring the record
    :blockAlign (short 0)
    :bitsPerSample (short 16) ;; this value should be set according to audio parameters duiring the record
    :subchunk2Id (.getBytes "data")
    :subchunk2Size (int 0))) ;;data size

(defn create-wav-header
  "This function creates initial WAV header according to audio parameters. Audio parameters must represent a map.
  Example: {:sample-rate 44100 :sample-size-bits 16 :channels-number 2 :signed? true :bigEndian? false }
  Return: WAV-header array-map initialized with audio parameters"
  [audio-params data-size]
  (let
    [wh (get-wave-header-template)
     wh (assoc-in wh [:bitsPerSample] (short (audio-params :sample-size-bits)))
     wh (assoc-in wh [:numChannels] (short (audio-params :channels-number)))
     wh (assoc-in wh [:sampleRate] (int (audio-params :sample-rate)))
     wh (assoc-in wh [:byteRate] (int (/
                                      (*
                                        (audio-params :channels-number)
                                        (audio-params :sample-rate)
                                        (audio-params :sample-size-bits))
                                      8)))

     wh (assoc-in wh [:blockAlign] (short (/
                                          (*
                                            (audio-params :channels-number)
                                            (audio-params :sample-size-bits))
                                          8)))
     wh (assoc-in wh [:subchunk2Size] (int data-size))
     wh (assoc-in wh [:chunkSize] (int (+ data-size 36)))]

    wh))

(defn get-byte-array-from-wav-header
  "This function transforms WAV-header array-map to byte array.
  Return: byte[] object - representation of WAV header"
  [wave-header]
  (let [bb (ByteBuffer/allocate 44)]
    (.order bb ByteOrder/LITTLE_ENDIAN)
    (.put bb (wave-header :chunkId))
    (.putInt bb (wave-header :chunkSize))
    (.put bb (wave-header :format))
    (.put bb (wave-header :subchunk1Id))
    (.putInt bb (wave-header :subchunk1Size))
    (.putShort bb (wave-header :audioFormat))
    (.putShort bb (wave-header :numChannels))
    (.putInt bb (wave-header :sampleRate))
    (.putInt bb (wave-header :byteRate))
    (.putShort bb (wave-header :blockAlign))
    (.putShort bb (wave-header :bitsPerSample))
    (.put bb (wave-header :subchunk2Id))
    (.putInt bb (wave-header :subchunk2Size))
    (.array bb)
    ))


  (defn start-record-audio-to-wav-file
    "This function perfoms record of audio stream from microphone to a WAV file.
    Stop flag is used to stop recording. After stop - output file will be closed and sound resources will be relesed."
    [config-audio-filename ;;configuration file with audio parameters in edn format
     output-filename ;;name of output binary file name
     stop-flag?] ;;boolean atom to stop audio recording
    (safe
      (with-open [out-stream (output-stream output-filename)]
        (let [params (get-audio-params config-audio-filename)
              target-data-line (init-target-data-line params)
              wh (create-wav-header
                   (read-config-file config-audio-filename)
                   (- Integer/MAX_VALUE 100))]   ;;It is better to replace (Integer/MAX_VALUE -100) with actual value
          (.write out-stream (get-byte-array-from-wav-header wh) 0 44)
          (start-capture-audio stop-flag? target-data-line params out-stream)))))


(defn start-record-audio-to-wav-stream
  "This function perfoms record of audio stream from microphone and write encoded wav stream to a temp buffer.
  Then temp buffer is processed by function process-wav-audio-fn
  Stop flag is used to stop recording. After stop - sound resources will be relesed.
  Audio parameters file should be map.
  Example: {:sample-rate 44100 :sample-size-bits 16 :channels-number 2 :signed? true :bigEndian? false }."
  [^String config-audio-filename ;;configuration file with audio parameters in edn format
   process-wav-audio-fn ;;this function is invoked on every ready buffer. (process-spx-audio-fn buffer length)
   stop-flag?] ;;boolean atom to stop audio recording
  (safe
    ;(future (start-record-audio-to-wav-file config-audio-filename "a.wav" stop-flag?))
    (let [audio-format (get-audio-params config-audio-filename)
          target-data-line (init-target-data-line audio-format)
          audio-params (read-config-file config-audio-filename)

          pcm-packet-size (* 8 (audio-params :sample-rate))
          _ (log/debug "pcm packet size " pcm-packet-size)]
      (do

        (if-not (target-data-line-is-open? target-data-line)
          (open-target-data-line target-data-line audio-format))
        (start-target-data-line target-data-line)

        (while (not @stop-flag?)
          (do
            (let [ bb (byte-array pcm-packet-size)
                   bytes-captured-length (read-audio-stream target-data-line bb pcm-packet-size)]
              (future (process-wav-audio-fn bb bytes-captured-length)))))

        (stop-capture-audio target-data-line)))))