(ns cloud-clj.api
  (:import
    [java.net URI]
    [javax.net.ssl
     SNIHostName SNIServerName SSLEngine SSLParameters])
  (:require
    [manifold.stream :as s]
    [cheshire.core :as json]
    [manifold.deferred :as d]
    [org.httpkit.client :as http]))

(def COLUMN_DEFAULTS
  {:type "platform__schema_column"
   :name nil
   :data {:audited false
          :default_data nil
          :description nil
          :extended false
          :immutable false
          :indexed false
          :maximum nil
          :minimum nil
          :name nil
          :namespace nil
          :plural_name nil
          :precision nil
          :primary false
          :relationship_id nil
          :required false
          :schema_id nil
          :singular_name nil
          :sort nil
          :translated false
          :type nil
          :unique nil}
   :permissions {}
   :options []})

(def SCHEMA_DEFAULTS
  {:type "platform__schema"
   :name nil
   :data {:cacheable false
          :description nil
          :frozen false
          :immutable false
          :internal false
          :metadata false
          :name nil
          :namespace nil
          :plural_name nil
          :restricted false
          :singular_name nil
          :stability nil
          :static false}
   :columns []
   :permissions {}
   :options []})

(defn make-schema
  "Creates a schema object with a required namespace and name plus any
   additional options provided as key/value pairs."
  [schema-ns schema-name & options]
  (let [opt-map (apply hash-map options)]
    (-> SCHEMA_DEFAULTS
        (update-in [:data] merge opt-map)
        (assoc :name (str schema-ns "__" schema-name))
        (assoc-in [:data :name] schema-name)
        (assoc-in [:data :namespace] schema-ns))))

(defn add-column
  "Creates a schema column object with a required schema map/object, column
   namespace, column name, and column type. Additional parameters can be
   provided as key/value pairs after the required arguments."
  [schema column-ns column-name column-type & options]
  (let [opt-map (apply hash-map options)]
    (update-in schema [:columns] conj
               (-> COLUMN_DEFAULTS
                   (update-in [:data] merge opt-map)
                   (assoc :name (str column-ns "__" column-name))
                   (assoc-in [:data :name] column-name)
                   (assoc-in [:data :namespace] column-ns)
                   (assoc-in [:data :type] column-type)))))

(defn sni-configure
  "Configures HTTP-Kit's SSL engine to handle SNI (Server Name Indication)."
  [^SSLEngine ssl-engine ^URI uri]
  (let [^SSLParameters ssl-params (.getSSLParameters ssl-engine)]
    (.setServerNames ssl-params [(SNIHostName. (.getHost uri))])
    (.setSSLParameters ssl-engine ssl-params)))

(defn make-sni-client
  "Creates a HTTP-Kit client using a special SSL engine to handle SNI."
  []
  (http/make-client {:ssl-configurer sni-configure}))

(def ^:dynamic *sni-client* (make-sni-client))

(defn full-url
  "Creates a full url using the non-hostnamed endpoint and the current session
   to determine a fully qualified URI that can be used for REST requests.."
  [endpoint session] (str (:host session) endpoint))

(defn include-auth
  "Takes a request and a session and adds the JWT auth token iff the session
   is logged in and the request isn't explicitly excluding auth."
  [request session]
  (if (and (:auth-token session) (:include-auth? request true))
    (assoc-in request [:headers "Authorization"]
              (str "Bearer " (:auth-token session))) request))

(defn make-cloud-response
  "Parses the cloud response to ensure that it's valid, then extracts the
   parsed JSON body and attaches the HTTP response to it as metadata."
  [{:keys [body error status] :as resp}]
  (when error
    (throw
      (ex-info "An error occurred making a request to the Cloud API."
               {:response resp})))
  (when (not (<= 200 status 299))
    (throw
      (ex-info "A non-2xx HTTP status code was returned from the Cloud API."
               {:response resp})))
  (when (not body) 
    (throw
      (ex-info "Expected a payload from the server but didn't get one from the Cloud API."
               {:response resp})))
  (with-meta body {::http-response resp}))

(defprotocol CloudAPI
  (request [_ req])

  ;;; Core API Section
  (login
    [_ instance username password]
    "This function takes the session, instance, username, and password and
     attempts to login. If the login is successful, a new session is returned
     containing authentication information for future requests. If an error
     occurs, an exception will be thrown.")

  (set-password!
    [_ instance username new-password]
    "This function resets the password of the currently logged in user.")

  (ping
    [_]
    "This function calls the 'ping' API. This would be used to check the
     health of the session or the host the session is connected to.")

  ;;; Data API Section
  (select-data
    [_ data-namespace data-schema q]
    "This function selects data out of the Cloud API for the current session.

     This function takes 4 parameters, the last one is optional.
     session: Is the session record we're using to query.
     data-namespace: Is the namespace of the records being queried.
     data-schema: Is the schema name of the records being queried.
     q: Is a map representing a query filter in the cloud API.

     This function returns a vector of maps. The vector itself will contain
     metadata describing the actual HTTP response that contained the data and
     each map inside of the vector will describe the entire payload from the
     cloud and not just the 'data' attribute for the record.")

  (insert-data!
    [_ data-namespace data-schema data]
    "Performs an insert for a new record in the Cloud API.

     This function takes 4 arguments:
     session: Is the session record we're using to make the assertion.
     data-namespace: Is the namespace of the object we're asserting against.
     data-schema: Is the schema name for the object we're asserting against.
     data: Is the data for the record being asserted.")

  (update-data!
    [_ data-namespace data-schema data]
    "The same as insert-data! but, expects an id field to update against.")

  (upsert-data!
    [_ data-namespace data-schema data]
    "The same as insert-data! except it updates if the id field is present.")

  (delete-data! [_ data-namespace data-schema id])

  ;;; Metadata API Section
  (select-meta
    [_]
    [_ meta-namespace]
    [_ meta-namespace meta-name])
  
  (upsert-meta!
    [_ schema-list]
    "Similar to upsert-data! except for metadata."))

(def retry-codes #{502 503 504})

(defrecord CloudSession [host]
  CloudAPI
  (request [session req]
    (let [{:keys [retry? retry-count]
           :or {retry? false
                retry-count 3}
           :as req}
          (-> req
              (update-in [:url] full-url session)
              (assoc :client *sni-client*)
              (include-auth session))
          {:keys [status] :as result} @(http/request req)
          {:keys [content-type]} (:headers result)
          has-json? (re-find #"application/json" (or content-type ""))]
      (when (:error result)
        (throw
          (ex-info "Cloud API request failed!" result)))
      (if (not
            (and
              (contains? retry-codes status)
              retry? (pos? (dec retry-count))))
        (if has-json?
          (let [result (update-in result [:body] json/parse-string true)]
            result) result)
        (recur (update-in req [:retry-count] dec)))))

  (login [session instance username password]
    (let [result (request
                   session
                   {:url "api/auth/database"
                    :include-auth? false
                    :method :post
                    :headers {"Content-Type" "application/json"}
                    :body (json/generate-string
                            {:username username
                             :password password
                             :instance instance})})]
      (let [{:keys [status]} (:body result)
            success? (= status "success")]
        (when (not success?)
          (throw (ex-info "Authentication failed." {:api-result result})))
        (let [token (get-in result [:body :data :access_token])]
          (when (not token)
            (throw (ex-info "Login successful but, payload doesn't contain an auth token."
                            {:api-result result})))
          (assoc session :auth-token token :last-auth result
                 :instance (get-in result [:body :data :instance]))))))

  (set-password! [session instance username new-password]
    (make-cloud-response
      (request
        session
        {:url (str "api/auth/password")
         :method :post
         :headers {"Content-Type" "application/json"}
         :body (json/generate-string {:instance instance
                                      :username username
                                      :password new-password})})))

  (select-data [session data-namespace data-schema q]
    (make-cloud-response
      (request
        session
        (merge
          {:url (str "api/data/" data-namespace "/" data-schema "/search")
           :method :post}
          (when q
            {:headers {"Content-Type" "application/json"}
             :body (json/generate-string q)})))))

  (insert-data! [session data-namespace data-schema data]
    (make-cloud-response
      (request
        session
        {:url (str "api/data/" data-namespace "/" data-schema)
         :method :post
         :headers {"Content-Type" "application/json"}
         :body (json/generate-string data)})))

  (update-data! [session data-namespace data-schema data]
    (make-cloud-response
      (request
        session
        {:url (str "api/data/" data-namespace "/" data-schema)
         :method :patch
         :headers {"Content-Type" "application/json"}
         :body (json/generate-string data)})))

  (upsert-data! [session data-namespace data-schema data]
    (make-cloud-response
      (request
        session
        {:url (str "api/data/" data-namespace "/" data-schema)
         :method :put
         :headers {"Content-Type" "application/json"}
         :body (json/generate-string data)})))

  (delete-data!
    [session data-namespace data-name id]
    (make-cloud-response
      (request
        session
        {:url (str "api/data/" data-namespace "/" data-name)
         :method :delete
         :headers {"Content-Type" "application/json"}
         :body (json/generate-string
                 (if (coll? id)
                   (map (fn [id] {:id id}) id)
                   {:id id}))})))

  (select-meta
    [session]
    (make-cloud-response
      (request session {:url "api/meta" :method :get})))

  (select-meta [session meta-namespace]
    (make-cloud-response
      (request session
               {:url (str "api/meta/" meta-namespace)
                :method :get})))

  (select-meta [session meta-namespace meta-name]
    (make-cloud-response
      (request session
               {:url (str "api/meta/" meta-namespace "/" meta-name)
                :method :get})))

  (upsert-meta! [session schema-list]
    (make-cloud-response
      (request session
               {:url (str "api/meta")
                :method :put
                :headers {"Content-Type" "application/json"}
                :body (json/generate-string schema-list)})))

  (ping [session]
    (make-cloud-response
      (request session {:url "api/ping" :method :get}))))

(defn config->session
  "Takes a configuration map and returns a logged in session."
  [config]
  (let [{{:keys [instance username password]} :user
         :keys [host]} config]
    (-> (map->CloudSession {:host host})
        (login instance username password))))

(defn file->session
  "Takes a string path, reads the edn file at that location, then returns a
   logged in session for that configuration."
  [path]
  (config->session (clojure.edn/read-string (slurp path))))

