(ns cloud-clj.etl.pace
  (:require
    [clojure.java.io :as io]
    [clojure.data.csv :as csv]
    [cloud-clj.api :as api]
    [cloud-clj.util :refer [rand-pw]]))

(def ^:dynamic *target-namespace* "pace")
(def ^:dynamic *target-instance-name* "pace")

;;; Loading Common Core State Standards
(defn standard? [i] (not (nil? (re-find #"^CCSS\." i))))
(defn standard-code [i] (re-find #"^CCSS\.[a-zA-Z0-9.\-_]*" i))

(defn load-csv
  "This function loads a single CSV file. The 'file' var is merely passed to
   clojure.java.io/reader which is used to read the CSV file. The option map
   may provide an optional 'drop-first?' argument that will (rest) the output.
   This may be useful if you want to throw away the first row containing row
   headers."
  ([file] (load-csv file {}))
  ([file {:keys [drop-first?]
          :or {drop-first? false}}]
   ((if drop-first? rest identity)
    (csv/read-csv (io/reader file)))))

(defn load-many-csv
  "This function takes a sequence of files and passes each one to the load-csv
   function and concats the parsed CSV files together."
  ([files] (load-many-csv files {}))
  ([files opts] (mapcat #(load-csv % opts) files)))

(defn prepare-standard
  "This function takes a raw standard from CSV and converts it to an object
   that can be inserted into the Cloud API."
  [raw-standard]
  {:namespace *target-namespace*
   :name raw-standard
   :statement raw-standard
   :pace__code (standard-code raw-standard)})

(defn split-standard-groups
  "This function takes standard CSV data and splits up standards by their group
   as defined by a regex that matches rows that are 'standards' and not groups."
  [standard-csv-data]
  (partition 2 (partition-by standard? (map first standard-csv-data))))

(defn prepare-standard-group
  "Takes a group and its associated standards from a CSV dump that has already
   been split and turns it into a 'standard_group' that can be inserted to the
   Cloud API."
  [[group standards]]
  {:name (first group)
   :namespace *target-namespace*
   :standards (map prepare-standard standards)})

(defn prepare-standard-groups
  "Parses the raw standard CSV data and converts that data into something that
   can be inserted to the cloud api. It also embeds the related standards into
   the group (this has to be dissoc'ed prior to insert.)"
  [standard-csv-data]
  (map
    prepare-standard-group
    (split-standard-groups standard-csv-data)))

(defn load-standards
  "This function takes a Cloud API session and a sequence of CSV files, parses
   them as PACE standards and standard_groups and inserts them into the Cloud
   API."
  [session csv-files]
  (let [csv-data (load-many-csv csv-files)
        prepared-data (prepare-standard-groups csv-data)]
    prepared-data
    (doseq [group prepared-data]
      (let [group-response
            (api/insert-data! session "lrm" "standard_group"
                              (list (dissoc group :standards)))
            new-id (-> group-response :data first :data :id)]
        (api/insert-data! session "lrm" "standard"
                          (map
                            #(assoc % :standard_group new-id)
                            (:standards group)))))))

(defn prepare-science-standard
  "Takes a 'special' science standard from the parsed CSV and turns it into
   data that can be inserted into the Cloud API."
  [[statement description]]
  {:namespace *target-namespace*
   :name statement
   :statement statement
   :description description})

(defn prepare-science-standards
  "Takes a sequence of science standards as loaded from CSV file(s) and turns
   those records into 'standard' records that can be inserted into the Cloud
   API."
  [standard-csv-data]
  (map prepare-science-standard standard-csv-data))

(defn load-science-standards
  "Takes a Cloud API session and a list of CSV files and loads them into the
   Cloud API as PACE science standards (which are different in format from all
   of the other standards.)"
  [session csv-files]
  (let [csv-data (load-many-csv csv-files {:drop-first? true})
        prepared-data (prepare-science-standards csv-data)]
    (api/insert-data!
      session "lrm" "standard" prepared-data)))

(defn special-load!
  [session csv-file schema content-areas grades]
  (api/insert-data!
    session "lrm" schema
    (let [csv-data (load-csv csv-file {:drop-first? true})]
      (map
        (fn [[title description]]
          {:namespace *target-namespace*
           :name title
           :statement title
           :description description
           :content_area content-areas
           :grade_level grades})
        csv-data))))

;;; Loading competencies.
(defn prepare-competency
  "This function takes a CSV row representing a competency and converts it into
   something that can be inserted to the Cloud API."
  [[statement description]]
  {:namespace *target-namespace*
   :name statement
   :statement statement
   :description description})

(defn prepare-competencies
  "This function takes parsed PACE competencies from CSV files and converts
   each row into something insertable to the Cloud API."
  [csv-data]
  (map prepare-competency csv-data))

(defn load-competencies
  "This function takes a Cloud API session and a list of CSV files and parses
   it as PACE competencies, then inserts them into the Cloud API."
  [session csv-files]
  (let [csv-data (load-many-csv csv-files {:drop-first? true})
        prepared-data (prepare-competencies csv-data)]
    (api/insert-data!
      session "lrm" "competency" prepared-data)))

;;; Loading users.
(defn prepare-user
  "This function takes an instance id followed by a row of CSV as provided by
   PACE and turns that row into a map that can be inserted into the Cloud API."
  [instance-id [fullname firstname lastname grade-level content-area email district]]
  {:namespace *target-namespace*
   :instance_id instance-id
   :name email
   :first_name firstname
   :last_name lastname
   :email email
   :display_name fullname
   :pace__grade_level grade-level
   :pace__content_area content-area
   :pace__district district})

(defn prepare-user-persona
  "This function takes a persona and a platform user and generates a map
   representing a 'user_persona' record that can be inserted into the
   database."
  [persona platform-user]
  {:namespace *target-namespace*
   :active true
   :user_id (get-in platform-user [:data :id])
   :persona (get-in persona [:data :id])})

(defn get-instance-id [session instance-name]
  (let [instance-resp
        (api/select-data
          session "platform" "instance"
          {:where {:name instance-name}
           :limit 1})
        instance-id
        (get-in instance-resp [:data 0 :data :id])]
    (when (nil? instance-id)
      (throw (ex-info "Unable to find instance."
                      {:name instance-name
                       :response instance-resp})))
    instance-id))

(defn get-persona
  [session persona-name]
  (let [ persona-resp
        (api/select-data session "platform" "persona"
                         {:where {:name persona-name}
                          :limit 1})
        persona
        (first (:data persona-resp))]
    (when (nil? persona)
      (throw (ex-info "Unable to find persona in the database."
                      {:session session :name persona-name 
                       :query-response persona-resp})))
    persona))

(defn load-users!
  "This function loads in users from a CSV file with the session, csv file
   path, and persona name in that order.
   
   The CSV file should have data in the following order on each row after a
   header row:
   Fullname, Firstname, Lastname, Grade Level, Content Area, Email,
   and District."
  [session csv-file persona-name]
  (let [users (rest (load-csv csv-file)) ;;; First row is column headers.
        instance-id
        (get-instance-id session *target-instance-name*)
        persona
        (get-persona session persona-name)]
    (let [platform-users
          (:data
            (api/insert-data!
              session "platform" "user"
              (map (partial prepare-user instance-id) users)))]
      (api/insert-data!
        session "platform" "user_persona"
        (map
          (partial prepare-user-persona persona)
          platform-users))
      (doall
        (for [user platform-users]
          (let [password (rand-pw)
                user-data (:data user)]
            (api/set-password!
              session *target-instance-name* (:name user-data) password)
            {:user user-data
             :password password}))))))



