(ns cloud-clj.etl.qa
  (:require
    [cloud-clj.api :refer [insert-data!]]
    [clj-uuid :as uuid]))

(def ^:dynamic *target-ns* "qa")

(def ^:dynamic *student-user-ids*
  ["4e019622-7b9e-582d-901b-d88766aaa2e8"
   "4a3a7e70-bb21-5e02-885c-44bd7caa9f60"
   "d9729ff1-a9d9-55dd-bf45-8adf0998a188"
   "4d3cfbff-d983-5df2-bae4-8877fcb00210"])

(def ^:dynamic *instructor-user-ids*
  ["02cec2ce-0187-5f87-9a35-f8a07cff2817"
   "93cd4b12-f246-52ac-85fa-32e3c4b201da"
   "8cafef26-2f0c-5539-9d5b-bdc5c3c38bf5"])

(defn map-name [m]
  (if (and (map? m) (:name m)) (:name m) m))

(defn const-id [chain]
  (reduce
    (fn [current-uuid chain-value]
      (uuid/v5
        current-uuid
        (map-name chain-value)))
    uuid/+null+
    chain))

(defn make-record
  ([id-chain]
   (make-record id-chain nil))
  ([id-chain object]
   (merge 
     {:id (const-id id-chain)
      :name (map-name (last id-chain))
      :namespace *target-ns*}
     object)))

(comment
  
  (def discipline (make-record ["A Discipline"]))

  (def course
    (make-record
      [discipline "Underwater Welding for the Blind"]
      {:discipline (:id discipline)}))

  (def course-version
    (make-record
      [discipline course "v1.0"]
      {:course (:id course)}))

  (def academic-offering
    (make-record
      [discipline course course-version "An Offering"]
      {:description "This is an offering for Underwater Welding for the Blind."
       :enrollment_cap 10
       :status "active"
       :course_version (:id course-version)}))

  (def competencies
    [(make-record
       ["Master the Art of Swimming"]
       {:statement "Master the Art of Swimming"
        :description "You will demonstrate the ability to swim and survive the event."})
     (make-record
       ["Master the Art of Being Blind"]
       {:statement "Master the Art of Being Blind"
        :description "It's hard to weld underwater while blind without being blind."})
     (make-record
       ["Master the Art of Welding"]
       {:statement "Master the Art of Welding"
        :description "You must demonstrate the ability to weld like a boss."})])

  (def offering-subscriptions
    (map
      #(make-record
         [discipline course course-version academic-offering %]
         {:academic_offering (:id academic-offering)
          :user %})
      *instructor-user-ids*))

  (def enrollments
    (map
      #(make-record
         [discipline course course-version academic-offering %]
         {:incomplete_grade "83.7"
          :complete_grade "61.1"
          :academic_offering (:id academic-offering)
          :user %
          :proposed_start_date "12/06/2018"
          :status "active"
          :percent_complete "79.4"
          :proposed_completion_date "7/4/2019"
          :points_possible "100"
          :points_attempted "80"
          :points_earned "65"
          :entry_date "12/12/2018"
          :completable false
          :is_active true})
      *student-user-ids*))

  (def competency-progress
    (for [enrollment enrollments
          competency competencies]
      (make-record
        [enrollment competency]
        {:competency (:id competency)
         :enrollment (:id enrollment)})))

  (def competency-transcript
    (for [student-id *student-user-ids*
          competency competencies]
      (make-record
        [student-id competency]
        {:competency (:id competency)
         :user student-id})))

  (def session jdoane.user/qa-session)

  (cloud-clj.api/select-data
    session "lrm" "competency" {})

  (insert-data! session "lrm" "competency" competencies)
  (insert-data! session "lrm" "discipline" [discipline])
  (insert-data! session "lrm" "course" [course])
  (insert-data! session "lrm" "course_version" [course-version])
  (insert-data! session "lrm" "academic_offering" [academic-offering])
  (insert-data! session "lrm" "offering_subscription" offering-subscriptions)
  (insert-data! session "lrm" "enrollment" enrollments)
  (insert-data! session "lrm" "competency_progress" competency-progress)
  (insert-data! session "lrm" "competency_transcript" competency-transcript)

  

  )



