(ns cloud-clj.schema
  (:require
    [clojure.java.io :as io]
    [cheshire.core :as json]
    [cloud-clj.api :as api]))

(defn read-json-file
  "This function takes a file path or java.util.File, reads the file at that
   location, then parses the contents as JSON."
  [file]
  (-> (slurp file) (json/parse-string true)))

(defn read-metadata-files
  "This function takes a file path. If the path is a directory, we read all of
   the files immediately inside that directory otherwise a single file is read
   and parsed as JSON. The output is a list JSON parsed content from the given
   file path."
  [file-path]
  (let [file (io/file file-path)]
    (when (not (.exists file))
      (throw (ex-info "File does not exist." {:file-path file-path})))
    (if (.isDirectory file)
      (let [dir-files (filter #(.isFile %) (.listFiles file))]
        (doall (map read-json-file dir-files)))
      (list (read-json-file file)))))

(defn migrate-metadata!
  "This function takes a CloudAPI session and a file path, reads the file or
   files in the directory as JSON, then pushes that metadata into the current
   session via the metadata API."
  [session file-path]
  (api/upsert-meta! session (read-metadata-files file-path)))

(defn refresh-schema-column!
  [session [schema-ns schema-name] [column-ns column-name]]
  (let [schema-resp
        (api/select-data
          session "platform" "schema"
          {:where {:namespace schema-ns
                   :name schema-name}
           :related
           [{:schema "platform__schema_column"
             :column "schema_id"
             :where {:namespace column-ns
                     :name column-name}}]})
        column-id
        (get-in schema-resp
                [:data 0 :related :platform__schema_column
                 :platform__schema_id 0 :data :id])]
    (when (nil? column-id)
      (throw (ex-info "Unable to find schema-column."
                      {:schema (str schema-ns "__" schema-name)
                       :column (str column-ns "__" column-name)})))
    (api/update-data! session "platform" "schema_column" {:id column-id})))
