(ns cloud-clj.api
  (:import
    [java.net URI]
    [javax.net.ssl
     SNIHostName SNIServerName SSLEngine SSLParameters]
    [java.io ByteArrayOutputStream]
    [java.util.zip GZIPOutputStream])
  (:require
    [clojure.java.io :as io]
    [manifold.stream :as s]
    [cheshire.core :as json]
    [manifold.deferred :as d]
    [org.httpkit.client :as http]))

(def COLUMN_DEFAULTS
  {:type "platform__schema_column"
   :name nil
   :data {:audited false
          :default_data nil
          :description nil
          :extended false
          :immutable false
          :indexed false
          :maximum nil
          :minimum nil
          :name nil
          :namespace nil
          :plural_name nil
          :precision nil
          :primary false
          :relationship_id nil
          :required false
          :schema_id nil
          :singular_name nil
          :sort nil
          :translated false
          :type nil
          :unique nil}
   :permissions {}
   :options []})

(def SCHEMA_DEFAULTS
  {:type "platform__schema"
   :name nil
   :data {:cacheable false
          :description nil
          :frozen false
          :immutable false
          :internal false
          :metadata false
          :name nil
          :namespace nil
          :plural_name nil
          :restricted false
          :singular_name nil
          :stability nil
          :static false}
   :columns []
   :permissions {}
   :options []})

(defn make-schema
  "Creates a schema object with a required namespace and name plus any
   additional options provided as key/value pairs."
  [schema-ns schema-name & options]
  (let [opt-map (apply hash-map options)]
    (-> SCHEMA_DEFAULTS
        (update-in [:data] merge opt-map)
        (assoc :name (str schema-ns "__" schema-name))
        (assoc-in [:data :name] schema-name)
        (assoc-in [:data :namespace] schema-ns))))

(defn make-column
  ([column-ns column-name column-type]
   (make-column column-ns column-name column-type {}))
  ([column-ns column-name column-type opt-map]
   (-> COLUMN_DEFAULTS
       (update-in [:data] merge opt-map)
       (assoc :name (str column-ns "__" column-name))
       (assoc-in [:data :name] column-name)
       (assoc-in [:data :namespace] column-ns)
       (assoc-in [:data :type] column-type))))

(defn add-column
  "Creates a schema column object with a required schema map/object, column
   namespace, column name, and column type. Additional parameters can be
   provided as key/value pairs after the required arguments."
  [schema column-ns column-name column-type & options]
  (let [opt-map (apply hash-map options)]
    (update-in
      schema [:columns] conj
      (make-column column-ns column-name column-type
                   (merge
                     {:schema_id (:name schema)}
                     opt-map)))))

(defn sni-configure
  "Configures HTTP-Kit's SSL engine to handle SNI (Server Name Indication)."
  [^SSLEngine ssl-engine ^URI uri]
  (let [^SSLParameters ssl-params (.getSSLParameters ssl-engine)]
    (.setUseClientMode ssl-engine true)
    (.setServerNames ssl-params [(SNIHostName. (.getHost uri))])
    (.setSSLParameters ssl-engine ssl-params)))

(defn make-sni-client
  "Creates a HTTP-Kit client using a special SSL engine to handle SNI."
  []
  (http/make-client {:ssl-configurer sni-configure}))

(def ^:dynamic *sni-client* (make-sni-client))
(def ^:dynamic *default-timeout* (* 60000 3))

(defn full-url
  "Creates a full url using the non-hostnamed endpoint and the current session
   to determine a fully qualified URI that can be used for REST requests.."
  [endpoint session]
  (str (:host session)
       (if (coll? endpoint)
         (apply str (interpose "/" endpoint))
         endpoint)))

(defn include-auth
  "Takes a request and a session and adds the JWT auth token iff the session
   is logged in and the request isn't explicitly excluding auth."
  [request session]
  (let [token (get-in session [:auth :access_token])]
    (if (:include-auth? request true)
      (assoc-in request [:headers "Authorization"]
                (str "Bearer " token))
      request)))

(defn include-json
  "Takes a http request and converts the body into JSON with the appropriate
   content type if the `json?` key has a truthy value or if the body is a
   collection."
  [request]
  (if (or (coll? (:body request))
          (:json? request false))
    (-> request
        (assoc-in [:headers "Content-Type"] "application/json")
        (update-in [:body] json/generate-string))
    request))

(defn include-rollback
  "Takes a http request and a session and adds the rollback query parameter if the
   :rollback? key is set in the session"
  [req {:keys [rollback?]}]
  (if rollback?
    (assoc req :query-params {:options (json/generate-string {:rollback true})})
    req))

(defn include-accept-encoding
  [request]
  (assoc-in request [:headers "Accept-Encoding"] "gzip"))

(defn process-json
  "Takes a ring response and checks to see if it contains a JSON string that
   can be parsed and parses it if it does."
  [response]
  (let [{:keys [content-type]} (:headers response)]
    (if (re-find #"application/json" (or content-type ""))
      (update-in response [:body] json/parse-string true)
      response)))

(defn gzip-compress
  "Takes string content, gzips it, and returns it as a Java InputStream."
  [content]
  (with-open [output (ByteArrayOutputStream.)
              gzip-output (GZIPOutputStream. output)]
    (.write gzip-output (.getBytes content))
    (.finish gzip-output)
    (io/input-stream (.toByteArray output))))

(defn gzip-request-content
  "If the request contains a body and it's not explicitly setting :gzip? to
   false, then the body will get gzip'ed and the Content-Encoding set."
  [request]
  (if (and (:body request)
           (:gzip? request true))
    (-> request
        (update-in [:body] gzip-compress)
        (assoc-in [:headers "Content-Encoding"] "gzip"))
    request))

(defn make-cloud-response
  "Parses the cloud response to ensure that it's valid, then extracts the
   parsed JSON body and attaches the HTTP response to it as metadata."
  [{:keys [body error status] :as resp}]
  (when error
    (throw
      (ex-info "An error occurred making a request to the Cloud API."
               {:response resp})))
  (when (not (<= 200 status 299))
    (throw
      (ex-info "A non-2xx HTTP status code was returned from the Cloud API."
               {:response resp})))
  (when (not body)
    (throw
      (ex-info "Expected a payload from the server but didn't get one from the Cloud API."
               {:response resp})))
  (with-meta body {::http-response resp}))

(defn first-id
  "Given a response from the Cloud API, this function returns the id of the
   first record."
  [resp]
  (get-in resp [:data 0 :data :id]))

(defprotocol CloudAPI
  (request [_ req]))

(def RETRY-CODES #{502 503 504})

(def ^:dynamic *default-retry-count* 10)
(def ^:dynamic *default-retry?* true)

(defn prepare-request
  [session req]
  (-> req
      (update-in [:url] full-url session)
      (assoc
        :client *sni-client*
        :timeout *default-timeout*)
      (include-rollback session)
      (include-auth session)
      (include-json)
      (gzip-request-content)
      (include-accept-encoding)))

(defrecord CloudSession [host rollback?]
  CloudAPI
  (request [session req]
    (let [{:keys [retry? retry-count post-process?]
           :or {retry? *default-retry?*
                post-process? true}
           :as req}
          (prepare-request session req)]
      (loop [{:keys [retry-count] :as req
              :or {retry-count *default-retry-count*}} req]
        (let [{:keys [status] :as result} @(http/request req)
              attempt-retry? (and
                               (contains? RETRY-CODES status)
                               retry? (pos? (dec retry-count)))]
          (when (:error result)
            (throw
              (ex-info "Cloud API request failed!" result)))
          (if (not attempt-retry?)
            (if post-process?
              (-> result
                  process-json
                  make-cloud-response)
              result)
            (recur (assoc-in req [:retry-count] (dec retry-count)))))))))

(defn login
  "This function takes the session, instance, username, and password and
   attempts to login. If the login is successful, a new session is returned
   containing authentication information for future requests. If an error
   occurs, an exception will be thrown."
  [session instance username password]
  (let [result (request
                 session
                 {:url "api/auth/database"
                  :include-auth? false
                  :method :post
                  :body {:username username
                         :password password
                         :instance instance}})]
    (let [{:keys [status]} result
          success? (= status "success")]
      (when (not success?)
        (throw (ex-info "Authentication failed." {:api-result result})))
      (let [token (get-in result [:data :access_token])]
        (when (not token)
          (throw (ex-info "Login successful but, payload doesn't contain an auth token."
                          {:api-result result})))
        (assoc session :auth (:data result))))))

(defn set-password!
  "This function resets the password of the currently logged in user."
  [session instance username new-password]
  (request
    session
    {:url (str "api/auth/password")
     :method :post
     :body {:instance instance
            :username username
            :password new-password}}))

(defn select-data
  "This function selects data out of the Cloud API for the current session.

   This function takes 4 parameters, the last one is optional.
   session: Is the session record we're using to query.
   data-namespace: Is the namespace of the records being queried.
   data-schema: Is the schema name of the records being queried.
   q: Is a map representing a query filter in the cloud API.

   This function returns a vector of maps. The vector itself will contain
   metadata describing the actual HTTP response that contained the data and
   each map inside of the vector will describe the entire payload from the
   cloud and not just the 'data' attribute for the record."
  [session data-namespace data-schema q]
  (request
    session
    {:url ["api" "data" data-namespace data-schema "search"]
     :method :post
     :json? true
     :body (or q {})}))

(defn select-one-data
  "This function selects a single record by id out of the cloud api.

   This function takes 4 parameters, all of them are required.
   session: Is the session record we're using to query.
   data-namespace: Is the namespace of the records being queried.
   data-schema: Is the schema name of the records being queried.
   id: The id of the record that's being fetched.

   This function returns a single map describing the record and is merely
   the JSON parsed response. This is similar to select-data but, with the
   expectation that only one object will be returned since we're getting it
   by id."
  [session data-namespace data-schema id]
  (request
    session
    {:url ["api" "data" data-namespace data-schema id]
     :method :get}))

(defn insert-data!
  "Performs an insert for a new record in the Cloud API.

   This function takes 4 arguments:
   session: Is the session record we're using to make the assertion.
   data-namespace: Is the namespace of the object we're asserting against.
   data-schema: Is the schema name for the object we're asserting against.
   data: Is the data for the record being asserted."
  [session data-namespace data-schema data]
  (request
    session
    {:url ["api" "data" data-namespace data-schema]
     :method :post :json? true :body data}))

(defn update-data!
  "The same as insert-data! but, expects an id field to update against."
  [session data-namespace data-schema data]
  (request
    session
    {:url ["api" "data" data-namespace data-schema]
     :method :patch :json? true :body data}))

(defn upsert-data!
  "The same as insert-data! except it updates if the id field is present with
   an additional fourth parameter which is provided as a query string."
  ([session data-namespace data-schema data]
   (upsert-data! session data-namespace data-schema data {}))
  ([session data-namespace data-schema data options]
   (request
     session
     {:url ["api" "data" data-namespace data-schema]
      :query-params options
      :method :put :json? true :body data})))

(defn delete-data!
  [session data-namespace data-schema data]
  (request
    session
    {:url ["api" "data" data-namespace data-schema]
     :method :delete :json? true :body data}))

(defn select-meta
  ([session]
   (request session {:url "api/meta" :method :get}))
  ([session meta-namespace]
   (request
   session
   {:url ["api" "meta" meta-namespace] :method :get}))
  ([session meta-namespace meta-name]
   (request
   session
   {:url ["api" "meta" meta-namespace meta-name] :method :get})))

(defn upsert-meta!
  "Similar to upsert-data! except for metadata. The 2-tuple arity variant is
   the legacy implementation for backwards compatibility."
  ([session metadata-list]
   (request session{:url (str "api/meta") :method :put
                    :json? true :body metadata-list}))
  ([session meta-schema-ns meta-schema-name data]
   (request session {:url ["api/meta" meta-schema-ns meta-schema-name]
                     :method :put :json? true :body data})))

(defn ping
  "This function calls the 'ping' API. This would be used to check the
   health of the session or the host the session is connected to."
  [session]
  (request session {:url "api/ping" :method :get}))

(defn upload-file!
  "Takes a session, namespace, file name, file and optional
   options and uploads it to the Cloud API to be stored.

   The file can be anything that http-kit accepts as a body which includes
   but may not be limited to: java.io.File, Byte Array, InputStream, or a
   String."
  [session file-ns file-name file
   {:keys [content-type storage-provider]}]
  (request
    session
    {:url ["api/file" file-ns "upload" file-name]
     :method :post
     :body file
     :headers {"Content-Type" content-type
               "Storage-Provider" storage-provider}
     :json? false}))

(defn get-file
  "Takes a session and a file(_version) UUID and fetches the file. Unlike
   a lot of the other functions, this one returns the raw HTTP response
   since the full response describes what happened since the payload is a
   file as opposed to JSON."
  [session file-id]
  (request
    session
    {:url ["api/file" file-id]
     :method :get
     :json? false
     :post-process? false}))

(defn su
  "This function is very similar to `login` except this one expects a logged
   in session with root or superuser access, an instance string, and a
   username. If access criteria are satisifed, a new session is created for
   that user."
  [session instance username]
  (let [result (request
         session
         {:url "api/auth/su"
          :method :put
          :body {:username username
             :instance instance}})]
  (let [{:keys [status]} result
      success? (= status "success")]
    (when (not success?)
    (throw (ex-info "Auth su call failed." {:api-result result})))
    (let [token (get-in result [:data :access_token])]
    (when (not token)
      (throw (ex-info "Auth su call successful but, payload doesn't contain an auth token."
              {:api-result result})))
    (assoc session :auth (:data result))))))

(defn config->session
  "Takes a configuration map and returns a logged in session."
  [config]
  (let [{{:keys [instance username password]} :user
         :keys [host]} config]
    (-> (map->CloudSession (dissoc config :user))
        (login instance username password))))

(defn file->session
  "Takes a string path, reads the edn file at that location, then returns a
   logged in session for that configuration."
  [path]
  (config->session (clojure.edn/read-string (slurp path))))
