(ns cloud-clj.etl.vlacs
  (:require
    [clj-uuid :as uuid]
    [clojure.set :refer [rename-keys]]
    [cheshire.core :as json]
    [manifold.stream :as s]
    [manifold.deferred :as d]
    [cloud-clj.api :as api]))

(defn extract-data [resp]
  (map :data (:data resp)))

(defn make-map
  "Takes sequential data and turns it into a map by mapping over it, using the
   key-fn to extract a key for the map and val-fn to extract the value for the
   map."
  [data key-fn val-fn]
  (apply hash-map (mapcat #(vector (key-fn %) (val-fn %)) data)))

;;; Database Select Functions
(defn get-persona-map
  "Queries all of the personas out of the cloud api and turns it into a map of
   persona name to its unique UUID."
  [session]
  (make-map
    (extract-data (api/select-data session "platform" "persona" nil))
    :name :id))

(defn get-existing-users
  "Query all existing VLACS users based on their internal sis_user_idstr."
  [session vlacs-users]
  (extract-data
    (api/select-data
      session "platform" "user"
      {:where {:namespace "vlacs"
               :vlacs__sis_user_idstr (map :sis_user_idstr vlacs-users)}})))

(defn get-existing-user-personas
  "Queries existing user personas for a given list of platformn users."
  [session platform-users]
  (extract-data
    (api/select-data
      session "platform" "user_persona"
      {:where {:namespace "vlacs"
               :user_id (map :id platform-users)}})))

(defn get-existing-identities
  "Queries existing identities for VLACS' sis_user_idstr values based on a
   provided list of platform users."
  [session users]
  (extract-data
    (api/select-data
      session "platform" "identity"
      {:where 
       {:name (map :vlacs__sis_user_idstr users)
        :namespace "vlacs"
        :type "vlacs-identity"}})))

(defn get-existing-user-identities
  "Query user identity records based on a list of platform users."
  [session users]
  (extract-data
    (api/select-data
      session "platform" "user_identity"
      {:where {:user_id (map :id users)
               :namespace "vlacs"}})))

(defn get-existing-courses
  "Query existing courses based on a list of VLACS courses from their
   PostgreSQL database."
  [session vlacs-courses]
  (api/select-data
    session "lrm" "course"
    {:related
     [{:schema "lrm__course_version"
       :column "lrm__course"}]
     :where {:namespace "vlacs"
             :vlacs__master_course_idstr
             (map :master_course_idstr vlacs-courses)}}))

(defn get-existing-course-versions
  "Query existing course version using VLACS' internal course versions from
   their PostgreSQL database."
  [session vlacs-course-versions]
  (extract-data
    (api/select-data
      session "lrm" "course_version"
      {:where {:namespace "vlacs"
               :vlacs__master_course_version_id
               (map :id vlacs-course-versions)}})))

(defn get-related-course-versions
  "Queries lrm courses with related course versions based on a list of
   classrooms/offerings for the VLACS PostgreSQL database using
   master_course_idstr as an external id."
  [session vlacs-offerings]
  (:data
    (api/select-data
      session "lrm" "course"
      {:where {:namespace "vlacs"
               :vlacs__master_course_idstr
               (set (map :master_course_idstr vlacs-offerings))}
       :related
       [{:schema "lrm__course_version"
         :column "lrm__course"}]})))

(defn get-existing-offerings
  "Queries existing offerings based on a list of classrooms/offerings from
   VLACS' PostgreSQL database using classroom_idstr as an external id."
  [session vlacs-offerings]
  (api/select-data
    session "lrm" "academic_offering"
    {:where {:namespace "vlacs"
             :vlacs__classroom_idstr
             (map :classroom_idstr vlacs-offerings)}
     :related
     [{:schema "lrm__offering_subscription"
       :column "lrm__academic_offering"}]}))

(defn get-offering-instructors
  "Queries all instructor users based on the instructors list from a list of
   classrooms/offerings from the VLACS database."
  [session vlacs-offerings]
  (extract-data
    (api/select-data
      session "platform" "user"
      {:where {:namespace "vlacs"
               :vlacs__sis_user_idstr
               (set
                 (mapcat
                   #(clojure.string/split
                      (:sis_user_idstr_list %)
                      #"\|") vlacs-offerings))}})))

(defn get-existing-enrollments
  "Queries enrollments based on data from VLACS' PostgreSQL database, using the
   enrolment_idstr as an external id."
  [session vlacs-enrollments]
  (extract-data
    (api/select-data
      session "lrm" "enrollment"
      {:where {:namespace "vlacs"
               :vlacs__enrollment_idstr
               (map :enrolment_idstr vlacs-enrollments)}})))

(defn get-existing-competencies
  "Queries existing competencies from a list of internal VLACS asmt_pod_ids
   which is an external id for VLACS' competency data."
  [session asmt-pod-ids]
  (extract-data
    (api/select-data
      session "lrm" "competency"
      {:where {:namespace "vlacs"
               :vlacs__asmt_pod_id {:$in asmt-pod-ids}}})))

(defn get-existing-competency-transcripts
  "Queries existing competency transcripts from lrm enrollments, lrm
   competencies, and vlacs e2ap or e2cg records."
  [session user-enrollment-map competency-map vlacs-e2cgs]
  (extract-data
    (api/select-data
      session "lrm" "competency_transcript"
      {:where
       {:$or
        (set
          (mapv
            (fn [e2cg]
              (let [user-id (get user-enrollment-map (:enrolment_idstr e2cg))
                    competency-id (get competency-map (:asmt_pod_id e2cg))]
                {:$and
                 [{:student {:$eq user-id}}
                  {:competency {:$eq competency-id}}]}))
            vlacs-e2cgs))}})))

(defn get-existing-schools
  [session vlacs-affiliations]
  (extract-data
    (api/select-data
      session "lrm" "school"
      {:where {:namespace "vlacs"
               :vlacs__affiliation_idstr
               (map :sis_affiliation_idstr vlacs-affiliations)}})))

(defn get-existing-learning-paths
  [session]
  (extract-data
    (api/select-data
      session "lrm" "learning_path"
      {:where {:namespace "vlacs"}})))

(defn translate-learning-path
  [vlacs-lp]
  {:name (:keyword_name vlacs-lp)
   :label (:full_name vlacs-lp)
   :namespace "vlacs"
   :id (-> uuid/+null+
           (uuid/v3 "lrm__learning_path")
           (uuid/v3 (:learning_path_id vlacs-lp)))})

(defn load-learning-paths!
  [session count-agent vlacs-lps]
  (d/future
    (let [existing-data (get-existing-learning-paths session)
          id-set (set (map uuid/as-uuid (map :id existing-data)))
          translated-data (map translate-learning-path vlacs-lps)
          insert-data (filter #(not (id-set (:id %))) translated-data)
          update-data (filter #(id-set (:id %)) translated-data)
          insert-result (api/insert-data! session "lrm" "learning_path" insert-data)
          update-result (api/update-data! session "lrm" "learning_path" update-data)]
      (send count-agent #(+ % (count vlacs-lps)))
      (concat
        (extract-data insert-result)
        (extract-data update-result)))))

(defn privilege->persona
  "Creates a VLACS privlege to persona map for loading user information."
  [session]
  (let [persona-map (get-persona-map session)]
    {"STUDENT" (get persona-map "student")
     "TEACHER" (get persona-map "instructor")
     "ADMIN" (get persona-map "administrator")}))

(defn translate-vlacs-users
  "Takes a VLACS user and transforms it into a platform user."
  [vlacs-users]
  (map
    #(-> %
         (select-keys [:sis_user_idstr :firstname :lastname :username :email])
         (clojure.set/rename-keys {:sis_user_idstr :vlacs__sis_user_idstr
                                   :firstname :first_name
                                   :lastname :last_name
                                   :username :name})
         (assoc :namespace "vlacs"))
    vlacs-users))

(defn get-instance-id [session]
  (get-in
    (api/select-data
      session "platform" "instance"
      {:where {:name "vlacs.org"}
       :limit 1})
    [:data 0 :data :id]))

(defn make-user-persona-dml
  "Using the persona map, user persona map, vlacs users, and platform users,
   generate a series of DML operations to insert and update users in the
   cloud api."
  [persona-map user-persona-map vlacs-users platform-users]
  (let [vlacs-user-privilege-map
        (make-map vlacs-users :sis_user_idstr :privilege)]
    {:insert
     (filter
       identity
       (map
         (fn [user]
           (let [privilege (get vlacs-user-privilege-map (:vlacs__sis_user_idstr user))
                 persona (get persona-map privilege)]
             (when
               (and
                 (not (nil? persona))
                 (not
                   (some
                     #(= persona (:persona %))
                     (get user-persona-map (:id user)))))
               {:namespace "vlacs"
                :user_id (:id user)
                :persona persona
                :active true}))) platform-users))
     ;;; TODO: Implement this later.
     :delete []}))

(defn load-user-personas!
  "Takes the session, vlacs users, and platform users and migrates vlacs users
   into the cloud api."
  [session vlacs-users platform-users]
  (let [persona-map (privilege->persona session)
        user-personas (get-existing-user-personas session platform-users)
        user-persona-map (group-by :user_id user-personas)
        {:keys [insert delete]}
        (make-user-persona-dml
          persona-map user-persona-map vlacs-users platform-users)]
    (when (not (empty? delete))
      (api/delete-data! session "platform" "user_persona" delete))
    (when (not (empty? insert))
      (api/insert-data! session "platform" "user_persona" insert))))

(defn make-new-identities
  "Takes platform users and a map of vlacs identities and creates new identies
   for those that don't already exist."
  [platform-users identity-map]
  (not-empty
    (filter
      not-empty
      (for [{sis-user-idstr :vlacs__sis_user_idstr :as user} platform-users]
        (when (and (not (nil? sis-user-idstr))
                   (nil? (get identity-map sis-user-idstr)))
          {:name sis-user-idstr :type "vlacs-identity"
           :namespace "vlacs"})))))

(defn load-user-identities!
  "Takes a session and platform-users and provisions user identities based on
   VLACS' sis_user_idstrs."
  [session platform-users]
  ;;; Get all identities that have the "vlacs-identity" type and match
  ;;; vlacs__sis_user_idstr. If the identity and relationship don't already
  ;;; exist, make them.
  (let [identity-list (get-existing-identities
                        session platform-users)
        identity-map (make-map identity-list :name :id)
        user-identity-list (get-existing-user-identities
                             session platform-users)
        user-identities-map (group-by :user_id user-identity-list)
        ;;; All identities we care about.
        identity-map
        (merge
          identity-map
          (when-let
            [new-identities (make-new-identities platform-users identity-map)]
            (make-map 
              (extract-data
                (api/insert-data! session "platform" "identity" new-identities))
              :name :id)))]
    (when-let
      [new-user-identities
       (not-empty
         (filter
           not-empty
           (for [{sis-user-idstr :vlacs__sis_user_idstr
                  :keys [id] :as user} platform-users]
             (let [identity-id (get identity-map sis-user-idstr)]
               (when (not
                       (contains?
                         (set (map :identity_id (get user-identities-map id)))
                         identity-id))
                 {:user_id id
                  :namespace "vlacs"
                  :identity_id identity-id
                  :instance_id (get-in session [:instance :id])})))))]
      (api/insert-data! session "platform" "user_identity" new-user-identities))))

(defn load-users!
  "User loading function to be provided to #'run-import!. Take the current
   session, a count (agent,) and VLACS users from their PostgreSQL database
   and imports them into the cloud api. Returns a deferred value representing
   the completion of the import."
  [session count-agent vlacs-users]
  (d/future
    (let [instance-id (get-instance-id session)
          platform-users (get-existing-users session vlacs-users)
          platform-user-id-map (make-map platform-users :vlacs__sis_user_idstr :id)
          translated-users (translate-vlacs-users vlacs-users)
          upsert-users (map
                         (fn [user]
                           (assoc
                             (if-let [existing-id (get platform-user-id-map
                                                       (:vlacs__sis_user_idstr user))]
                               (assoc user :id existing-id) user)
                             :instance_id instance-id))
                         translated-users)
          platform-users 
          (when-let [upsert-users (not-empty upsert-users)]
            (extract-data
              (api/upsert-data! session "platform" "user" upsert-users)))]
      (load-user-identities! session platform-users)
      #_
      (load-user-personas! session vlacs-users platform-users)
      (send count-agent #(+ % (count vlacs-users)))
      nil)))

(defn form-competency
  [vlacs-competency]
  (-> vlacs-competency
      (select-keys [:name :description :id])
      (clojure.set/rename-keys
        {:id :vlacs__asmt_pod_id})
      (merge
        {:namespace "vlacs"
         :statement (:description vlacs-competency)
         :vlacs__is_segment
         (= (:asmt_pod_type_id vlacs-competency) 2)})))

(defn load-competencies!
  [session count-agent vlacs-competencies]
  (d/future
    (api/upsert-data!
      session "lrm" "competency"
      (map form-competency vlacs-competencies)
      {:options {:discover true}})
    (send count-agent #(+ % (count vlacs-competencies)))
    nil))

(defn load-student-data!
  "Student loading function to be run by #'run-import!"
  [session count-agent vlacs-students]
  (d/future
    (let [platform-users (get-existing-users session vlacs-students)
          platform-user-map (make-map platform-users :vlacs__sis_user_idstr identity)
          platform-schools (get-existing-schools
                             session (set
                                       (map
                                         #(-> %
                                              (select-keys [:affiliation_idstr])
                                              (rename-keys {:affiliation_idstr
                                                            :sis_affiliation_idstr}))
                                         vlacs-students)))
          school-map (make-map platform-schools :vlacs__affiliation_idstr :id)
          update-data
          (filter
            identity
            (map
              (fn [vlacs-student]
                (when-let
                  [platform-user (platform-user-map (:sis_user_idstr vlacs-student))]
                  (merge
                    platform-user
                    (when-let [email (:email vlacs-student)]
                      {:email email})
                    (when-let [school-id (school-map (:affiliation_idstr vlacs-student))]
                      {:lrm__school school-id})
                    {:vlacs__is_fulltime (= "FULL_TIME" (:ptft vlacs-student))
                     :vlacs__has_iep (= "1" (:iep vlacs-student))
                     :vlacs__has_section_504 (= "1" (:section504 vlacs-student))
                     :vlacs__dob (not-empty (:dob vlacs-student))
                     :vlacs__grade_level
                     (when-let [grade-level (Integer/parseInt (:grade_level vlacs-student 0))]
                       (if (< 0 grade-level) grade-level nil))})))
              vlacs-students))]
      (let [result (when (not (empty? update-data))
                     (api/update-data! session "platform" "user" update-data))]
        (send count-agent #(+ % (count vlacs-students)))
        result))))

(defn form-courses
  "Takes courses from VLACS' PostgreSQL database and converts them into a
   format consumable by the cloud api."
  [vlacs-courses discipline-map]
  (map
    #(-> %
         (select-keys [:master_course_idstr :name :department_idstr
                       :current_version])
         (rename-keys {:master_course_idstr :vlacs__master_course_idstr
                       :department_idstr :discipline})
         (update-in [:discipline] discipline-map)
         (assoc :namespace "vlacs"))
    vlacs-courses))

(defn get-existing-disciplines
  [session vlacs-disciplines]
   (extract-data
    (api/select-data
      session "lrm" "discipline"
      {:where {:vlacs__department_idstr
               (map :department_idstr vlacs-disciplines)}})))

(defn add-current-version
  [course-version-map platform-course]
  (assoc platform-course :current_version
         (course-version-map
           [(:id platform-course)
            (:current_version platform-course)])))

(defn load-courses!
  "Course loading function to be run by #'run-import!"
  [session count-agent vlacs-courses]
  (d/future
    (let [full-platform-courses (get-existing-courses session vlacs-courses)
          platform-courses (extract-data full-platform-courses)
          platform-disciplines
          (get-existing-disciplines session vlacs-courses)
          discipline-map (make-map platform-disciplines :vlacs__department_idstr :id)
          platform-course-map (make-map platform-courses :vlacs__master_course_idstr :id)
          current-version-map
          (make-map
            (mapcat
              #(map :data (get-in % [:related :lrm__course_version :lrm__course]))
              (:data full-platform-courses))
            #(vector (:course %) (:name %)) :id)
          vlacs-courses
          (map
            #(if-let [course-id (get platform-course-map (:vlacs__master_course_idstr %))]
               (assoc % :id course-id) %)
            (form-courses vlacs-courses discipline-map))]
      (let [rval (api/upsert-data!
                   session "lrm" "course"
                   (map
                     (partial add-current-version current-version-map)
                     vlacs-courses))]
        (send count-agent #(+ % (count vlacs-courses)))
        rval))))

(defn form-course-versions
  "Transforms VLACS course versions from their PostgreSQL database and platform
   courses and create course versions consumable by the cloud api."
  [vlacs-course-versions platform-courses]
  (let [platform-course-map (make-map platform-courses
                                      :vlacs__master_course_idstr
                                      :id)]
    (map
      #(-> %
           (select-keys [:master_course_idstr :version :id])
           (update-in [:master_course_idstr] (fn [id] (get platform-course-map id)))
           (rename-keys {:version :name
                         :id :vlacs__master_course_version_id
                         :master_course_idstr :course})
           (assoc :namespace "vlacs"))
      vlacs-course-versions)))

(defn load-course-versions!
  "Course version loading function to be run by #'run-import!"
  [session count-agent vlacs-course-versions]
  (d/future
    (let [platform-course-versions (get-existing-course-versions
                                     session vlacs-course-versions)
          platform-courses (extract-data
                             (get-existing-courses
                               session vlacs-course-versions))
          platform-cv-map (make-map platform-course-versions
                                    :vlacs__master_course_version_id :id)
          formed-course-versions (form-course-versions
                                  vlacs-course-versions
                                  platform-courses)
          rval (api/upsert-data!
                 session "lrm" "course_version"
                 (map
                   #(if-let [course-version-id (get platform-cv-map (:vlacs__master_course_version_id %))]
                      (assoc % :id course-version-id) %)
                   formed-course-versions))]
      (send count-agent #(+ % (count vlacs-course-versions)))
      rval)))

(defn attach-offering-lookups
  "Takes a vlacs offering and a platform course version map and attaches course
   version ids if the mapping exists."
  [offering platform-cv-map]
  (if-let [platform-cv-id 
           (get-in platform-cv-map
                   [(:master_course_idstr offering)
                    (:version offering)])]
    (assoc offering :course_version platform-cv-id) offering))

(defn form-offerings
  "Takes a list of VLACS offerings from their PostgreSQL database as well as a
   platform course version map and creates offerings consumable by the cloud
   api."
  [vlacs-offerings platform-cv-map]
  (map
    #(-> %
         (select-keys [:classroom_idstr :name :master_course_idstr :version :learning_path_id
                       :enrollment_cap])
         (attach-offering-lookups platform-cv-map)
         (dissoc :master_course_idstr :version)
         (update-in [:learning_path_id]
                    (fn [lp-id]
                      (when lp-id 
                        (-> uuid/+null+
                            (uuid/v3 "lrm__learning_path")
                            (uuid/v3 lp-id)))))
         (rename-keys {:classroom_idstr :vlacs__classroom_idstr
                       :learning_path_id :lrm__learning_path
                       :enrollment_cap :lrm__enrollment_cap})
         (assoc :namespace "vlacs"))
    vlacs-offerings))

(defn make-platform-cv-map
  "Takes a list of platform courses with related version records and creates a
   lookup map for relating offerings to the appropriate course versions."
  [platform-courses]
  (make-map
    platform-courses
    #(get-in % [:data :vlacs__master_course_idstr])
    #(make-map
       (get-in % [:related :lrm__course_version])
       (fn [i] (get-in i [:data :name]))
       (fn [i] (get-in i [:data :id])))))

(defn form-offering-subscriptions
  "Create offerings to be consumed by the cloud API using VLACS offerings from
   their PostgreSQL database, a map of instructors (sis_user_idstr to cloud
   api id,) and an offering map (classroom_idstr to cloud api id.)"
  [vlacs-offerings instructor-map offering-map]
  (filter
    identity
    (mapcat
      (fn [offering]
        (let [instructor-ids (set
                               (clojure.string/split
                                 (:sis_user_idstr_list offering "") #"\|"))]
          (map
            (fn [sis-user-idstr]
              (let [platform-offering (get offering-map (:classroom_idstr offering))
                    instructor-id (get instructor-map sis-user-idstr)
                    subscription-map (make-map (map
                                                 :data
                                                 (get-in
                                                   platform-offering
                                                   [:related :lrm__offering_subscription]))
                                               :user :id)]
                (when (and platform-offering instructor-id)
                  (merge
                    {:user instructor-id
                     :academic_offering (:id platform-offering)
                     :namespace "vlacs"}
                    (when-let [platform-subscription-id (get subscription-map instructor-id)]
                      {:id platform-subscription-id})))))
            instructor-ids)))
      vlacs-offerings)))

(defn load-offerings!
  "Offering loading function to be run by #'run-import!"
  [session count-agent vlacs-offerings]
  (d/future
    (let [platform-courses (get-related-course-versions session vlacs-offerings)
          platform-offerings (make-map
                               (:data (get-existing-offerings session vlacs-offerings))
                               #(get-in % [:data :vlacs__classroom_idstr])
                               #(get-in % [:data :id]))
          platform-instructors (get-offering-instructors session vlacs-offerings)
          instructor-map (make-map platform-instructors :vlacs__sis_user_idstr :id)
          platform-cv-map (make-platform-cv-map platform-courses)
          vlacs-offerings (form-offerings vlacs-offerings platform-cv-map)
          offering-id-map (make-map vlacs-offerings :vlacs__classroom_idstr :id)
          rval
          (api/upsert-data!
            session "lrm" "academic_offering"
            (map
              #(if-let [id (get platform-offerings (:vlacs__classroom_idstr %))]
                 (assoc % :id id) %)
              vlacs-offerings))]
      ;;; Sync offering subscriptions based on the initial offerings fetched.
      ;;; New offerings won't have any subscriptions yet.
      (form-offering-subscriptions
        vlacs-offerings instructor-map platform-offerings)
      (send count-agent #(+ % (count vlacs-offerings)))
      rval)))

(defn load-offering-subscriptions!
  "Offering subscription loading function to be run by #'run-import!"
  [session count-agent vlacs-offerings]
  (d/future
    (let [existing-offerings (:data (get-existing-offerings session vlacs-offerings))
          platform-offerings (make-map
                               existing-offerings
                               #(get-in % [:data :vlacs__classroom_idstr]) identity)
          platform-instructors (get-offering-instructors session vlacs-offerings)
          instructor-map (make-map platform-instructors :vlacs__sis_user_idstr :id)
          formed-offering-subscriptions
          (form-offering-subscriptions
            vlacs-offerings instructor-map platform-offerings)
          rval
          (api/upsert-data!
            session "lrm" "offering_subscription"
            formed-offering-subscriptions)]
      (send count-agent #(+ % (count vlacs-offerings)))
      ;;; TODO: Handle delete later.
      )))

(def enrollment-status-map
  {"ACTIVE"               "active"
   "WITHDRAWN_NO_GRADE"   "dropped"
   "WITHDRAWN_FAILING"    "withdrawn"
   "COMPLETED"            "completed"
   "ASSIGNED"             "assigned"
   "ASSIGNED_EMAILED"     "awaiting_welcome_call"
   ;;; Not a real status on key, but captured for target value.
   "WAITLISTED"           "waitlisted"
   "ENABLED"              "active"
   "CONTACT_INSTRUCTOR"   "contact_instructor"
   "DENIED"               "denied"
   "CANCELLED"            "cancelled"
   "APPROVED"             "approved"
   "WAITING_REVIEW"       "requested"
   "APPROVED_BY_PARENT"   "approved_by_parent"})

(defn form-enrollments
  "Takes VLACS enrollments from their PostgreSQL database, platform
   enrollments, platform users, and platform offerings and forms enrollments
   to be consumed by the cloud api."
  [vlacs-enrollments platform-enrollments platform-users platform-offerings]
  (let [user-map (make-map platform-users :vlacs__sis_user_idstr :id)
        offering-map (make-map platform-offerings :vlacs__classroom_idstr :id)
        enrollment-map (make-map platform-enrollments :vlacs__enrollment_idstr :id)]
    (filter
      identity
      (map
        (fn [vlacs-enrollment]
          (let [user-id (get user-map (:sis_user_idstr vlacs-enrollment))
                offering-id (get offering-map (:classroom_idstr vlacs-enrollment))]
            (when (and user-id offering-id)
              (merge
                {:namespace "vlacs"
                 :user user-id
                 :academic_offering offering-id
                 :status (or (when (= (:status_idstr vlacs-enrollment) "ACTIVE")
                               (enrollment-status-map (:activation_status_idstr vlacs-enrollment)))
                             (enrollment-status-map (:status_idstr vlacs-enrollment)))
                 :is_active (and
                              (= (:status_idstr vlacs-enrollment) "ACTIVE")
                              (contains?
                                #{"ENABLED" "CONTACT_INSTRUCTOR"}
                                (:activation_status_idstr vlacs-enrollment)))
                 :vlacs__enrollment_idstr (:enrolment_idstr vlacs-enrollment)}
                (when-let [enrollment-id (get enrollment-map (:enrolment_idstr vlacs-enrollment))]
                  {:id enrollment-id})))))
        vlacs-enrollments))))

(defn load-enrollments!
  "Enrollment loading function to be run by #'run-import."
  [session count-agent vlacs-enrollments]
  (d/future
    (let [queries [(d/future (get-existing-enrollments session vlacs-enrollments))
                   (d/future (get-existing-users session vlacs-enrollments))
                   (d/future (extract-data (get-existing-offerings session vlacs-enrollments)))]
          [platform-enrollments
           platform-users
           platform-offerings] (map deref queries)
          result (api/upsert-data!
                   session "lrm" "enrollment"
                   (form-enrollments
                     vlacs-enrollments platform-enrollments
                     platform-users platform-offerings))]
      (send count-agent #(+ % (count vlacs-enrollments)))
      result)))

(defn load-student-enrollments!
  [session count-agent vlacs-student-enrollments]
  (d/future
    (let [platform-enrollments
          (get-existing-enrollments
            session
            (map
              #(when (:genius_enrollment_id %)
                 (-> %
                     (select-keys [:genius_enrollment_id])
                     (rename-keys {:genius_enrollment_id :enrolment_idstr})))
              vlacs-student-enrollments))
          enrollment-map
          (make-map platform-enrollments :vlacs__enrollment_idstr :id)
          platform-course-versions
          (get-existing-course-versions
            session 
            (set
              (map
                #(-> %
                     (select-keys [:master_course_version_id])
                     (rename-keys {:master_course_version_id :id}))
                vlacs-student-enrollments)))
          course-version-map
          (make-map platform-course-versions :vlacs__master_course_version_id :id)]
      (api/update-data!
        session "lrm" "enrollment"
        (filter
          identity
          (map
            (fn [vlacs-se]
              (when-let [id (enrollment-map (str (:genius_enrollment_id vlacs-se)))]
                (merge
                  {:id id
                   :learning_path (-> uuid/+null+
                                      (uuid/v3 "lrm__learning_path")
                                      (uuid/v3 (:learning_path_id vlacs-se)))
                   :vlacs__student_enrollment_id (:student_enrollment_id vlacs-se)}
                  (when-let [cv-id (course-version-map (:master_course_version_id vlacs-se))]
                    {:course_version cv-id}))))
            vlacs-student-enrollments)))
      (send count-agent #(+ % (count vlacs-student-enrollments))))))

(defn convert-from-fixed-point
  [value]
  (when (number? value) (/ value 100)))

(defn translate-viewm-student-progress
  [platform-enrollments vlacs-viewm-student-progresses]
  (let [enrollment-map (make-map platform-enrollments :vlacs__enrollment_idstr :id)]
    (filter
      identity
      (for [sp vlacs-viewm-student-progresses]
        (when-let [enrollment-id (get enrollment-map (:enrolment_idstr sp))]
          (-> sp
              (clojure.set/rename-keys
                {:assessments_in_classroom  :assessments_in_offering
                 :iscompleteable            :completable
                 :points_course_possible    :points_possible
                 :isoverridden              :manual_override})
              (update-in [:completable] #(= % 1))
              (update-in [:manual_override] #(= % 1))
              (update-in [:complete_grade] convert-from-fixed-point)
              (update-in [:incomplete_grade] convert-from-fixed-point)
              (update-in [:percent_complete] convert-from-fixed-point)
              (select-keys
                [:incomplete_grade :complete_grade :assessments_in_offering
                 :assessments_attempted :assesments_required :completable
                 :percent_complete :points_possible :points_attempted
                 :points_earned :manual_override])
              (assoc :id enrollment-id)))))))

(defn load-student-progress!
  "Loads current (not historical,) student progress information."
  [session count-agent src-progress-records]
  (d/future
    (let [result (api/update-data!
                   session "lrm" "enrollment"
                   (translate-viewm-student-progress
                     (get-existing-enrollments session src-progress-records)
                     src-progress-records))]
      (send count-agent #(+ % (count src-progress-records)))
      result)))

(defn translate-schools
  [platform-schools vlacs-affiliations]
  (let [school-id-map (make-map platform-schools
                                :vlacs__affiliation_idstr :id)]
    (map
      (fn [affiliation]
        (-> affiliation
            (clojure.set/rename-keys
              {:sis_affiliation_idstr :vlacs__affiliation_idstr
               :contactname :contact_name
               :contactphone :contact_phone
               :contactemail :contact_email
               :status_idstr :is_active
               :districtcode :vlacs__district_code})
            (dissoc :id :sis_billset_idstr :timecreated :timemodified
                    :salesforce_affiliation_id)
            (assoc :namespace "vlacs")
            (merge
              (when-let [internal-id (school-id-map (:sis_affiliation_idstr
                                                      affiliation))]
                {:id internal-id}))
            (update-in [:is_active] #(= "ACTIVE" %))))
      vlacs-affiliations)))

(defn load-schools!
  [session count-agent vlacs-affiliations]
  (d/future
    (let [result (api/upsert-data!
                   session "lrm" "school"
                   (translate-schools
                     (get-existing-schools session vlacs-affiliations)
                     vlacs-affiliations))]
     (send count-agent #(+ % (count vlacs-affiliations))))))

(defn load-competency-grades!
  "Competency transcript loading function to be run by #'run-import!"
  [session count-agent vlacs-e2cgs]
  (d/future
    (let [platform-enrollments (get-existing-enrollments session vlacs-e2cgs)
          platform-competencies (get-existing-competencies session (map :asmt_pod_id vlacs-e2cgs))
          enrollment-map (make-map platform-enrollments :vlacs__enrollment_idstr :user)
          competency-map (make-map platform-competencies :vlacs__asmt_pod_id :id)
          platform-competency-transcripts (get-existing-competency-transcripts
                                            session enrollment-map competency-map vlacs-e2cgs)
          ct-map (make-map platform-competency-transcripts
                           #(vector (:student %) (:competency %)) identity)]
      (when-let [data
                 (not-empty
                   (vals
                     (reduce
                       (fn [ct-map e2cg]
                         (let [student-id (get enrollment-map (:enrolment_idstr e2cg))
                               competency-id (get competency-map (:asmt_pod_id e2cg))]
                           (if (not (and student-id competency-id))
                             ct-map
                             (update-in
                               ct-map [[student-id competency-id]]
                               (fn [ct]
                                 (update-in
                                   (or ct
                                       {:student student-id
                                        :competency competency-id})
                                   [:vlacs__enrollment_idstr_list]
                                   #(conj (set %) (:enrolment_idstr e2cg))))))))
                       ct-map vlacs-e2cgs)))]
        (api/upsert-data! session "lrm" "competency_transcript" data))
      (send count-agent #(+ % (count vlacs-e2cgs))))))


(defn load-disciplines!
  [session count-agent vlacs-disciplines]
  (d/future
    (let [platform-disciplines
          (get-existing-disciplines session vlacs-disciplines)
          disc-id-map (make-map platform-disciplines :vlacs__department_idstr :id)
          result
          (api/upsert-data!
            session "lrm" "discipline"
            (map
              (fn [{:keys [department_idstr name]}]
                (merge
                  {:vlacs__department_idstr department_idstr
                   :name name
                   :namespace "vlacs"}
                  (when-let [id (disc-id-map department_idstr)]
                    {:id id})))
              vlacs-disciplines))]
      (send count-agent #(+ % (count vlacs-disciplines)))
      result)))

(defn read-json-file
  "Takes a path, reads the contents of that file, and parsed the JSON."
  [path] (json/parse-string (slurp path) true))

(defn run-import!
  "Imports data using a session, loading function, and json data path into the
   cloud api. Immediately returns a map with a finished? key which is a
   deferred value that becomes realized when loading is complete or errors out,
   a load-fn! which is used to tell the importer how to load the data and
   handles counting processed records, and json-data-path which is a path to
   the data to be loaded."
  [session load-fn! json-data-path]
  (let [count-agent (agent 0)
        data (read-json-file json-data-path)
        data-stream (s/batch 500 (s/->source data))
        ;;; Pending buffer length determines how parallel this import will run.
        ;;; Higher numbers means more threads will be handling requests at any
        ;;; given time.
        pending-buffer (s/stream 1)
        completion-deferred (d/deferred)]
    (s/connect
      (s/map (partial load-fn! session count-agent) data-stream)
      pending-buffer)
    (s/consume
      (fn [_] true)
      (s/realize-each pending-buffer))
    (s/on-closed
      pending-buffer
      (fn [] (d/success! completion-deferred true)))
    {:finished? completion-deferred
     :total (count data)
     :processed count-agent}))

(defn import-competencies! [session competency-json-path]
  (run-import! session load-competencies! competency-json-path))

(defn import-student-info! [session student-json-path]
  (run-import! session load-student-data! student-json-path))

(defn import-competency-transcripts! [session competency-grade-json-path]
  (run-import! session load-competency-grades! competency-grade-json-path))

(defn import-enrollments! [session enrollment-json-path]
  (run-import! session load-enrollments! enrollment-json-path))

(defn import-student-enrollments! [session student-enrollment-json-path]
  (run-import! session load-student-enrollments! student-enrollment-json-path))

(defn import-progress! [session student-progress-json-path]
  (run-import! session load-student-progress! student-progress-json-path))

(defn import-offering-subscriptions! [session offering-json-path]
  (run-import! session load-offering-subscriptions! offering-json-path))

(defn import-offerings! [session offering-json-path]
  (run-import! session load-offerings! offering-json-path))

(defn import-course-versions! [session course-version-json-path]
  (run-import! session load-course-versions! course-version-json-path))

(defn import-courses! [session course-json-path]
  (run-import! session load-courses! course-json-path))

(defn import-users! [session user-json-path]
  (run-import! session load-users! user-json-path))

(defn import-learning-paths! [session user-json-path]
  (run-import! session load-learning-paths! user-json-path))

(defn import-schools! [session affiliation-json-path]
  (run-import! session load-schools! affiliation-json-path))

(defn import-disciplines! [session discipline-json-path]
  (run-import! session load-disciplines! discipline-json-path))

