(ns cloud-clj.api
  (:require
    [clojure.edn :as edn]
    [cloud-clj.request :as req]
    [cheshire.core :as json]
    [org.httpkit.client :as http]))

(defprotocol CloudAPI
  (request [_ req]))

(defrecord CloudSession [host rollback?]
  CloudAPI
  (request [session raw-req]
    (req/request session raw-req)))

(defn login
  "This function takes the session, instance, username, and password and
   attempts to login. If the login is successful, a new session is returned
   containing authentication information for future requests. If an error
   occurs, an exception will be thrown."
  [session instance username password]
  (let [result (request
                 session
                 {:url "api/auth/database"
                  :include-auth? false
                  :method :post
                  :body {:username username
                         :password password
                         :instance instance}})]
    (let [{:keys [status]} result
          success? (= status "success")]
      (when (not success?)
        (throw (ex-info "Authentication failed." {:api-result result})))
      (let [token (get-in result [:data :access_token])]
        (when (not token)
          (throw (ex-info "Login successful but, payload doesn't contain an auth token."
                          {:api-result result})))
        (assoc session :auth (:data result))))))

(defn set-password!
  "This function resets the password of the currently logged in user."
  [session instance username new-password]
  (request
    session
    {:url (str "api/auth/password")
     :method :post
     :body {:instance instance
            :username username
            :password new-password}}))

(defn bulk-data!
  [session payload]
  ;;; TODO:
  ;;; We might want to do some validation to make sure that the bulk payload is
  ;;; actually valid instead of just assuming that it's good and ready to roll.
  (request
    session
    {:url "api/data"
     :method :post
     :json? true
     :body payload}))

(defn select-data
  "This function selects data out of the Cloud API for the current session.

   This function takes 4 parameters, the last one is optional.
   session: Is the session record we're using to query.
   data-namespace: Is the namespace of the records being queried.
   data-schema: Is the schema name of the records being queried.
   q: Is a map representing a query filter in the cloud API.

   This function returns a vector of maps. The vector itself will contain
   metadata describing the actual HTTP response that contained the data and
   each map inside of the vector will describe the entire payload from the
   cloud and not just the 'data' attribute for the record."
  [session data-namespace data-schema q]
  (request
    session
    {:url ["api" "data" data-namespace data-schema "search"]
     :method :post
     :json? true
     :body (or q {})}))

(defn select-one-data
  "This function selects a single record by id out of the cloud api.

   This function takes 4 parameters, all of them are required.
   session: Is the session record we're using to query.
   data-namespace: Is the namespace of the records being queried.
   data-schema: Is the schema name of the records being queried.
   id: The id of the record that's being fetched.

   This function returns a single map describing the record and is merely
   the JSON parsed response. This is similar to select-data but, with the
   expectation that only one object will be returned since we're getting it
   by id."
  [session data-namespace data-schema id]
  (request
    session
    {:url ["api" "data" data-namespace data-schema id]
     :method :get}))

(defn insert-data!
  "Performs an insert for a new record in the Cloud API.

   This function takes 4 arguments:
   session: Is the session record we're using to make the assertion.
   data-namespace: Is the namespace of the object we're asserting against.
   data-schema: Is the schema name for the object we're asserting against.
   data: Is the data for the record being asserted."
  [session data-namespace data-schema data]
  (request
    session
    {:url ["api" "data" data-namespace data-schema]
     :method :post :json? true :body data}))

(defn update-data!
  "The same as insert-data! but, expects an id field to update against."
  [session data-namespace data-schema data]
  (request
    session
    {:url ["api" "data" data-namespace data-schema]
     :method :patch :json? true :body data}))

(defn upsert-data!
  "The same as insert-data! except it updates if the id field is present with
   an additional fourth parameter which is provided as a query string."
  ([session data-namespace data-schema data]
   (upsert-data! session data-namespace data-schema data {}))
  ([session data-namespace data-schema data options]
   (request
     session
     {:url ["api" "data" data-namespace data-schema]
      :query-params options
      :method :put :json? true :body data})))

(defn delete-data!
  [session data-namespace data-schema data]
  (request
    session
    {:url ["api" "data" data-namespace data-schema]
     :method :delete :json? true :body data}))

(defn select-meta
  [session]
  (request session {:url "api/meta" :method :get}))

(defn upsert-meta!
  "Similar to upsert-data! except for metadata. The 2-tuple arity variant is
   the legacy implementation for backwards compatibility."
  [session metadata-list]
  (request session {:url (str "api/meta") :method :put
                    :json? true :body metadata-list}))

(defn ping
  "This function calls the 'ping' API. This would be used to check the
   health of the session or the host the session is connected to."
  [session]
  (request session {:url "api/ping" :method :get}))

(defn upload-file!
  "Takes a session, namespace, file name, file and optional
   options and uploads it to the Cloud API to be stored.

   The file can be anything that http-kit accepts as a body which includes
   but may not be limited to: java.io.File, Byte Array, InputStream, or a
   String."
  [session file-ns file-name file
   {:keys [content-type storage-provider]}]
  (request
    session
    {:url ["api/file" file-ns "upload" file-name]
     :method :post
     :body file
     :headers {"Content-Type" content-type
               "Storage-Provider" storage-provider}
     :json? false}))

(defn get-file
  "Takes a session and a file(_version) UUID and fetches the file. Unlike
   a lot of the other functions, this one returns the raw HTTP response
   since the full response describes what happened since the payload is a
   file as opposed to JSON."
  [session file-id]
  (request
    session
    {:url ["api/file" file-id]
     :method :get
     :json? false
     :post-process? false}))

(defn su
  "This function is very similar to `login` except this one expects a logged
   in session with root or superuser access, an instance string, and a
   username. If access criteria are satisifed, a new session is created for
   that user."
  [session instance username]
  (let [result (request
                 session
                 {:url "api/auth/su"
                  :method :put
                  :body {:username username
                         :instance instance}})]
    (let [{:keys [status]} result
          success? (= status "success")]
      (when (not success?)
        (throw (ex-info "Auth su call failed." {:api-result result})))
      (let [token (get-in result [:data :access_token])]
        (when (not token)
          (throw (ex-info "Auth su call successful, but payload doesn't contain an auth token."
                          {:api-result result})))
        (assoc session :auth (:data result))))))

(defn config->session
  "Takes a configuration map and returns a logged in session."
  [config]
  (let [{{:keys [instance username password]} :user
         :keys [host]} config]
    (-> (map->CloudSession (dissoc config :user))
        (login instance username password))))

(defn file->session
  "Takes a string path, reads the edn file at that location, then returns a
   logged in session for that configuration."
  [path]
  (config->session (edn/read-string (slurp path))))
   

