(ns cloud-clj.request
  (:import
    [java.net URI]
    [javax.net.ssl
     SNIHostName SNIServerName SSLEngine SSLParameters]
    [java.io ByteArrayOutputStream IOException]
    [java.util.zip GZIPOutputStream])
  (:require
    [clojure.java.io :as io]
    [cheshire.core :as json]
    [org.httpkit.client :as http]))

(defn sni-configure
  "Configures HTTP-Kit's SSL engine to handle SNI (Server Name Indication)."
  [^SSLEngine ssl-engine ^URI uri]
  (let [^SSLParameters ssl-params (.getSSLParameters ssl-engine)]
    (.setUseClientMode ssl-engine true)
    (.setServerNames ssl-params [(SNIHostName. (.getHost uri))])
    (.setSSLParameters ssl-engine ssl-params)))

(defn make-sni-client
  "Creates a HTTP-Kit client using a special SSL engine to handle SNI."
  []
  (http/make-client {:ssl-configurer sni-configure}))

(def ^:dynamic *sni-client* (make-sni-client))
(def ^:dynamic *default-timeout* (* 60000 3))

(defn full-url
  "Creates a full url using the non-hostnamed endpoint and the current session
   to determine a fully qualified URI that can be used for REST requests.."
  [endpoint session]
  (str (:host session)
       (if (coll? endpoint)
         (apply str (interpose "/" endpoint))
         endpoint)))

(defn include-auth
  "Takes a request and a session and adds the JWT auth token iff the session
   is logged in and the request isn't explicitly excluding auth."
  [request session]
  (let [token (get-in session [:auth :access_token])]
    (if (:include-auth? request true)
      (assoc-in request [:headers "Authorization"]
                (str "Bearer " token))
      request)))

(defn include-json
  "Takes a http request and converts the body into JSON with the appropriate
   content type if the `json?` key has a truthy value or if the body is a
   collection."
  [request]
  (if (or (coll? (:body request))
          (:json? request false))
    (-> request
        (assoc-in [:headers "Content-Type"] "application/json")
        (update-in [:body] json/generate-string))
    request))

(defn include-rollback
  "Takes a http request and a session and adds the rollback query parameter if the
   :rollback? key is set in the session"
  [req {:keys [rollback?]}]
  (if rollback?
    (assoc req :query-params {:options (json/generate-string {:rollback true})})
    req))

(defn include-accept-encoding
  [request]
  (assoc-in request [:headers "Accept-Encoding"] "gzip"))

(defn process-json
  "Takes a ring response and checks to see if it contains a JSON string that
   can be parsed and parses it if it does."
  [response]
  (let [{:keys [content-type]} (:headers response)]
    (if (re-find #"application/json" (or content-type ""))
      (update-in response [:body] json/parse-string true)
      response)))

(defn gzip-compress
  "Takes string content, gzips it, and returns it as a Java InputStream."
  [content]
  (with-open [output (ByteArrayOutputStream.)
              gzip-output (GZIPOutputStream. output)]
    (.write gzip-output (.getBytes content))
    (.finish gzip-output)
    (io/input-stream (.toByteArray output))))

(defn gzip-request-content
  "If the request contains a body and it's not explicitly setting :gzip? to
   false, then the body will get gzip'ed and the Content-Encoding set."
  [request]
  (if (and (:body request)
           (:gzip? request true))
    (-> request
        (update-in [:body] gzip-compress)
        (assoc-in [:headers "Content-Encoding"] "gzip"))
    request))

(defn make-cloud-response
  "Parses the cloud response to ensure that it's valid, then extracts the
   parsed JSON body and attaches the HTTP response to it as metadata."
  [{:keys [body error status] :as resp}]
  (when error
    (throw
      (ex-info "An error occurred making a request to the Cloud API."
               {:response resp})))
  (when (not (<= 200 status 299))
    (throw
      (ex-info (get-in body [:data :message]
                       "A non-2xx HTTP status code was returned from the Cloud API.")
               {:response resp})))
  (when (not body)
    (throw
      (ex-info "Expected a payload from the server but didn't get one from the Cloud API."
               {:response resp})))
  (with-meta body {::http-response resp}))

(defn first-id
  "Given a response from the Cloud API, this function returns the id of the
   first record."
  [resp]
  (get-in resp [:data 0 :data :id]))

(def RETRY-CODES #{502 503 504 nil})

(def ^:dynamic *default-retry-count* 10)
(def ^:dynamic *default-retry?* true)

(defn prepare-request
  [session req]
  (-> req
      (update-in [:url] full-url session)
      (assoc
        :client *sni-client*
        :timeout *default-timeout*)
      (include-rollback session)
      (include-auth session)
      (include-json)
      (gzip-request-content)
      (include-accept-encoding)))

(defn request
  [session raw-req]
  (let [{:keys [retry? retry-count post-process?]
         :or {retry? *default-retry?*
              post-process? true}
         :as req}
        (prepare-request session raw-req)]
    (loop [{:keys [retry-count] :as req
            :or {retry-count *default-retry-count*}} req]
      (let [{:keys [status] :as result} @(http/request req)
            ;;; This tends to happen when a broken 504 gets returned from
            ;;; Google's load balancer. We should treat IOExceptions as 504s.
            io-exception?
            (when-let [t (:error result)]
              (instance? IOException t))
            attempt-retry? (and
                             (or (contains? RETRY-CODES status)
                                 io-exception?)
                             retry? (pos? (dec retry-count)))]
        (when (and (:error result)
                   (not attempt-retry?))
          (throw
            (ex-info "Cloud API request failed!"
                     {:raw-request raw-req
                      :result result})))
        (if (not attempt-retry?)
          (if post-process?
            (-> result
                process-json
                make-cloud-response)
            result)
          (recur (assoc-in req [:retry-count] (dec retry-count))))))))

