(ns nedap.lacinia.pedestal.test
  (:require
   [cheshire.core :as cheshire]
   [clj-http.multipart]
   [clojure.string :as string]
   [io.pedestal.test :refer [response-for]]
   [medley.core :refer [assoc-some]]
   [nedap.lacinia.pedestal.multipart.impl.content-type :refer [probe-content-type]]
   [nedap.lacinia.pedestal.multipart.impl.test :refer [map->paths]])
  (:import
   (java.io ByteArrayInputStream ByteArrayOutputStream File)))

(defn perform-multipart-query
  "performs a multipart query over the full pedestal stack

  A file can be passed into `variables`"
  [{:keys [url headers query variables service-fn]}]
  (let [files            (->> (map->paths variables)
                              (reduce (fn [memo path]
                                        (let [v (get-in variables path)]
                                          (if (instance? File v)
                                            (assoc memo path v)
                                            memo)))
                                      {}))

        mapping          (->> files
                              (map-indexed (fn [idx, [file-path, _file]]
                                             [(str idx)
                                              [(str "variables." (->> file-path
                                                                      (map (fn [x] (if (keyword? x)
                                                                                     (name x)
                                                                                     x)))
                                                                      (string/join ".")))]]))
                              (into {}))

        variables        (reduce (fn [memo [path, _]] (assoc-in memo path nil)) variables files)

        multipart-entity (-> (into [{:part-name "map", :content (cheshire/generate-string mapping)},
                                    {:part-name "operations", :content (cheshire/generate-string {:query query,
                                                                                                  :variables variables})}]
                                   (map-indexed (fn [idx, [_ ^File file]]
                                                  (assoc-some {:name (.getName file)
                                                               :content file
                                                               :part-name (str idx)
                                                               :encoding "utf-8"}
                                                              :mime-type (probe-content-type file))) files))
                             (clj-http.multipart/create-multipart-entity {}))]

    (-> (response-for service-fn :post url
                      :headers (assoc headers "Content-Type" (.. multipart-entity getContentType getValue))
                      :body (let [out (ByteArrayOutputStream.)]
                              (.writeTo multipart-entity out)
                              (.flush out)
                              (ByteArrayInputStream. (.toByteArray out))))
        (select-keys [:status :body])
        (update :body cheshire/parse-string keyword))))
