(ns crimson.pool
  (:require [clojure.spec.alpha :as s]
            [crimson.spec :as spec]
            [crimson.util :as util])
  (:import [redis.clients.jedis Jedis JedisPool JedisPoolConfig]))

(defn pool-config
  "Initialise jedis pool configuration.

  Accepts:
    A map containing any of the following keys:
     - max-total:                 Maximum number of connections that can be created at a given time
     - max-idle:                  Maximum number of connections that can be idle in the pool
                                  without being immediately evicted (closed)
     - min-idle:                  Minimum number of connections ready for immediate use,
                                  that remain in the pool
     - max-wait-millis:           Maximum time to wait in milliseconds if getting a resource
                                  from a pool will block
     - min-evictable-time-millis: Minimum amount of time a connection may sit idle in the pool
                                  before it is eligible for eviction due to idle time"
  ^JedisPoolConfig
  [{:keys [max-total
           max-idle
           min-idle
           max-wait-millis
           min-evictable-time-millis] :as params}]
  (if (false? (s/valid? ::spec/pool-config params))
    (s/explain-data ::spec/pool-config params)
    (cond-> (JedisPoolConfig.)
      max-total                 (doto (.setMaxTotal max-total))
      max-idle                  (doto (.setMaxIdle max-idle))
      min-idle                  (doto (.setMinIdle min-idle))
      max-wait-millis           (doto (.setMaxWaitMillis max-wait-millis))
      min-evictable-time-millis (doto (.setMinEvictableIdleTimeMillis min-evictable-time-millis)))))

(defn init
  "Creates a new redis pool and returns a `JedisPool` instance.

   Accepts a host string (host or host:port or tcp://host:port) and
   an (optional) options map containing the following keys:

     - max-total:                 Maximum number of connections that can be created at a given time
     - max-idle:                  Maximum number of connections that can be idle in the pool
                                  without being immediately evicted (closed)
     - min-idle:                  Minimum number of connections ready for immediate use,
                                  that remain in the pool
     - max-wait-millis:           Maximum time to wait in milliseconds if getting a resource
                                  from a pool will block
     - min-evictable-time-millis: Minimum amount of time a connection may sit idle in the pool
                                  before it is eligible for eviction due to idle time"
  ^JedisPool [host options]
  (let [{:keys [^String host port]} (util/parse-host-string host)]
    (if port
      (JedisPool. (pool-config options) host ^int port)
      (JedisPool. (pool-config options) host))))

(defn connection
  "Retrieves a connection resource from pool. Returns a `Jedis` instance."
  ^Jedis [^JedisPool pool]
  (.getResource pool))

(defn connected?
  "Checks if the provided connection resource is connected to redis."
  [^Jedis conn]
  (.isConnected conn))

(defn close
  "Closes a connection or a pool resource."
  [conn-or-pool]
  (.close conn-or-pool))

(defn closed?
  "Checks if a connection or pool resource has been closed."
  [conn-or-pool]
  (.isClosed conn-or-pool))