(ns goose.retry
  (:require
    [goose.brokers.redis.commands :as redis-cmds]
    [goose.defaults :as d]
    [goose.utils :as u]

    [clojure.tools.logging :as log]))

(defn default-error-handler
  "Default error handler of a Job.
  Called when a job fails.
  Logs exception & job details."
  [_ job ex]
  (log/error ex "Job execution failed." job))

(defn default-death-handler
  "Default death handler of a Job
  Called when a job fails & has exhausted retries.
  Logs exception & job details."
  [_ job ex]
  (log/error ex "Job retries exhausted." job))

(defn default-retry-delay-sec
  "Calculates backoff seconds
  before a failed Job is retried."
  [retry-count]
  (+ 20
     (* (rand-int 20) (inc retry-count))
     (reduce * (repeat 4 retry-count)))) ; retry-count^4

(def default-opts
  "Default config for Error Handling & Retries."
  {:max-retries            27
   :retry-delay-sec-fn-sym `default-retry-delay-sec
   :retry-queue            nil
   :error-handler-fn-sym   `default-error-handler
   :skip-dead-queue        false
   :death-handler-fn-sym   `default-death-handler})

(defn- prefix-retry-queue
  [retry-opts]
  (if-let [retry-queue (:retry-queue retry-opts)]
    (assoc retry-opts :prefixed-retry-queue (d/prefix-queue retry-queue))
    retry-opts))

(defn ^:no-doc prefix-queue-if-present
  [opts]
  (->> opts
       (prefix-retry-queue)
       (merge default-opts)))

(defn- failure-state
  [{{:keys [retry-count first-failed-at]} :state} ex]
  {:error           (str ex)
   :last-retried-at (when first-failed-at (u/epoch-time-ms))
   :first-failed-at (or first-failed-at (u/epoch-time-ms))
   :retry-count     (if retry-count (inc retry-count) 0)})

(defn- set-failed-config
  [job ex]
  (assoc
    job :state
        (failure-state job ex)))

(defn- retry-job
  [{:keys [redis-conn error-service-cfg]}
   {{:keys [retry-delay-sec-fn-sym
            error-handler-fn-sym]} :retry-opts
    {:keys [retry-count]}          :state
    :as                            job}
   ex]
  (let [error-handler (u/require-resolve error-handler-fn-sym)
        retry-delay-sec ((u/require-resolve retry-delay-sec-fn-sym) retry-count)
        retry-at (u/add-sec retry-delay-sec)
        job (assoc-in job [:state :retry-at] retry-at)]
    (u/log-on-exceptions (error-handler error-service-cfg job ex))
    (redis-cmds/enqueue-sorted-set redis-conn d/prefixed-retry-schedule-queue retry-at job)))

(defn- bury-job
  [{:keys [redis-conn error-service-cfg]}
   {{:keys [skip-dead-queue
            death-handler-fn-sym]} :retry-opts
    {:keys [last-retried-at]}      :state
    :as                            job}
   ex]
  (let [death-handler (u/require-resolve death-handler-fn-sym)
        dead-at (or last-retried-at (u/epoch-time-ms))
        job (assoc-in job [:state :dead-at] dead-at)]
    (u/log-on-exceptions (death-handler error-service-cfg job ex))
    (when-not skip-dead-queue
      (redis-cmds/enqueue-sorted-set redis-conn d/prefixed-dead-queue dead-at job))))

(defn ^:no-doc wrap-failure
  [next]
  (fn [opts job]
    (try
      (next opts job)
      (catch Exception ex
        (let [failed-job (set-failed-config job ex)
              retry-count (get-in failed-job [:state :retry-count])
              max-retries (get-in failed-job [:retry-opts :max-retries])]
          (if (< retry-count max-retries)
            (retry-job opts failed-job ex)
            (bury-job opts failed-job ex)))))))
