;; Copyright (c) 2011-2014 Michael S. Klishin
;;
;; The use and distribution terms for this software are covered by the
;; Eclipse Public License 1.0 (http://opensource.org/licenses/eclipse-1.0.php)
;; which can be found in the file epl-v10.html at the root of this distribution.
;; By using this software in any fashion, you are agreeing to be bound by
;; the terms of this license.
;; You must not remove this notice, or any other, from this software.

(ns langohr.core
  "Functions that work with connections and shared features between connections
   and channels (e.g. shutdown listeners).

   Relevant guides:

    * http://clojurerabbitmq.info/articles/connecting.html
    * http://clojurerabbitmq.info/articles/tls.html"
  (:import [com.rabbitmq.client Connection Channel Address
            ConnectionFactory ShutdownListener BlockedListener
            Consumer TopologyRecoveryException
            ExceptionHandler Recoverable RecoveryListener]
           [com.rabbitmq.client.impl DefaultExceptionHandler AMQConnection]
           [clojure.lang IFn]
           java.util.concurrent.ThreadFactory)
  (:require langohr.channel
            [clojure.string :as s]
            [clojure.walk   :as walk]))

;;
;; Implementation
;;

(def ^{:dynamic true :doc "Default connection settings."} *default-config*
  {:username "guest"
   :password "guest"
   :vhost     "/"
   :host      "localhost"
   :port      ConnectionFactory/DEFAULT_AMQP_PORT})

;;
;; API
;;

(defprotocol Closeable
  (close [c] "Closes given entity"))

(extend-protocol Closeable
  com.rabbitmq.client.Connection
  (close [this] (.close this))

  com.rabbitmq.client.Channel
  (close [this] (.close this)))


(defprotocol Openable
  (open? [this] "Checks whether given entity is open")
  (closed? [this] "Checks whether given entity is closed"))

(extend-protocol Openable
  com.rabbitmq.client.Connection
  (open? [conn] (.isOpen conn))
  (closed? [conn] (not (.isOpen conn)))

  com.rabbitmq.client.Channel
  (open? [ch] (.isOpen ch))
  (closed? [ch] (not (.isOpen ch))))

(declare create-connection-factory normalize-settings)
(defn- address-array-from
  [addresses port]
  (into-array Address
              (map (fn [arg]
                     (let [[host port] (if (coll? arg)
                                         [(first arg) (second arg)]
                                         [arg port])]
                       (Address. host port)))
                   (remove nil? addresses))))

(defn ^Connection connect
  "Creates and returns a new connection to RabbitMQ."
  ;; defaults
  ([]
     (let [^ConnectionFactory cf (create-connection-factory {})]
       (doto (com.novemberain.langohr.Connection. cf)
         .init)))
  ;; settings
  ([settings]
     (let [settings'             (normalize-settings settings)
           ^ConnectionFactory cf (create-connection-factory settings')
           xs                    (address-array-from (get settings' :hosts #{})
                                                     (get settings' :port ConnectionFactory/DEFAULT_AMQP_PORT))]
       (doto (com.novemberain.langohr.Connection. cf (dissoc settings :password :username))
         (.init xs)))))

(defn ^Channel create-channel
  "Delegates to langohr.channel/open, kept for backwards compatibility"
  [& args]
  (apply langohr.channel/open args))


(defn ^ShutdownListener shutdown-listener
  "Adds new shutdown signal listener that delegates to given function"
  [^clojure.lang.IFn f]
  (reify ShutdownListener
    (shutdownCompleted [this cause]
      (f cause))))

(defn ^ShutdownListener add-shutdown-listener
  "Adds a shutdown listener on connection and returns it"
  [^Connection c ^IFn f]
  (let [lnr (shutdown-listener f)]
    (.addShutdownListener c lnr)
    lnr))

(defn ^BlockedListener blocked-listener
  "Reifies connection.blocked and connection.unblocked listener from Clojure
   functions"
  [^IFn on-blocked ^IFn on-unblocked]
  (reify BlockedListener
    (^void handleBlocked [this ^String reason]
      (on-blocked reason))
    (^void handleUnblocked [this]
      (on-unblocked))))

(defn ^BlockedListener add-blocked-listener
  "Adds a connection.blocked and connection.unblocked listener
   on connection and returns it"
  [^Connection c ^IFn on-blocked ^IFn on-unblocked]
  (let [lnr (blocked-listener on-blocked on-unblocked)]
    (.addBlockedListener c lnr)
    lnr))

(defn settings-from
  "Parses AMQP connection URI and returns a persistent map of settings"
  [^String uri]
  (if uri
    (let [cf (doto (ConnectionFactory.)
               (.setUri uri))]
      {:host     (.getHost cf)
       :port     (.getPort cf)
       :vhost    (.getVirtualHost cf)
       :username (.getUsername cf)
       :password (.getPassword cf)})
    *default-config*))

(defn capabilities-of
  "Returns capabilities of the broker on the other side of the connection"
  [^Connection conn]
  (walk/keywordize-keys (into {} (-> conn .getServerProperties (get "capabilities")))))

;;
;; Recovery
;;

(defn automatic-recovery-enabled?
  "Returns true if provided connection uses automatic connection recovery
   mode, false otherwise"
  [^com.novemberain.langohr.Connection conn]
  (.automaticRecoveryEnabled conn))

(defn ^{:deprecated true} automatically-recover?
  "See automatic-recovery-enabled?"
  [^com.novemberain.langohr.Connection c]
  (automatic-recovery-enabled? c))

(defn automatic-topology-recovery-enabled?
  "Returns true if provided connection uses automatic topology recovery
   mode, false otherwise"
  [^com.novemberain.langohr.Connection conn]
  (.automaticTopologyRecoveryEnabled conn))

(defn on-recovery
  "Registers a network recovery callback on a (Langohr) connection or channel"
  [^Recoverable target ^IFn callback]
  (.addRecoveryListener target (reify RecoveryListener
                                 (^void handleRecovery [this ^Recoverable it]
                                   (callback it)))))


;;
;; Advanced Customization
;;

(defn thread-factory-from
  "Instantiates a java.util.concurrent.ThreadFactory that delegates
   #newThread to provided Clojure function"
  [f]
  (reify java.util.concurrent.ThreadFactory
    (^Thread newThread [this ^Runnable r]
      (f r))))

(defn exception-handler
  [{:keys [handle-connection-exception-fn
           handle-return-listener-exception-fn
           handle-flow-listener-exception-fn
           handle-confirm-listener-exception-fn
           handle-blocked-listener-exception-fn
           handle-consumer-exception-fn
           handle-connection-recovery-exception-fn
           handle-channel-recovery-exception-fn
           handle-topology-recovery-exception-fn]}]
  (proxy [DefaultExceptionHandler] []
    (handleUnexpectedConnectionDriverException [^Connection conn ^Throwable t]
      (when handle-connection-exception-fn
        (handle-connection-exception-fn conn t)))
    (handleReturnListenerException [^Channel ch ^Throwable t]
      (when handle-return-listener-exception-fn
        (handle-return-listener-exception-fn ch t)))
    (handleFlowListenerException [^Channel ch ^Throwable t]
      (when handle-flow-listener-exception-fn
        (handle-flow-listener-exception-fn ch t)))
    (handleConfirmListenerException [^Channel ch ^Throwable t]
      (when handle-confirm-listener-exception-fn
        (handle-confirm-listener-exception-fn ch t)))
    (handleBlockedListenerException [^Connection conn ^Throwable t]
      (when handle-blocked-listener-exception-fn
        (handle-blocked-listener-exception-fn conn t)))
    (handleConsumerException [^Channel ch ^Throwable t
                              ^Consumer consumer ^String consumer-tag
                              ^String method-name]
      (when handle-consumer-exception-fn
        (handle-consumer-exception-fn ch t consumer consumer-tag method-name)))
    (handleConnectionRecoveryException [^Connection conn ^Throwable t]
      (when handle-connection-recovery-exception-fn
        (handle-connection-recovery-exception-fn conn t)))
    (handleChannelRecoveryException [^Channel ch ^Throwable t]
      (when handle-channel-recovery-exception-fn
        (handle-channel-recovery-exception-fn )))
    (handleTopologyRecoveryException [^Connection conn ^Channel ch
                                      ^TopologyRecoveryException t]
      (when handle-topology-recovery-exception-fn
        (handle-topology-recovery-exception-fn conn ch t)))))

;;
;; Implementation
;;

(defn normalize-settings
  "For setting maps that contain keys such as :host, :username, :vhost, returns the argument"
  [config]
  (let [{:keys [host hosts]} config
        hosts' (into #{} (remove nil? (or hosts #{host})))]
    (merge (settings-from (:uri config (System/getenv "RABBITMQ_URL")))
           {:hosts hosts'}
           config)))

(defn- platform-string
  []
  (let []
    (format "Clojure %s on %s %s"
            (clojure-version)
            (System/getProperty "java.vm.name")
            (System/getProperty "java.version"))))

(def ^{:private true}
  client-properties {"product"      "Langohr"
                     "information"  "See http://clojurerabbitmq.info/"
                     "platform"     (platform-string)
                     "capabilities" (get (AMQConnection/defaultClientProperties) "capabilities")
                     "copyright"    "Copyright (C) 2011-2014 Michael S. Klishin, Alex Petrov"
                     "version"      "3.0.x"})

(defn- ^ConnectionFactory create-connection-factory
  "Creates connection factory from given attributes"
  [settings]
  (let [{:keys [host port username password vhost
                requested-heartbeat connection-timeout ssl ssl-context socket-factory sasl-config
                requested-channel-max thread-factory exception-handler]
         :or {requested-heartbeat ConnectionFactory/DEFAULT_HEARTBEAT
              connection-timeout  ConnectionFactory/DEFAULT_CONNECTION_TIMEOUT
              requested-channel-max ConnectionFactory/DEFAULT_CHANNEL_MAX}} (normalize-settings settings)
              cf   (ConnectionFactory.)
              final-port (if (and ssl (= port ConnectionFactory/DEFAULT_AMQP_PORT))
                           ConnectionFactory/DEFAULT_AMQP_OVER_SSL_PORT
                           port)]
    (when (or ssl
              (= port ConnectionFactory/DEFAULT_AMQP_OVER_SSL_PORT))
      (.useSslProtocol cf))
    (doto cf
      (.setClientProperties   client-properties)
      (.setUsername           username)
      (.setPassword           password)
      (.setVirtualHost        vhost)
      (.setHost               host)
      (.setPort               final-port)
      (.setRequestedHeartbeat requested-heartbeat)
      (.setConnectionTimeout  connection-timeout)
      (.setRequestedChannelMax requested-channel-max))
    (when sasl-config
      (.setSaslConfig cf sasl-config))
    (when ssl-context
      (.useSslProtocol cf ^javax.net.ssl.SSLContext ssl-context))
    (when thread-factory
      (.setThreadFactory cf ^ThreadFactory thread-factory))
    (when exception-handler
      (.setExceptionHandler cf ^ExceptionHandler exception-handler))
    cf))

