;; This source code is dual-licensed under the Apache License, version
;; 2.0, and the Eclipse Public License, version 1.0.
;;
;; The APL v2.0:
;;
;; ----------------------------------------------------------------------------------
;; Copyright (c) 2011-2016 Michael S. Klishin, Alex Petrov, and the ClojureWerkz Team
;;
;; Licensed under the Apache License, Version 2.0 (the "License");
;; you may not use this file except in compliance with the License.
;; You may obtain a copy of the License at
;;
;;     http://www.apache.org/licenses/LICENSE-2.0
;;
;; Unless required by applicable law or agreed to in writing, software
;; distributed under the License is distributed on an "AS IS" BASIS,
;; WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
;; See the License for the specific language governing permissions and
;; limitations under the License.
;; ----------------------------------------------------------------------------------
;;
;; The EPL v1.0:
;;
;; ----------------------------------------------------------------------------------
;; Copyright (c) 2011-2016 Michael S. Klishin, Alex Petrov, and the ClojureWerkz Team.
;; All rights reserved.
;;
;; This program and the accompanying materials are made available under the terms of
;; the Eclipse Public License Version 1.0,
;; which accompanies this distribution and is available at
;; http://www.eclipse.org/legal/epl-v10.html.
;; ----------------------------------------------------------------------------------

(ns langohr.consumers
  "Functions that instantiate consumers and provide convenient ways of
   registering them.

   Relevant guides:

   * http://clojurerabbitmq.info/articles/queues.html"
  (:require [langohr.basic :as lhb]
            [langohr.conversion :refer :all]
            [langohr.channel :refer [as-non-recovering-channel]])
  (:import [com.rabbitmq.client Channel Consumer DefaultConsumer QueueingConsumer
            QueueingConsumer$Delivery ShutdownSignalException Envelope
            AMQP$BasicProperties QueueingConsumer$Delivery]))

;;
;; API
;;

(defn ^Consumer create-default
  "Instantiates and returns a new consumer that handles various consumer life cycle events. See also langohr.basic/consume."
  [^Channel channel {:keys [handle-consume-ok-fn
                            handle-cancel-fn
                            handle-cancel-ok-fn
                            handle-shutdown-signal-fn
                            handle-recover-ok-fn
                            handle-delivery-fn]}]
  (proxy [DefaultConsumer] [(as-non-recovering-channel channel)]
    (handleConsumeOk [^String consumer-tag]
      (when handle-consume-ok-fn
        (handle-consume-ok-fn consumer-tag)))

    (handleCancelOk [^String consumer-tag]
      (when handle-cancel-ok-fn
        (handle-cancel-ok-fn consumer-tag)))


    (handleCancel [^String consumer-tag]
      (when handle-cancel-fn
        (handle-cancel-fn consumer-tag)))

    (handleRecoverOk []
      (when handle-recover-ok-fn
        (handle-recover-ok-fn)))

    (handleShutdownSignal [^String consumer-tag ^ShutdownSignalException sig]
      (when handle-shutdown-signal-fn
        (handle-shutdown-signal-fn consumer-tag sig)))

    (handleDelivery [^String consumer-tag ^Envelope envelope ^AMQP$BasicProperties properties ^bytes body]
      (when handle-delivery-fn
        (handle-delivery-fn channel (to-message-metadata (QueueingConsumer$Delivery. envelope properties body)) body)))))


(defn ^Consumer create-queueing
  "Instantiates and returns a new queueing consumer that handles various consumer life cycle events. See also langohr.basic/consume.

   Consumer tag must be unique per queue. By default it will be generated by RabbitMQ and thus
   guaranteed to be unique."
  [^Channel channel {:keys [handle-consume-ok-fn
                            handle-cancel-ok-fn
                            handle-recover-ok-fn
                            handle-delivery-fn]}]
  (proxy [QueueingConsumer] [(as-non-recovering-channel channel)]
    (handleConsumeOk [^String consumer-tag]
      (when handle-consume-ok-fn
        (handle-consume-ok-fn consumer-tag)))

    (handleCancelOk [^String consumer-tag]
      (when handle-cancel-ok-fn
        (handle-cancel-ok-fn consumer-tag)))

    (handleRecoverOk []
      (when handle-recover-ok-fn
        (handle-recover-ok-fn)))))

(defn subscribe
  "Adds a new default consumer to a queue using the basic.consume AMQP 0.9.1 method.

   Consumer-related options will be passed to `create-default`. Consumer tag
   must be unique per queue. By default it will be generated by RabbitMQ and thus
   guaranteed to be unique."
  ([^Channel ch ^String queue f]
     (subscribe ch queue f {}))
  ([^Channel ch ^String queue f options]
     (let [keys      [:handle-consume-ok :handle-cancel :handle-cancel-ok :handle-recover-ok :handle-shutdown-signal]
           keys      (concat keys (map #(keyword (str (name %) "-fn")) keys))
           cons-opts (select-keys options keys)
           options'  (dissoc options keys)
           consumer  (create-default ch
                                     {:handle-delivery-fn f
                                      :handle-consume-ok-fn      (or (get cons-opts :handle-consume-ok-fn)
                                                                     (get cons-opts :handle-consume-ok))
                                      :handle-cancel-ok-fn       (or (get cons-opts :handle-cancel-ok-fn)
                                                                     (get cons-opts :handle-cancel-ok))
                                      :handle-cancel-fn          (or (get cons-opts :handle-cancel-fn)
                                                                     (get cons-opts :handle-cancel))
                                      :handle-recover-ok-fn      (or (get cons-opts :handle-recover-ok-fn)
                                                                     (get cons-opts :handle-recover-ok))
                                      :handle-shutdown-signal-fn (or (get cons-opts :handle-shutdown-signal-fn)
                                                                     (get cons-opts :handle-shutdown-signal))})]
       (lhb/consume ch queue consumer options'))))

(defn deliveries-seq
  "Builds a lazy seq of delivery instances from a queueing consumer."
  [^QueueingConsumer qcs]
  (lazy-seq (cons (.nextDelivery qcs) (deliveries-seq qcs))))

(defn ack-unless-exception
  "Wrapper for delivery handlers which auto-acks messages.

   This differs from `:auto-ack true', which tells the broker to
   consider messages acked upon delivery. This explicitly acks, as
   long as the consumer function doesn't throw an exception."
  [f]
  (fn [^Channel channel {:keys [delivery-tag] :as metadata} body]
    (f channel metadata body)
    (.basicAck channel delivery-tag false)))

(defn blocking-subscribe
  "Adds new a QueueingConsumer to a queue using the basic.consume AMQP 0.9.1 method"
  ([^Channel ch ^String queue f]
     (blocking-subscribe ch queue f {}))
  ([^Channel channel ^String queue f options]
     (let [consumer (QueueingConsumer. channel)]
       (lhb/consume channel queue consumer options)
       (doseq [^QueueingConsumer$Delivery d (deliveries-seq consumer)]
         (f channel (to-message-metadata d) (.getBody d))))))
