(ns orcette.conf
  (:require [clojure.string :as s])
  (:import [java.io File]))

;;
;; API
;;

(defprotocol FileBasedConfiguration
  (expand    [path] "Expands filesystem path to be absolute")
  (load-from [path] "Reads and evaluates configuration from given resource."))

(extend-protocol FileBasedConfiguration
  File
  (expand    [f] (.getAbsolutePath f))
  (load-from [f] (read-string (with-open [rdr (clojure.java.io/reader f)]
                                (reduce str "" (line-seq rdr)))))

  String
  (expand    [s] (.getAbsolutePath (File. s)))
  (load-from [s] (load-from (File. s))))


(defn conventional-config-path-for
  "Returns an absolute path for conventions-based configuration file for given environment. For example, for test environment
   the result would be absolutized ./config/test.clj path"
  [^String environment]
  (expand (s/join (System/getProperty "file.separator") ["config" (str environment ".clj")])))

(defn config-path-for
  "Takes a map of options as produced by the tools.cli library and returns configuration file location.
   One of the keys should be given: :environment (development, test, staging or production) or :config-file (path to local configuration file)"
  [{ :keys [config-file environment] }]
  (expand (if config-file
            config-file
            (conventional-config-path-for environment))))

(defn config-for
  "Takes a map of options as produced by the tools.cli library and returns merged configuration
   for given environment.
   One of the keys should be given: :environment (development, test, staging or production) or :config-file (path to local configuration file)"
  [options]
  (when-let [path (config-path-for options)]
    (merge options (load-from path))))
