(ns pallet.compute.test-provider
  "A programmable provider for testing"
  (:require
   [clojure.string :as string]
   [pallet.algo.fsm.event-machine :refer [update-state]]
   [pallet.algo.fsm.fsm-dsl
    :refer [event-handler
            event-machine-config
            initial-state
            state
            valid-transitions]]
   [pallet.compute :as compute]
   [pallet.compute.implementation :as implementation]
   [pallet.utils :refer [apply-map]]))

(defrecord Node
    [name group-name ip os-family os-version id ssh-port private-ip is-64bit
     running service]
  pallet.node.Node
  (ssh-port [node] ssh-port)
  (primary-ip [node] ip)
  (private-ip [node] private-ip)
  (is-64bit? [node] (:is-64bit node))
  (group-name [node] group-name)
  (running? [node] running)
  (terminated? [node] (not running))
  (os-family [node] os-family)
  (os-version [node] os-version)
  (hostname [node] name)
  (id [node] id)
  (compute-service [node] service))

;;; Node utilities
(defn make-node
  "Returns a node, suitable for use in a node-list."
  [group-name
   & {:keys [name ip os-family id ssh-port private-ip is-64bit running
             os-version service]
      :or {ssh-port 22 is-64bit true running true service (atom nil)}}]
  (Node.
   name
   group-name
   ip
   os-family
   os-version
   (or id (str name "-" (string/replace ip #"\." "-")))
   ssh-port
   private-ip
   is-64bit
   running
   service))

(defn process-cmd
  [fsm cmd & args]
  ((:event fsm) :cmd args)
  (let [{:keys [state-data] :as state} ((:state fsm))
        {:keys [delay exception reply]} state-data]
    (when delay
      (Thread/sleep delay))
    (if exception
      (throw exception)
      reply)))

(deftype TestProvider
    [fsm environment]
  pallet.compute.ComputeService
  (nodes [compute-service] (process-cmd fsm :nodes))
  (run-nodes [compute node-type node-count user init-script options]
    (process-cmd
     fsm :run-nodes node-type node-count user init-script options))
  (reboot [compute nodes] (process-cmd fsm :reboot nodes))
  (boot-if-down [compute nodes] (process-cmd fsm :boot-if-down nodes))
  (shutdown-node [_ node user] (process-cmd fsm :shutdown-node node user))
  (shutdown [_ nodes user] (process-cmd fsm :shutdown nodes user))
  (ensure-os-family [compute-service group-spec]
    (process-cmd fsm :ensure-os-family group-spec))
  (destroy-nodes-in-group [compute group-name]
    (process-cmd fsm :destroy-nodes-in-group group-name))
  (destroy-node [compute node]
    (process-cmd fsm :destroy-node node))
  (images [compute]
    (process-cmd fsm :images))
  (close [compute]
    (process-cmd fsm :close))
  pallet.environment.Environment
  (environment [_] environment))


;;;; ## Compute Service SPI
(defn supported-providers
  {:no-doc true
   :doc "Returns a sequence of providers that are supported"}
  [] ["test-provider"])

(defmethod implementation/service :test-provider
  [_ {:keys [fsm environment]}]
  (TestProvider. fsm environment))

;;;; ## Compute service constructor
(defn test-provider-service
  "Create a node-list compute service, based on a sequence of nodes. Each
   node is passed as either a node object constructed with `make-node`,
   or as a vector of arguments for `make-node`.

   Optionally, an environment map can be passed using the :environment keyword.
   See `pallet.environment`."
  {:added "0.6.8"}
  [fsm & {:keys [environment] :as options}]
  (apply-map
   compute/instantiate-provider :test-provider (assoc options :fsm fsm)))

;;; ## Default fsm
(defn list-nodes
  [{:keys [state-data]}]
  (:nodes state-data))

(defn reply-with-delay
  [state reply]
  (update-state state :delay assoc :reply reply))

(defn default-idle-handler
  [state event event-data]
  (case event
    :nodes (reply-with-delay state (list-nodes state))))

(defn default-delay-handler
  [state event event-data]
  (case event
    :nodes (reply-with-delay state (list-nodes state))))

(def default-fsm
  (event-machine-config
    (initial-state :idle)
    (state :idle
      (valid-transitions :idle :delay)
      (event-handler default-idle-handler))
    (state :delay
      (valid-transitions :possible-error))
    (state :possible-error
      (valid-transitions :idle))))
