(ns
 noon.doc.noon
 (:require
  sci-utils/sci-namespaces
  [noon.updates :refer :all]
  [noon.events :as events :refer [ef_ efn]]
  [noon.score :as score :refer [score sf_ sfn e->s]]
  [noon.output :as out :refer [noon play]]
  [noon.harmony :as hc]
  [noon.midi :as midi]
  [noon.numbers :refer [mul div add sub]]
  [noon.lib.harmony :as h]
  [noon.lib.melody :as m]
  [noon.lib.rythmn :as r]
  [noon.utils.misc :as u]
  [noon.constants :as constants]
  [noon.utils.pseudo-random :as rand]
  [noon.utils.sequences :as seqs]
  [noon.vst.general-midi]
  [clojure.math.combinatorics :as combinatorics]
  [noon.utils.multi-val :as multi-val]
  [noon.vst.vsl :as vsl :refer [vsl]]))


;; #+OPTIONS: H:9
["Noon"

  ;; Noon is a clojure library that helps with *musical composition and exploration*.
  ;; It provides a way to compose, play and export music as *MIDI*.

  ;; The initial goal was to be able to define musical fragments in a very declarative way.
  ;; Leveraging clojure and functional programming powers in order to *generate, manipulate and organize* musical content.

  ["Motivation"

   ;; Firstly, I was somewhat frustrated by the experience of computer-assisted music composition.
   ;; The tools commonly used for this, called digital audio workstations (DAWs), which include for example Logic Audio or Cubase, are certainly powerful, but quite laborious in use in my opinion.
   ;; The feedback loop seems a bit long to me. Changes are repetitive and quite slow, we spend a lot of time clicking, drag and dropping, and little time actually listening.

   ;; On the other hand, having primarily studied jazz, I would have liked to find the same kind of approach when composing with a computer.
   ;; In jazz (or other non classical music genres), a composition often consists of a melody and a chord grid. Its realization is largely left to the musician who interprets it.
   ;; This allows a composition to be relatively different from one version to another, compared to a piece composed in MIDI in a digital audio workstation which is quite rigid.

   ;; The improvisation and interpretation techniques used by musicians can be partially emulated by a computer.
   ;; Of course, what makes the essence of a very beautiful improvisation or interpretation will probably never be artificially reproduced, nevertheless it would be interesting to see how far we can go. Certain musical contexts and techniques require great speed of execution or calculation, and for that the machine is unbeatable.

   ;; Trying to use already existing musical libraries, I felt that this level of abstraction was not accessible to a degree that suits my needs.
   ;; - The [[https://github.com/overtone/overtone][overtone library]] is focusing mainly on sound synthesis, but not much on high level musical abstraction.
   ;; - The [[https://github.com/daveyarwood/alda-clj][alda library]] is low level, close to the MIDI representation in terms of abstraction level.
   ;; - The [[https://github.com/ctford/leipzig][leipzig library]] is aiming in this direction but I think we can go even further.

   ;; To sum up a bit, we could say that the level of musical abstraction currently in force in digital audio workstations is quite low.
   ;; In contrast to the level of abstraction generally used by musicians to think about music.
   ;; A level of abstraction that is too low will require a lot of work to realize ideas that are thought of quickly, and this leads to the relative slowness of the feedback cycle.

   ;; One of the objectives of this language will therefore be to allow the user to manipulate higher level musical abstractions, such as for example:
   ;; - a modulation
   ;; - a melodic contour
   ;; - a melodic or harmonic, diatonic or chromatic approach or embellishment.
   ;; - voice leading
   ;; - an inversion of a chord or melody.
   ;; The kind of ideas that musicians use to talk about music in a more concise and above all more general way.

  ]

  ["Elements"

   ["Building blocks"

    ["Events"

     ;; An event is the basic building block we are dealing with.

     ;; It can represent any MIDI event, a note, a control change etc...

     ;; It is represented using a clojure map

     noon.events/DEFAULT_EVENT

    ]

    ["Score"

     ;; A score is a collection of events, represented using a clojure set.

     (score) ; the `noon.eval/score` function is used to build score, more on it later

    ]]

   ["Top form"

    ;; =noon.eval/noon= is the top level form of the library.

    '(noon <option-map> <score>)

    ;; Here a minimal example:

    (noon
     ;; the play option let you play the given score
     {:play true}
     ;; calling score without argument just build the default score (middle C)
     (score))

    ["Options"

     ;; The option map let you stay what you want to do with your score.

     ;; - =:midi true=
     ;;   Write the score as MIDI file.
     ;;   Default to false.

     ;; - =:bpm <number>=
     ;;   Set the tempo to <number> beats per minute.
     ;;   Default value is 60.

     ;; - =:play true=
     ;;   Play the score.
     ;;   Default to false.

     ;; - =:filename <path>=
     ;;   Setting the name and location of emitted files.
     ;;   File extensions will be appended to this filename.

     ;; - =:tracks {<track-idx> <sequencer> ...}=
     ;;   Set each track idxs to a =javax.sound.midi.Sequencer=
     ;;   This library comes with 4 nice soundfonts that sounds way better than builtin one (=:default=):
     ;;   1. =:squid=
     ;;   2. =:chorium=
     ;;   3. =:fluid=
     ;;   4. =:airfont=

     ;; example:

     (noon {:play true
            :tracks {0 :chorium}} ;; try to change soundfont here

           ;; this will be explained later
           ;; it repeats an ascending scale with different patches
           ;; in order to demonstrate the soundfont
           (score dur2
                  (rup 8 d1)
                  (lin (patch :clarinet)
                       (patch :electric-piano-1)
                       (patch :trumpet)
                       (patch :ocarina))))

     ;; In addition to those soundfonts, you can send the output of noon to any output device available on your machine.

     (require 'noon.midi)
     ;; retrieve a device by name
     (def bus1 (noon.midi/get-output-device "Bus 1"))
     ;; build a sequencer from it
     (def bus1-sequencer (noon.midi/init-device-sequencer bus1))
     ;; use it to play a score
     (noon {:play true
            :tracks {0 bus1-sequencer}}
           (score (par s0 s1 s2)))

     ["Musescore options"

      ;; If you have [[https://musescore.org/en][musecore]] installed on your machine, you can emit music XML and pdf score.

      ;; - =:xml true=
      ;;   write the score as musicXML file.

      ;; - =:pdf true=
      ;;   write the score pdf file.

     ]

     ["mp3 export"

      ;; It is possible to create an mp3 file by passing this option:

      ;; =:mp3 true=

      (noon {:mp3 true}
            (score (tup s0 s1 s2)))

      ;; [[https://ffmpeg.org/][FFmpeg]] and [[https://www.fluidsynth.org/][FluidSynth]] have to be installed on your machine.

     ]]

    ["score"

     ;; As we've just seen, we can create a score with the =score= function.
     ;; With no arguments it simply returns the default score containing only a middle C.

     (score)

     ;; The =score= function can take any number of arguments, each one being a score transformation.

     ;; Those transformations are applied in order to the default score.

     '(score transformation1 transformation2 ...)

    ]

    ["=noon.eval/play="

     ;; As a convenience, this thin =noon.eval/noon= wrapper let you play a score with fewer keystrokes:

     '(play transformation1 transformation2 ...)

     ;; Which is is roughly equivalent to:

     '(noon {:play true}
            (score transformation1 transformation2 ...))

     ;; More concretly:

     (play dur2
           (tup s0 s1 s2 s3))

    ]]

   ["Transformations 1"

    ;; There is a bunch of transformations available, let's see the basics.

    ["Pitches"

     ;; We can set the current pitch by using pitch vars.

     ;; Pitch vars names consist of a pitch-class name followed by an octave offset.
     ;; (pitch classes are simply musical notes names like C, Db, F#, E, Bbb, Fx (x mean double sharp)).
     ;; The middle C is named C0, the C above is C1, the C below is C-1.

     ;; Here some examples of pitches:

     (play Eb0)

     (play F#-1)

     (play Gb2)

     ;; Pitches are not often used as is, we will prefer more relative constructs like intervals, patterns etc...
     ;; But it may be a little overwhelming to start with, so for now we will use them to introduce the basics building blocks of the system.

    ]

    ["Durations"

     ;; We can operate on durations by multiplying or dividing them.

     (play dur2) ; multiplies the duration of our middle C by 2

     (play dur:3) ; divides it by 3

     ;; There is also a more flexible (and verbose) way to build duration transformations.

     (dur 2) ; sets the duration to 2

     (dur 1/4) ; sets the duration to 1/4

     (dur (fn [x] (* x 2))) ; multiply by 2 the current duration.

     ;; Those 3 forms return a transformation that can be used in =score= or =play=

     (play (dur 1/4))

    ]

    ["Velocities"

     ;; Velocity is the force with which a note is played, and it is vitally important in making MIDI performances sound human.

     ;; In midi the velocity is a number between 0 and 127.

     ;; For easing notation, 12 levels of velocity are defined as vars.

     (play vel0) ; silent

     (play vel3) ; piano

     (play vel8) ; forte

     (play vel12) ; fortissimo

     ;; Like for duration there is also a more flexible form:

     (play (vel 100)) ; sets the midi velocity of the current event to 100 (forte).

     (play (vel (fn [x] (/ x 2)))) ; divide the current velocity by 2 (by default the velocity is 80)

    ]

    ["Composition"

     ;; We can compose any number of transformations together using a clojure vector.

     (play [Eb0 dur:2]) ; plays a Eb of half duration

     (play [F#-1 dur4 (vel 127)]) ; F# above the middle C with quadruple duration and max velocity.

     (play [(vel 127) dur4 F#-1]) ; the order do not matter in this case.

     ;; The =play= and =score= forms, when given several arguments are doing exactly this

     (play F#-1 dur4) ; is the same as (play [F#-1 dur4])

    ]

    ["Concatenation"

     ;; Using the =lin= function we can create our first melody.
     ;; The =lin= function takes an arbitrary number of transformations and concatenate their results into one score.

     (play (lin C0 E0 G0 B0))

     ;; =lin= accept any valid transformation, here we are using composite transformations.

     (play (lin [C0 dur:2]
                [Eb0 dur:4]
                [G0 dur:4]
                C1))

    ]

    ["Superposition"

     ;; Using the =par= function we can stack things up.

     (play (par C0 Eb0 G0)) ; a C minor chord.

     ;; A pianissimo, double duration, Csus4 chord:

     (play vel2
           dur2
           (par C0 F0 G0))

    ]

    ["Sounds"

     ;; By default, we are using general MIDI to emit sounds, it is not the most exciting way to play MIDI but it is everywhere and gives you a rapid feedback without extra setup.

     ;; Of course if you want to use fancy VSTs in a proper DAW you can, one of the feature of this library is to export MIDI files after all.

     ;; Here how you can leverage general MIDI sounds:

     (play (patch :clarinet) (lin C0 E0 G#0 B0))

     (play (patch :vibraphone) [dur:4 (lin C0 E0 G0 (par D1 B0))])

     ;; You can look at what is available here

     noon.vst.general-midi/summary

    ]

    ["Channels"

     ;; In most of the tunes we write, we want several instruments playing together.

     ;; In MIDI there is this concept of channel, it serve the purpose to separate different streams of events.

     (play
      (chans
       [(patch :ocarina) dur:2 (lin G0 Eb0 C0 G-1 F0 D0 A-1 F-1)]
       [(patch :vibraphone) dur2 vel3 (lin (par C0 Eb0 G0) (par A-1 F0 D0))]
       [(patch :acoustic-bass) (lin [dur3 C-2] G-2)])
      (dup 4))

    ]]

   ["Transformations 2"

    ["Intervals 1"

     ;; It is now time to brings intervals into the equation, pitches were nice for introduction purposes but lacks the flexibility that intervals have. When musicians think about music, they do not think in precise pitches most of the time, they more often thinks of scales, intervals, degrees, melodic contour etc... Those higher level abstractions are available in this system and in fact it is the whole point of it. Some really nice libraries already exists to deal with low levels aspects of music notation and sound synthesis.

     ;; In noon there is two types of intervals: *steps* and *shifts*.

     ["Steps"

      ;; Steps are the most commonly used type of interval.

      ;; The 2 most common types of steps are chromatic steps and diatonic steps

      ["Chromatic"

       ;; A chromatic step is a movement by semitones.

       (c-step 3) ; going up 3 semitones from wherever we are.

       (c-step -1) ; going down one semitone

       ;; Those kind of transformation are so common that they are available as vars:

       c1 ; equivalent to (c-step 1)

       c2- ; equivalent to (c-step -2)

       ;; All chromatic steps from =c36= to =c36-= are available.

       ;; If we apply the =c3= step to the default score, it transpose the default middle C (=C0=) 3 semitones up to =Eb0= (or =D#0=).

       (play c3)

       (play (c-step -3)) ; going down 3 semitones to A-1

       (play c12-) ; going 12 semitones down (one octave) to C-1

      ]

      ["Diatonic"

       ;; A diatonic step is a movement toward a note that belong to the current scale.

       (d-step 1) ; move to the upper scale note (or degree).

       (d-step -1) ; moves to the above scale note (or degree).

       (d-step 4) ; moves four scale degree up...

       ;; Those kind of transformation are so common that they are available as vars:

       d1 ; is equivalent to (d-step 1)

       d2- ; is equivalent to (d-step -2)

       ;; all diatonic steps from =d21= to =d21-= are available.

       ["Example"

        (play dur:4 (lin d0 d1 d2 d3 d4 d5 d6 d7)) ; ascending scale

        (play dur:4 (lin d0 d2 d1 d3 d2 d4 d3 d5 d4)) ; broken scale pattern

        (play dur:4 (lin d0 d2- d1- d3- d2- d4- d3- d5- d4-)) ; same downward

        ;; By default, we are in the C major scale, but of course it can be changed. (see [[*Harmony][Harmony]] section)

        ;; As a quick example, pretty self explanatory (but explained in more details later).

        (play dur:4 (root :Eb) (scale :hungarian) (lin d0 d1 d2 d3 d4 d5 d6 d7))

        ;; There is 2 more type of steps: *structural* and *tonic*, but we will see them later.

       ]]]

     ["Octaves"

      ;; Paraphrasing wiki:
      ;; #+BEGIN_QUOTE
      ;; In music, an octave is the interval between one musical pitch and another with double its frequency. The octave relationship is a natural phenomenon that has been referred to as *the basic miracle of music*. The interval between the first and second harmonics of the harmonic series is an octave.
      ;; #+END_QUOTE

      ;; In noon, octaves are a different kind of interval, they belong to the =shift= family.

      ;; The nuance will appear more clearly later... Until then, let see how to use them:

      (play (t-shift 1)) ; one octave up.

      (play (t-shift -1)) ; one octave down.

      (play o2-) ; 2 octaves down in var notation

     ]]

    ["=lin="

     ;; As we have seen, =lin= let you create a succession of events:

     (play (lin C0 E0 G0 B0))

     ;; Let's try to go further with it by composing it with another =lin=:

     (play dur:8 (lin c0 c3 c6) (lin c0 c2 c3 c5))

     ;; Let see what happens here:

     ;; 3 transformations are chained:

     ;; 1. We are dividing the duration of our base note by 8.
     ;; 2. We are creating a series of 3 notes using chromatic intervals (diminished triad C,Eb,Gb).
     ;; 3. Then this 3 notes score is passed to each member of the second =lin= expression, each one transposing it from the indicated chromatic interval.

    ]

    ["=tup="

     ;; =tup= stands for tuplet and is analogous to =lin= but keep the duration of the given score unchanged.

     (play (tup c1 c2 c3 c4 c5 c6 c7 c8))

     ;; The resulting notes are fitted into the duration of the base note.

     ;; Like =lin= it can of course be chained with other transformations, as an example, here is a classic jazz melodic pattern.

     (play (tup c0 c2 c4 c7) (tup c0 c3) (rep 3 c4-))

    ]

    ["=dup="

     ;; =dup= stands for duplicate and let you repeat a score n times.

     (play (tup c0 c3 c6 c9) (dup 3))

    ]

    ["=rep="

     ;; =rep= let you apply a transformation several times in a row accumulating intermediate results.

     ;; A melody of 8 successive major thirds (4 semitones):

     (play dur:4 (rep 8 c4))

     ;; Be careful, with more complex transformations it can quickly become hairy:

     (play (rep 6 (tup c5 c10)))

     ;; You can remove the input score at the start of the result by giving an extra argument:

     (play (rep 3 o1 :skip-first))

    ]

    ["=fit="

     ;; =fit= is used to make a transformation fit the current duration of the score.
     ;; The 2 previous transformations introduced: =dup= and =rep=, are changing the score duration, but sometimes we want to transform our score in place, stretching or compressing it, in the same way =tup= is acting.

     (play (tup c0 c4) (fit (rep 4 c2)))

     ;; In fact =tup= is just a composition of =fit= and =lin=.

     (= (score (tup c0 c3 c8)) (score (fit (lin c0 c3 c8))))

     ;; The composition of =fit= and =rep= is also defined as =rup= for lack of a better name:

     (play (rup 15 d1))

     ;; A fitted version of =dup= also exists under the name =dupt=

     (play (tup d0 d3 d6 d7) (dupt 3))

    ]

    ["=nlin="

     ;; concat the results of the given transformation n times

     (play (nlin 4 (tup d0 d1 d2 d3)))

     ;; it is the same thing as:

     (play (tup d0 d1 d2 d3) (dup 4))

    ]

    ["=ntup="

     ;; the fitted version of =nlin=

     (play (ntup 4 (tup d0 d1 d2 d3)))

    ]

    ["=lin>="

     ;; =lin>= stands for accumulative concatenation, it accumulates the given transformations concatenating the intermediate results.

     (play (lin> c0 c2 c2 c2 c2 c2 c2))

    ]

    ["=tup>="

     ;; =tup>= is doing the same as =lin>=, except it maintains the score original duration.

     (play (tup> d0 d1 d1 d1 d1 d1 d1 d1))

    ]]

   ["Polyphony"

    ;; As we have seen, we can parallelize things with the =par= function.

    (play (par c0 c3 c7 c9 c14)) ; a Cm69 chord.

    (play (par c10 c0 c16 c5)) ; a C7sus4add10 using set literal

    ;; But we are not limited to use simple intervals, we can use any score transformations.

    (play
     (patch :electric-piano-1)
     (par (tup d0 d2 d4 o1)
          [vel3 (par> o1 d4) (fit (rep 8 d1))]
          o1-))

    ;; Parallels transformations can be used anywhere of course. Here inside a =tup=.

    (play o1
          (tup c0 (par c15 c10)
               c9 (par c6 c4))
          (rep 3 c3))

    (play (par (rep 12 c1)
               (rep 12 c1-)))

    ;; Like =lin= and =tup=, =par= has its accumulative counterpart:

    (play (par> d0 d2 d2 d2 d2)) ; piling diatonic thirds.

    (play (patch :string-ensemble-1)
          o2-
          (par> c0 c7 c7 c7 c7 c7)) ; piling perfect fifths.

    ["Channels"

     ;; the =chans= function is doing the same thing as =par= except that it put each element on a separate MIDI channel.

     (play (chans c0 c3 c7))

     ;; To be more precise it put each of its argument on subsequent midi channels starting at the current one. By default, we are on channel 0, so here the C will stay on channel 0, the Eb will go on channel 1 and the G on channel 2.

     ;; When we want more fine control we can use the =chan= function, that works like =vel= and =dur=

     (chan 1) ; set midi channel to 1

     (chan 3) ; set midi channel to 3

     (chan inc) ; increment the current midi channel.

     ;; We can achieve the same thing as the first expression of the section using =par= and =chan= like this:

     (play (par [(chan 0) c0]
                [(chan 1) c3]
                [(chan 2) c7]))

    ]

    ["Tracks"

     ;; Tracks are a way of not be limited to only 16 channels, you can create virtually as many as you want. Most of the time, 16 channels are enough but who knows... The =tracks= function works exactly like the =chans= function, except that it operates on the =:track= entry of events.

     (play
      (patch :flute)
      (tracks (tup> c0 c5 c5 c5- c2- c7-)
              (tup> c0 c2- c5 c5))
      (dup 4))

     ;; By default we are on track 0. So the second argument of tracks goes on track 1. Like with channels we can be more precise by using the =track= function.

     (track 1)

     (track 12)

     (track (fn [x] (+ x 3)))

    ]]

   ["Mapping"

    ;; All the transformations we've seen so far are acting on a score to produce another score, but sometimes what we need is to apply a transformation on each event of a score, for this we are using the =each= function.

    ;; As an illustration, here those two fragments:

    (play (lin c0 c1 c2 c3)
          (tup c0 o1)) ; each member of this `tup` form receives and operate on the whole score

    (play (lin c0 c1 c2 c3)
          (each (tup c0 o1))) ; each event of the score is transformed using this `tup` transformation.

    ;; One important thing to be aware of is that events will be mapped in place, so if the given transformation expand the score, some superposition will occur.

    (play (lin c0 o1)
          (each [dur:4 (rep 8 c1-)]))

    ;; Some others functions exists to transform only subparts of the score, if interested you can look at =$by= and/or =parts=.

   ]

   ["Dynamism"

    ;; For now our scores are pretty static, and don't use the power of clojure much. Since this library is built out of simple functions it should be a easy to do so.

    ;; There is a bunch of things to know in order to ease things.

    ["Star functions"

     ;; Variadic functions have a 'star' counterpart that accepts a sequence instead of variadic args.

     (tup c1 c2 c3)

     ;; Is similar to:

     (tup* [c1 c2 c3])

     ;; or

     (tup* (list c1 c2 c3))

     ;; It ease things a bit when using clojure to generate arguments of those functions. Avoiding to write =apply= everywhere.

    ]

    ["Map functions"

     ;; maps can be used to compose event transformations

     (play {:velocity (fn [x] (/ x 2)), :duration (fn [x] (* x 2))})

    ]

    ["Examples"

     (play (tup* (shuffle [c0 c3 c7 c9])))

     (play
      (patch :electric-piano-1)
      (tup* (map (fn [v] {:velocity v}) (range 0 127 15))))

    ]]

   ["Non determinism"

    ;; It is quite fun to insert a bit of randomness in our scores.

    (play
     (rand-nth [(tup c0 c4 c7) (tup c0 c3 c7)])
     (rep 4 (rand-nth [c3 c4 c3- c4-])))

    ;; We can use some great available tools like =test.check.generators= to handle non determinism. That being said, some commonly used non-deterministic functions are available directly.

    ["=one-of="

     ;; =one-of= picks randomly one of the given transformations and apply it.

     (play (one-of o1- o1))

     (play dur:8 (rep 50 (one-of c1 c1-)))

    ]

    ["=maybe="

     ;; =maybe= is very similar to =one-of= except it has a chance to do nothing (identity transformation).

     (play (maybe o1 o2)) ; may do nothing, or one octave up, or two octave up

     (play (one-of same o1 o2)) ; the equivalent `one-of` form

     (play dur:8 (rep 50 (maybe c1 c1-))) ; you can notice melodic repetitions unlike with the corresponding one-of example.

    ]

    ["=probs="

     ;; =probs= gives you more control over the probability of occurence of the given transformations.

     (play (probs {o1 4, o1- 1})) ; 4/5 to go up one octave, 1/5 chance to go down one octave

     (play dur:4 (rep 24 (probs {c1 6, c6- 1, (par c0 o1-) 1})))

    ]

    ["=any-that="

     ;; =any-that= is similar to =one-of= except it takes an extra first argument that check if the picked transformation is valid.

     ;; A melody of 60 notes using the 6 given intervals but remaining in the given pitch bounds:

     (play dur:8
           (rep 60
                (any-that (within-pitch-bounds? :C-1 :C1)
                          c2 c5 c7 c2- c5- c7-)))

     ;; The =within-pitch-bounds?= is just a score transformation that return the score unchanged if it is within the given bounds, else it returns =nil=. Any function of this kind can be used has first argument to =any-that=.

    ]

    ["=!="

     ;; the =!= macro can be useful to deal with raw non deterministic expressions. here the docstring:

     ;; #+begin_quote
     ;; Takes a non deterministic expression resulting in a score transformation. return a score transformation that wraps the expression so that it is evaluated each time the transformation is used.
     ;; #+end_quote

     (play (nlin 4 (! (tup* (shuffle [d0 d2 d4 d6])))))

     (play (nlin 4 (tup* (shuffle [d0 d2 d4 d6])))) ; without the bang the shuffle expression is executed only one time.

    ]

    ["Shuffling"

     ;; As in the previews example, building a =tup= or a =lin= with shuffled sequence of transformation is quite fun.

     ;; So two shortcuts are defined:

     (play (shuftup d0 d2 d4 d6))

     (play (shuflin d0 d2 d4 d6))

    ]]

   ["Harmony"

    ;; It is time to enter more deeply into the harmonic system. In this part we will see how to deal with scales, modes, chords, modulations and more...

    ["Intervals 2"

     ;; So far we've seen 3 types of intervals, chromatic steps, diatonic steps and octaves (aka tonic shifts). Let see the two remaining kinds of steps.

     ["Steps"

      ["Structural"

       ;; Most of the time, our music is based on chords.

       ;; Structural steps are targeting chord notes. By default the harmony is set to C Major scale, and C Major chord (C major triad).

       (play (s-step 1)) ; ascending third

       (play (s-step 2)) ; ascending fifth

       ;; As other steps, corresponding vars are defined:

       (play s1)

       (play s2)

       (play s1-)

       ["Examples"

        ["Arpegios"

         (play (tup s0 s1 s2 s3))

         (play (rup 6 s1))

         (play (rep 4 s1-) (each (tup> s2 s2 s2 s1- s2- s1-)))

        ]

        ["Passing tones"

         (play (scale :eolian) dur:2 o2 (rep 12 s1-) (each (tup s0 c1- d1 s0)))

        ]]]

      ["Tonic"

       ;; The last kind of step is the tonic one.

       ;; It let you jump to the root of the tonality.

       (play (t-step 1)) ; upper tonic

       (play (t-step -1)) ; above tonic

       ;; As other steps corresponding vars are defined:

       (play t1)

       (play t2)

       (play t1-)

       ["Examples"

        (play (rup 4 t1))

        (play (rep 3 t1) (each (tup> s0 s1 s1 d1-)))

       ]]]]

    ["Implementation"

     ;; Those four types of steps can be seen as belonging to 4 successive layers build on each others.

     ;; 1. chromatic: =[0 1 2 3 4 5 6 7 8 9 10 11]=
     ;;   the chromatic layer, 12 successive semitones
     ;; 2. diatonic: =[0 2 4 5 7 9 11]=
     ;;   we select indexes from the above layer (chromatic) to form the diatonic layer (here the major scale)
     ;; 3. structural: =[0 2 4]=
     ;;   same here but based on the diatonic layer to form the structural layer (here the basic triad)
     ;; 4. tonic: =[0]=
     ;;   the root

     ;; As you see, the chromatic layers and tonic layers are trivial, so they are omitted in the harmonic context representation.

     ;; The harmonic context can be found under the :pitch key of any event.

     (=
      (:pitch noon.events/DEFAULT_EVENT)
      {:scale [0 2 4 5 7 9 11],
       :structure [0 2 4],
       :origin {:d 35, :c 60},
       :position {:t 0, :s 0, :d 0, :c 0}})

     ;; The :origin key hold the pitch from where our layers starts (in both directions).

     ;; The :position key holds a map with the 4 layers indexes

     ;; - =:t= tonic

     ;; - =:s= structural

     ;; - =:d= diatonic

     ;; - =:c= chromatic

    ]

    ["Shifts"

     ;; At least we will understand the nuance between steps and shifts. To do so let's compare tonic steps and tonic shifts (aka octaves).

     ;; At first glance they seems to be equivalent:

     (play (t-shift 1))

     (play (t-step 1))

     ;; In this case they are indeed equivalent, in each case a C1 is played. But how about this ?

     (play s1 (t-shift 1)) ; plays a E1

     (play s1 (t-step 1)) ; plays a C1

     ;; In the first expression (the shift) we have transposed the score (a E0 note) by 1 tonic layer index. In the second one (the step) we have stepped to the next tonic layer index.

     ;; In practice, apart for octaves, shifts are not used so often, thats the reason why they don't have defined vars as steps have. They are mainly used in more complex harmonic operations (voice leading etc...).

    ]

    ["Tonality"

     ["=scale="

      ;; By default the major scale is used, but it can be changed. Most of the known scales and modes are available via the =scale= function or directly by name.

      noon.constants/modes ; modes full list

      (play (scale :dorian) dur:4 (rep 8 d1)) ; dorian scale

      (score harmonic-minor) ; sets scale to harmonic-minor

     ]

     ["=structure="

      ;; By default we use the triad structure (tonic, third, fifth), but it can be changed. Some common structures are predefined and available by name.

      noon.constants/structures ; full structure list

      (score (structure :tetrad)) ; sets structure to tetrad

      (score sus47) ; set-structure-to-sus47

     ]

     ["=origin="

      ;; The last thing we need to setup an harmonic context is an origin pitch.

      ;; By default the origin is setup to middle C.

      ;; We can use the =origin= function to change this

      (score (origin :Eb0))

      ["Examples"

       (play (lin (origin :C0) (origin :E0) (origin :G#0)) (each (rup 6 s1)))

      ]]

     ["=root="

      ;; The root update works a bit like =origin= but it takes a pitch-class instead of a pitch. It moves the :origin of the harmonic context to the closest pitch matching the given pitch class.

      ;; For instance if the origin is on =C0=, =(root :B)= will put the origin on =B-1= because =B-1= is closer to =C0= than =B0=.

      (score (root :D))

      (score (root :B))

      ["Examples"

       (play
        (lin* (map root [:C :E :G#]))
        (each (chans (par d0 d3 d6 d9) [(rup 4 d3) (rup 3 d2)]))
        (rep 4 s1))

      ]]

     ["=transpose="

      ;; The transpose update takes an interval or a position and use it to update the origin of the harmonic context

      (play (scale :lydianb7) (rup 6 d2) (rep 4 (transpose c3-)))

     ]

     ["=rebase="

      ;; Sometimes when changing the harmonic context, we want to stay on the same pitch, the =rebase= function let you do that.

      (score (rebase (root :E)))

      ;; Here we are modulating to E major, but we are staying on the pitch we were on (=C0=).

      (=
       (get-in (score (rebase (root :E))) [:pitch :position])
       {:t 0, :s -1, :d 0, :c 1})

      ;; This position points to =C0= but in the context of E major.

      ;; The =rebase= function can take several harmonic context transformations.

      (score (rebase (root :E) (scale :mixolydianb6)))

     ]

     ["=degree="

      ;; Move to the nth degree of the current scale (mode), negative indexes are allowed.

      (score (degree 2)) ; move to the 3rd degree of C major, E phrygian

      (score (scale :melodic-minor) (degree -1)) ; move to the 7th degree of C melodic minor, B superlocrian.

      ;; Roman numeral vars are also available to change degree.

      (play (patch :trumpet) (lin I IV V I) (each (tup s0 s1 s2)))

     ]

     ["parser"

      ;; It is possible to express harmonic transformations via keyword notation.
      ;; Chord color, key, mode and structure can be combined as a keyword for more concision and expressiveness.

      ;; This feature is still very much a work in progress but is already usable.

      ;; Here is a list of components that can be composed together to express complex harmonic context:

      ["root"

       ;; Pitch classes can be used to specify the root of the tonality.
       ;; They can be followed or not with extra components like chord structure or mode.


       (h/upd :A) ; sets the root to A


       (h/upd :C#) ; set the root to C#

       (h/upd :Am) ; set the tonality to A minor (more on chord colors later)

       (h/upd :Dmixolydian) ; set the tonality to D mixolydian (more on modes later)

      ]

      ["degree"

       ;; Roman degree notation can be used to denote relative motion to another harmonic context.


       (h/upd :II) ; go to the second degree of the current context


       (h/upd :IIm7) ; same with with additional chord color minor seventh.


       (h/upd :bIIIionian) ; go to bIII degree setting the mode to ionian.


      ]

      ["triads"

       ;; Common triad notation can be used:

       ;; - =m= (minor triad)
       ;; - =M= (major triad)
       ;; - =o= (diminished triad)
       ;; - =+= (augmented triad)

       ;; An example using the four 4 triad colors.

       (play (lin (h/upd :CM)
                  (h/upd :C#o)
                  (h/upd :Dm)
                  (h/upd :Eb+))
             (each (par s0 s1 s2)))

       ;; With =noon.lib.harmony/lin= (a variant of =lin= that understand this notation), it can be expressed in a more concise manner.

       (play (h/lin :CM :C#o :Dm :Eb+)
             (each (par s0 s1 s2)))

      ]

      ["tetrad"

       ;; Common tetrad notation:

       ;; - major-seventh: =Δ= or =M7=
       ;; - diminished-seventh: =o7=
       ;; - minor-seventh: =m7=
       ;; - dominant: =7=
       ;; - half-diminished: =ø= or =m7b5=
       ;; - minor-major-seventh: =mΔ= or =mM7=

       (play (h/lin :IM7 :#Io7 :IIm7 :#IIo7 :III7 :VI7 :IIm7b5 :V7)
             (dup 2)
             (each (par s0 s1 s2 s3)))

      ]

      ["modifiers"

       ;; Modes or structures can be altered with degree modifiers:

       (play (h/lin :G7b9 :CM79)
             (par s0 s1 s2 s3 s4))


       (play (h/lin :G7sus4b9 :CM7#11)
             (par s0 s1 s2 s3 s4))

      ]

      ["structure shorthand"

       ;; This a non standard way to write structures can be handy for uncommon cases.

       ;; The =s= character is followed by a suite of scale-idxs representing the structure.

       (play (h/upd :s1234) ; tonic + second +third + fourth
             (par s0 s1 s2 s3))


       (play (h/upd :s3467) ; third + fourth + sixth + seventh
             (par s0 s1 s2 s3))


      ]

      ["modes"

       ;; Common modes names are supported:

       ;; - dorian: =dorian= or =dor=
       ;; - phrygian: =phrygian= or =phry=
       ;; - lydian: =lydian= or =lyd=
       ;; - mixolydian: =mixolydian= or =mix=
       ;; - aeolian: =aeolian= or =eolian= or =eol=
       ;; - locrian: =locrian= or =loc=
       ;; - harmonic-minor: =harmonic-minor= or =harmm=
       ;; - melodic-minor: =melodic-minor= or =melm=
       ;; - altered: =superlocrian= or =altered= or =alt=
       ;; - harmonic-major: =harmonic-major= or =harmM=
       ;; - double-harmonic: =double-harmonic=
       ;; - ultraphrygian: =ultraphrygian=
       ;; - hungarian-minor: =hungarian-minor= or =hungarian=
       ;; - oriental: =oriental=
       ;; - ultralocrian: =ultralocrian=


       ;; Basics examples:

       (play (h/upd :ionian)
             (rup 7 d1))
       (play (h/upd :dorian)
             (rup 7 d1))

       ;; With extra modifiers:

       (play (h/upd :lydian+)
             (rup 7 d1))
       (play (h/upd :lydianb7)
             (rup 7 d1))


      ]

      ["=upd="
       ;; As seen in above examples, the =noon.harmony/upd= function turns a keyword describing an harmonic-context into a usable score update.
      ]

      ["=lin="
       ;; The =noon.harmony/lin= function is the same as the =noon.updates/lin= function, excepts it can understand ahrmonic keywords.
      ]

      ["=tup="
       ;; The =noon.harmony/tup= function is the same as the =noon.updates/tup= function, excepts it can understand ahrmonic keywords.
      ]

      ["implementation"
       ;; The instaparse library is used to parse harmonic keywords, the grammar is located at =noon.parse.harmony-grammar/grammar=


      ]]]]]

  ["Composing"

   ;; When composing music, 4 major aspects are considered: melody, rythmn, harmony and tone. In this section some tools to deal with those aspects will be introduced.

   (require
    '[noon.lib.harmony :as h]
    '[noon.lib.melody :as m]
    '[noon.lib.rythmn :as r]
    '[noon.utils.sequences :as seqs])

   ["Melody"

    ;; Let see some ways to deal with melodies.

    ["Bounding"

     ;; One of the most common things we want to be able to control when generating melodies is the range.

     ["within-pitch-bounds?"

      ;; This function returns nil if any event of the score is not in the given pitch bounds.

      (= (score Eb0 (within-pitch-bounds? :C-1 :C0)) nil)

      (= (score Eb0 (within-pitch-bounds? :C0 :C1)) (score Eb0))

      ;; This function is handy in conjuction with the =any-that= or =fst-that= forms.

      (play
       (patch :electric-piano-1)
       dur:8
       (rep 60 (any-that (within-pitch-bounds? :C0 :C1) c1 c1- c5 c5-)))

      ;; The =fst-that= form takes a test and any number of update that will be tried in order until one pass the test.

      (play
       dur:8
       (rep
        60
        (fst-that (within-pitch-bounds? :C0 :C1) (one-of c5 c5-) c2 c2-)))

      ;; Random melodies are nice at first but can quickly become boring. It is often more pleasing to develop one or more ideas gradually via simple transformations.

     ]]

    ["Rotations"

     ;; Rotating a melody is a way to evolve it while preserving its identity.

     ["Example"

      (play (fit (rep 8 d1)) (m/rotation 3))

     ]

     ["Forms"

      ;; The =noon.lib.melody/rotation= accepts several types of argument:

      (m/rotation 2) ; rotate two notes forward

      (m/rotation -3) ; rotate three notes backward

      (m/rotation 1/2) ; rotate half the size forward

      (m/rotation -1/3) ; rotate third the size backward

      (m/rotation :rand) ; random rotation

      (m/rotation [0 1/2]) ; random rotation between first and half the size

      ;; This kind of argument (that I will call a 'member-pick') will be used at many other places within this section, it came from the =noon.utils.sequences/member= function, here the docstring:

      ;; Find or pick an element within a sequence 's.
      ;; available forms:
      ;; - =(member s <integer>)= normal nth like get
      ;; - =(member s <negative-integer>)= nth from the end of the list
      ;; - =(member s <float-or-rational>)= a non integer between -1 and 1, is picking a member relatively to the length of the list, forward if positive, backward if negative.
      ;; - =(member s <[min max]>)= picks a member randomly between the given idxs (every type of index allowed)
      ;; - =(member s <:rand|:random>)= picks a random member

     ]

     ["Chords"

      ;; Not only pure melodies can be rotated, if we feed chords into the =rotation= transformation it behaves as intended.

      (play (fit (rep 8 d1)) (each (par d0 d3 d6)) (m/rotation 1/4))

     ]]

    ["Permutations"

     ;; Another way to transform a melody while preserving a bit of its identity is to permute it. But for long melody, a random permutation can make it so distant to the original that it miss the point. For this reason, permutations are ordered and requested by complexity (similarity degree with the original)

     ["Forms"

      ;; Like the rotation function, the =permutation= function uses a 'member-pick argument.

      (m/permutation 2) ; the second most similar permutation

      (m/permutation -1) ; the less similar permutation

      (m/permutation 1/2) ; half way between most similar and most different

      (m/permutation -1/4) ; one quite distant permutation

      (m/permutation :rand) ; random permutation

      (m/permutation [1/4 -1/4]) ; a not too much similar nor different permutation

     ]

     ["Example"

      (let
       [space [vel0 dur:8]]
       (play
        (patch :electric-piano-1)
        (tup d0 d2 d1 d3 d2 d4 d3 d5)
        (lin
         same
         space
         (m/permutation 1)
         space
         (m/permutation 2)
         space
         (m/permutation -1/4))))

     ]

     ["Options"

      ["Grade"

       ;; The permutations are categorised by grade, the grade of a permutation correspond to the number of splits that has to be made on the original seq to obtain it. For instance a grade 1 permutation is one that we can obtain by splitting our original sequence in 2 parts.

       (require '[noon.utils.sequences :as seqs])

       (=
        (seqs/grade-permutations [0 1 2 3] 1)
        '((2 3 0 1) (1 2 3 0) (3 0 1 2)))

       ;; This way to categorise permutations can be helpful to have more control over the similarity of the resulting permutation. In addition to this the returned permutations for a given grade are ordered starting from the more balanced splits. As you can see in the previous example, (2 3 0 1) is the first permutation of grade 1, and contains 2 splits of size 2: (2 3) and (0 1).

       ;; We can leverage those grades via our =m/permutation= function like this:

       (m/permutation 0 {:grade 1}) ; get the first grade 1 permutation.

       (m/permutation -1 {:grade [1 3]}) ; get the last permutation for a randomly picked grade between 1 and 3.

      ]

      ["Layers"

       ;; As we've seen, our melodies are built on different harmonic layers (chromatic, diatonic, structural and tonic), the =m/permutation= function is letting you act on or inside a particular layer.

       ;; As an example for this, please consider this kind of melody.

       (play dur2 (tup s0 s1 s2 s3) (each (tup d1 d1- d0)))

       ;; We start with an ascension on the structural layer, then adding some diatonic ornementation on each structural degree. Those diatonic notes have meaning relatively to the structural degrees they are based upon. If we do a raw permutation on this melodic line we are losing those relations. With the :layer option we can permute only the structural layer keeping those diatonic ornementations untouched.

       (play
        dur2
        (tup s0 s1 s2 s3)
        (each (tup d1 d1- d0))
        (m/permutation 1 {:layer :s}))

      ]

      ["Split sizes"

       ;; TODO

      ]]]

    ["Mixed example"

     ;; In the following example you can get a sense of the effect of deriving a melody from simple transformations.

     (play
      {:description
       "rand harmonic seq using IV II and VI degrees on vibraphone,
        ocarina melody derives using transposition, rotation and permutation."}
      (chans
       [(patch :vibraphone)
        vel3
        (ntup 4 [(one-of IV II VI) tetrad (par [t2- vel5] s0 s1 s2 s3)])]
       [(patch :ocarina)
        vel5
        (shuftup d1 d2 d3 d4 d5)
        (each (maybe (par d0 d3)))
        (rup
         16
         (probs
          {(m/permutation :rand) 1,
           (m/rotation :rand) 3,
           (one-of* (map d-step (range -3 4))) 5}))])
      (adjust 10)
      (append [d2- (transpose c3)] [d2 (transpose c3-)] same))

    ]

    ["Contour"

     ;; The idea of contour is quite simple. When you see a melody on a score or a pianoroll, by linking the successive notes you can make a line. This line has a certain shape, some melodies with different notes share the same shape (contour). The contour of a melody greatly participate to its identification by the listener. So by keeping a contour and changing the notes, we can ensure a kind of continuity in our melodic developments.

     ;; For instance those different melodies are all sharing the same contour: [0 2 1 2]

     (play (tup s0 s2 s1 s2))

     (play (tup s0 s3 s2 s3))

     (play (tup d0 d2 d1 d2))

     (play (tup d1 d5 d2 d5))

     (play (tup s2 s4 d8 s4))

     ;; You can clearly hear the similarity between those

     ["contour"

      ;; :docstring

      ;; #+begin_quote
      ;; changing the melodic contour of a score.

      ;;         forms:
      ;;         (contour :mirror <options>) : mirror the contour of the score.
      ;;         (contour :rotation <options>) : rotate the contour of the score.
      ;;         (contour :similar <options>) : get a different score with the same contour.

      ;;         <options>
      ;;         a map that may contain some of those keys:

      ;;         :layer : (all commands, default to score's lowest harmonic layer)
      ;;             The harmonic layer on which the contour transformation is performed

      ;;         :pick | :nth : (:rotation and :similar commands, default to :random)
      ;;             A 'member-pick (see `member function) to select one particular outcome.

      ;;         :extent : (:similar command only)
      ;;             A vector of min and max amount of deformation that we want to apply to the score.

      ;;         :delta : (:similar command only)
      ;;             The amount of shrinking or growing we want to apply to the score.
      ;; #+end_quote

     ]

     ["Demo"

      ;; Let's take this simple arpegio to start

      (play (tup s0 s1 s2 s3 s1 s2)) ; {:contour [0 1 2 3 1 2]}

      ;; Here the way to obtain the mirror contour of the previous arpegio.

      (play (tup s0 s1 s2 s3 s1 s2) (m/contour :mirror)) ; {:contour [3 2 1 0 3 2]}

      ;; Next let's try contour rotations:

      ;; Here we are picking the first rotation (with the option =:nth=)

      (play (tup s0 s1 s2 s3 s1 s2) (m/contour :rotation {:nth 1})) ; {:contour [1 2 3 0 2 3]}

      ;; Every contour index has been shifted one step up, the highest one returning all the way down.

      ;; Lets get the last rotation using a 'member-pick argument.

      (play (tup s0 s1 s2 s3 s1 s2) (m/contour :rotation {:pick -1})) ; {:contour [3 0 1 2 0 1]}

      ;; If no :pick or :nth option is given, select a random one.

      (play (tup s0 s1 s2 s3 s1 s2) (m/contour :rotation))

      ;; One of the nice things with contours is that it can serve to generate many melodies. Using the =:similar= commands we can do this.

      ;; Here we are randomly picking a similar score that is one structural step wider (:delta 1) that the original one.

      (play (tup s0 s1 s2 s3 s1 s2) (m/contour :similar {:delta 1}))

      ;; In all the previous exemples, the contour was computed over the structural layer. When the layer is not specified, the score's lowest harmonic layer is used, here the structural layer.

      ;; As an illustration let's look at the effect of specifying the layer within the :mirror contour operation:

      (play (tup s0 s1 s2 s3 s1 s2) (m/contour :mirror)) ; Original example

      (play (tup s0 s1 s2 s3 s1 s2) (m/contour :mirror {:layer :d})) ; Mirrored diatonically, resulting in a F major arpegio

      (play (tup s0 s1 s2 s3 s1 s2) (m/contour :mirror {:layer :c})) ; Mirror chromatically, resulting in a F minor arpegio (it can help with 'negative harmony')

      ;; One of the similar scores between those shrinked by 2 diatonic step and those expanded by 3 diatonic steps (:extent [-2 3] :layer :d).

      (play
       (tup s0 s1 s2 s3 s1 s2)
       (m/contour :similar {:extent [-2 3], :layer :d}))

     ]]

    ["Line"

     ;; One simple way to build a melody is to concatenate some little fragments one after another, building the next fragment on the last note of the previous one.

     ;; There are several ways to do this:

     (play
      {:description
       "building a melodic line of 32 notes by chaining fragments of differerent lengths."}
      (patch :ocarina)
      dur:4
      (m/simple-line
       32
       (one-of
        (nlin> 4 (one-of d1- d1))
        (tup d1 d1- s0)
        (lin s2 s1 s1-)
        (nlin> 4 (one-of s1- s1)))))

     ;; The =simple-line= function is built on top of the more general function =noon.lib.melody/line=

     (play
      {:description
       "another way to build a melodic line from a bunch of randomly chosen transformations."}
      (patch :acoustic-guitar-nylon)
      (repeat-while
       (within-time-bounds? 0 24)
       (append
        [start-from-last
         (any-that
          (within-pitch-bounds? :C-1 :C2)
          (rep 3 d3)
          (rep 3 d3-)
          d1
          d1-)]))
      (adjust 3))

    ]]

   ["Rythmn"

    ;; So far we havn't discuss rythmn so much, let see what we have at our disposal to deal with it.

    ["Simple"

     ;; As we've seen earlier, we can use the =duration= related transformations to write simple rythmns

     (play
      (patch :woodblock)
      dur:4
      (lin same dur:2 dur:2 same dur2 same same))

     ;; This is not a pretty way to write it ! We can use the =_= shortcut instead of =same=, and the =tup= function for making this a bit more readable.

     (play (patch :woodblock) dur:4 (lin _ (tup _ _) _ dur2 _ _))

     ;; We can also use the =dupt= function if we prefer.

     (play (patch :woodblock) dur:4 (lin _ (dupt 2) _ dur2 _ _))

     ;; We could have done it like so too:

     (play (patch :woodblock) dur2 (tup* (map dur [1 1/2 1/2 1 2 1 1])))

     ;; There is a function to help writing a rythmn this way:

     (play dur2 (r/durtup 1 1/2 1/2 1 2 1 1))

     (play dur2 (r/durtup* [1 1/2 1/2 1 2 1 1]))

     ;; Writing those kind of rythmns is not the funniest thing to do of course, let see how we can generate and transform rythmns.

    ]

    ["Generation"

     ;; The main tool we have at our disposal to create a rythmn is the noon.lib.melody/gen-tup

     ["gen-tup"

      ;; form:
      ;;        (gen-tup resolution size & options)

      ;;        Generates a rythmic tup based on the given arguments:
      ;;        resolution: the number of subdivisions that we will use.
      ;;        size: the number of notes that the generated tup will contain.
      ;;        options:
      ;;          euclidean: generates an euclydean tup.
      ;;          durations: the multiples of =resolution= that we are allowed to use (fractionals allowed).
      ;;          shifted: the possibility for the generated tup to not begin on beat.
      ;;          

     ]

     ["Examples"

      ;; randomly dispose 5 notes into 8 subdivisions.

      (play (patch :woodblock) (r/gen-tup 8 5) (dup 4))

      ;; Lets add a metronome

      (play
       (chans
        [(patch :tinkle-bell) o1-]
        [(patch :woodblock) (r/gen-tup 8 5)])
       (dup 4))

      ;; A bit slower

      (play
       dur2
       (chans
        [(patch :tinkle-bell) (tup o1- o1)]
        [(patch :woodblock) (r/gen-tup 16 8)])
       (dup 4))

      ;; Let's try 12/8

      (play
       dur2
       (chans
        [(patch :tinkle-bell) (tup o1- o1)]
        [(patch :woodblock) (r/gen-tup 12 6) (each (maybe o1 o1-))])
       (dup 4))

      ;; Using the =:shifted= keyword you can give your tup a chance to not start on beat.

      (play
       dur2
       (chans
        [(patch :tinkle-bell) (tup o1- o1)]
        [(patch :woodblock) (r/gen-tup 16 7 :shifted) (each (maybe o1 o1-))])
       (dup 4))

      ;; You can specifies which durations are allowed with the =:durations= option

      ;; here we are generating a tuple of resolution 12 and size 5, using only 2/12 and 3/12 durations.

      (play
       dur2
       (chans
        [(patch :tinkle-bell) (tup o1- o1)]
        [(patch :woodblock) (r/gen-tup 12 5 :durations [2 3])])
       (dup 4))

      ;; A 3 voices example:

      (play
       (patch :tinkle-bell)
       dur2
       (par
        [o1- (dupt 2)]
        (r/gen-tup 12 5 :shifted :durations [1 2 3])
        [o1 (r/gen-tup 12 7 :shifted :durations [2 1 3])])
       (dup 4))

      ;; The =:euclidean= flag let you generate euclidean rythmns: https://blog.landr.com/euclidean-rhythms/

      (play
       {:description "~trésillo"}
       (chans
        (patch :tinkle-bell)
        [(patch :woodblock) (r/gen-tup 8 3 :euclidean)])
       (dup 4))

      (play
       {:description "~bembé"}
       dur2
       (chans
        [(patch :tinkle-bell) (tup o1- _)]
        [(patch :woodblock) (r/gen-tup 12 7 :euclidean)])
       (dup 4))

      (play
       {:description "~bossa"}
       dur2
       (chans
        [(patch :tinkle-bell) (tup o1- _)]
        [(patch :woodblock) (r/gen-tup 16 5 :euclidean)])
       (dup 4))

      ;; 2 more examples:

      (let
       [rtup (! (r/gen-tup 16 5 :euclidean :shifted))]
       (play
        (patch :tinkle-bell)
        (chans (ntup 2 o1-) rtup [o1 rtup] [o2 rtup] [o3 rtup])
        (dup 4)
        (adjust {:duration 8})))

      ;; Fancy variation:

      (let
       [rtup
        (!
         [(r/gen-tup 16 5 :euclidean :shifted)
          (each [(maybe o1 o2) (one-of vel4 vel6 vel8)])])]
       (play
        mixolydian
        (patch :vibraphone)
        (lin same (transpose c4-))
        (h/align-contexts)
        (each
         (chans
          [(patch :tinkle-bell) o1-]
          [(patch :acoustic-bass) t1- (tup same s1-)]
          rtup
          [d4 rtup]
          [d6 rtup]
          [d10 rtup]))
        (dup 8)
        (adjust {:duration 32})))

     ]]

    ["Transformation"

     ;; Once we have written or generated a rythmn we may want to make it evolve, here is some functions that can help.

     ["noon.lib.melody"

      ;; We can use the previously seen functions from =noon.lib.melody= to permute or rotate a rythmn.

      (play
       dur2
       (chans
        [(patch :tinkle-bell) o1- (tup same [vel5 o1]) (dup 8)]
        [(patch :woodblock)
         (r/gen-tup 12 5 :euclidean)
         (rep 8 (probs {(m/permutation :rand) 1, (m/rotation :rand) 3}))]))

     ]

     ["r/rotation"

      ;; Unlike =noon.lib.melody/rotation= this function do not operates on a note basis

      ["Example"

       ;; Rotating a score by the given duration

       (play
        (chans
         [(patch :tinkle-bell) o1- (dup 4)]
         [(patch :woodblock)
          (r/durtup 2 1 1 4)
          (lin _ (r/rotation 1/2) (r/rotation 1/4) (r/rotation -1/4))]))

       ;; You can rotate by any duration, even if it do not really make sense.

       (play
        (chans
         [(patch :tinkle-bell) o1-]
         [(patch :woodblock) (r/durtup 2 1 1 4) (r/rotation -1/5)])
        (dup 4))

       ;; You can also rotate relatively to score duration. Here we are starting with a score of duration 2. With the form (r/rotation :relative -1/4) we are rotating it a quarter of its duration backward.

       (play
        dur2
        (chans
         [(patch :tinkle-bell) o1-]
         [(patch :woodblock) (r/durtup 2 1 1 4) (r/rotation :relative -1/4)])
        (dup 4))

       ;; There is also forms to randomly pick a rotation (rotation :rand-by <increment>) : pick a random rotation using increment as resolution. (rotation :rand-sub <n>) : split the score in 'n parts and rotate to a randomly picked one.

       (play
        dur2
        (chans
         [(patch :tinkle-bell) o1-]
         [(patch :woodblock) (r/durtup 2 1 1 4) (r/rotation :rand-by 1/2)])
        (dup 4))

       (play
        dur2
        (chans
         [(patch :tinkle-bell) o1-]
         [(patch :woodblock) (r/durtup 2 1 1 4) (r/rotation :rand-sub 4)])
        (dup 4))

      ]]

     ["r/permutation"

      ;; Like =noon.lib.rythmn/rotation=, =noon.lib.rythmn/permutation= do not operate on a note basis like =noon.lib.melody/permutation=. It operates on even time splits

      ["Example"

       ;; Let's start with this tup:

       (play (patch :woodblock) (r/durtup 2 1 1 4) (dup 4))

       ;; Here we are picking a random permutation of our score splitted in 4 equal parts.

       (play
        (chans
         [(patch :tinkle-bell) o1-]
         [(patch :woodblock) (r/durtup 2 1 1 4) (r/permutation 4)])
        (dup 4))

       ;; Like we've seen with =noon.lib.melody/permutation=, there is several way to choose a particular permutation. With the second argument we can specify how to pick one.

       (r/permutation 4 1) ; picking the most similar base 4 permutation

       (r/permutation 4 -1) ; picking the least similar base 4 permutation

       (r/permutation 8 [0 1/2]) ; picking one of the most similar base 8 permutation

       (r/permutation 8 :rand) ; picking a random base 8 permutation

       ;; fun:

       (play
        {:description "rythmic permutation demo"}
        (chans
         [(patch :taiko-drum) vel5 (dup 4)]
         [(patch :woodblock)
          (r/durtup 2 1 1 1/2 1/2)
          (each (maybe o1 o1-))
          (nlin 4 (r/permutation 5))]
         [(patch :electric-piano-1)
          o1-
          vel4
          lydian
          (par> d0 d3 d3 d3 d3)
          (lin (root :C) (root :Eb) (root :Ab) (root :Db))])
        (dup 4))

      ]]]]

   ["Harmony"

    ;; Within the lib.harmony module you will find some tools to deal with chords.

    ["Voicings"

     ;; In musical terms, a voicing is a particular agencement of a chord. When we speak of a chord like for instance G7, we are not specifying the precise way we will dispose its components.

     ;; It can be played in closed position

     (play (patch :electric-piano-1) V tetrad (par s0 s1 s2 s3))

     ;; Inverted (first inversion)

     (play (patch :electric-piano-1) V tetrad (par [o1 s0] s1 s2 s3))

     ;; Or dropped (drop 2)

     (play (patch :electric-piano-1) V tetrad (par s1 [o1 s2] s3 s4))

     ;; and many other ways...

     ["Inversions"

      ;; upward inversions

      (play (patch :vibraphone) (par s0 s1 s2) (rep 4 (h/inversion 1)))

      ;; downward double inversions

      (play (patch :vibraphone) o1 (par s0 s1 s2) (rep 4 (h/inversion -2)))

      ;; In those particular exemples we could have done the same using s1 and s2-, here the equivalent of the first example:

      (play (patch :vibraphone) (par s0 s1 s2) (rep 4 s1))

      ;; But it is not always the case with more complex chords

      (play
       {:description "4 successive double inversions upward on a Cmaj79 "}
       (patch :vibraphone)
       o1-
       (par d0 d2 d4 d6 d8)
       (rep 4 (h/inversion 2)))

     ]

     ["Drops"

      ;; A drop is voicing where some notes have been sent into upper octaves.

      ;; Here some common drops:

      (let
       [closed
        (par s0 s1 s2 s3)
        drop2
        (par s0 [o1 s1] s2 s3)
        drop3
        (par s0 s1 [o1 s2] s3)
        drop23
        (par s0 [o1 s1] [o1 s2] s3)]
       (play
        (patch :vibraphone)
        tetrad
        (lin closed drop2 drop3 drop23)
        (each dur:2)))

      ["drop"

       ;; This function help you to drop a voicing. It takes the same polymorphic kind of argument (called a 'member-pick') that we've seen with =noon.lib.melody/permutation= and =noon.lib.melody/rotation=.

       ["Examples"

        ;; pick a random drop of Cmaj7

        (play (patch :vibraphone) tetrad (par s0 s1 s2 s3) (h/drop :rand))

        ;; first drop

        (play (patch :vibraphone) tetrad (par s0 s1 s2 s3) (h/drop 1))

        ;; last drop

        (play (patch :vibraphone) tetrad (par s0 s1 s2 s3) (h/drop -1))

        ;; one-of the least wide drop

        (play (patch :vibraphone) tetrad (par s0 s1 s2 s3) (h/drop [0 1/2]))

       ]]]]

    ["Chord progressions"

     ;; A chord progression is simply a succession of different chords, cyclic or not.

     ["Voice leading"

      ;; When dealing with chord progression one of the first thing to consider is called voice leading, it is the way voicings succession is handled.

      ;; Let's start with a very common chord progression.

      (play
       (patch :electric-piano-1)
       (lin I VI IV V)
       (each (par s0 s1 s2))
       (dup 2))

      ;; It do not sound bad but it can arguably be better.

      (play
       (patch :electric-piano-1)
       (lin I VI II V)
       (each [(par s0 s1 s2) (h/drop -1)])
       h/voice-led
       (dup 2))

      ;; The =voice-led= transformation is using inversions and drops in order to minimize voices motion between successive chords.

      ;; It is a really smooth way to transition between voicings but it would be nice to get the original bass motion back.

      (play
       (lin I VI II V)
       (chans
        [(patch :acoustic-bass) C-2 (each t-round)]
        [(patch :electric-piano-1) (each (par s0 s1 s2)) h/voice-led])
       (dup 2))

      ;; It works on any voicings.

      (play
       (structure :tetrad)
       (lin I VI II V)
       (chans
        [(patch :acoustic-bass) C-2 (each [t-round (tup _ s2-)])]
        [(patch :electric-piano-1)
         (each [(par s0 s1 s2 s3) (h/inversion -3) (h/drop 1/2)])
         h/voice-led])
       (dup 2))

      ;; The voice-led function is quite resource consuming and remain to be optimized...

     ]

     ["Melodies"

      ;; Once you have a chord progression, you may want to apply a melody on it.

      ;; One way to do so is to use the =noon.lib.harmony/align-contexts= transformation

      ["align-contexts"

       ;; Let's start with a simple chord progression in minor.

       (play
        (patch :clarinet)
        (scale :harmonic-minor)
        (lin I IV VII I)
        (each (tup s0 s1 s2)))

       ;; the tup is applied on each chord without any inversion.

       ;; With =noon.lib.harmony/align-contexts= we can connect contexts together with minimal offsets, resulting in more conjoint motions.

       (play
        (patch :clarinet)
        (scale :harmonic-minor)
        (lin I IV VII I)
        (h/align-contexts :s)
        (each (tup s0 s1 s2)))

       ;; The word 'context' may seem a bit confusing, what it really stands for is 'harmonic context', the harmonic context can be found under the =:pitch= key of any event.

       ;; A more elaborated example

       (play
        dur2
        (scale :harmonic-minor)
        (lin I IV VII I)
        (h/align-contexts :s)
        (lin same (transpose c3) same)
        (chans
         [(patch :choir-aahs)
          vel4
          (each [(par s0 s1 s2) (maybe (tup s0 s1-) (tup s0 s1))])]
         [(patch :ocarina)
          vel6
          (each
           [(shuftup s0 s1 s2)
            (each
             (one-of
              (tup s0 (shuflin (one-of c1- s-) s+) s0)
              (tup s0 c1- s0 (one-of s2- s2))))])]
         [(patch :acoustic-bass) vel3 o2-]))

      ]

      ["harmonic-zip"

       ;; This transformation helps you to zip a melody on a chord progression. This way you don't have to worry at all about the chords, just write a melody it will be adjusted to chord changes.

       ;; Let's first write a simple melodic pattern.

       (play
        (patch :ocarina)
        (tup s0 s1 [s2 (lin d1 d1- _)] s1)
        (dupt 4)
        (adjust {:duration 4}))

       ;; Now let's use the =h/harmonic-zip= function to apply this to a chord progression.

       (play
         (h/harmonic-zip
          [(scale :harmonic-minor) (tup I IV VII I) (h/align-contexts :s)]
          [(patch :ocarina) (tup s0 s1 [s2 (lin d1 d1- _)] s1) (dupt 4)])
        (dup 2)
        (adjust {:duration 6}))

       ;; Almost the same with comping.

       (play
         (h/harmonic-zip
          [(scale :harmonic-minor) (tup I IV VII I) (h/align-contexts :s)]
          (chans
           [(patch :ocarina)
            (tup s0 s1 [s2 (lin d1 d1- _)] s1)
            (dupt 4)]
           [(patch :acoustic-bass) t2-]
           [(patch :choir-aahs) vel4 (par s0 s2 s4)]))
         (dup 2)
         (adjust {:duration 12}))
      ]]]]]

  ["Experiments"

   ;;Let's use the chorium soundfont:
   (swap! out/options* assoc :tracks {0 :chorium})

   ["Harmonic"

    ;; Some experiences built on top of an harmonic idea.

    ["simple I IV VII I"

     (play (scale :harmonic-minor)
           (lin I IV VII I)
           (h/align-contexts :s)
           (each (tup s0 s1 s2)))

     ;; Experimenting passing notes:

     (play (scale :harmonic-minor)
           (lin I IV VII I)
           (h/align-contexts :s)
           (lin s0 s1 s2-)
           (each [(tup s0 s2)
                  (each (tup s0 c1- s+ s0))])
           (append rev))

     (play dur2
           (scale :harmonic-minor)
           (lin I IV VII I)
           (h/align-contexts :s)

           (lin same (transpose c3) same)

           (chans

            [(patch :choir-aahs) vel4
             (each [(par s0 s1 s2)
                    (maybe (tup s0 s1-) (tup s0 s1))])]

            [(patch :ocarina) vel6
             (each [(shuftup s0 s1 s2)
                    (each (one-of (tup s0 (shuflin (one-of c1- s-) s+) s0)
                                  (tup s0 c1- s0 (one-of s2- s2))))])]

            [(patch :kalimba) vel4 o2
             (each [(shuftup s0 s1 s2)
                    (each (one-of vel0 (par s0 s2-) (shuftup s0 s1 s2)))])]

            [(patch :acoustic-bass) vel3
             o2-]))

    ]

    ["simple I IV I V"

     (play dur2
           ;; grid
           (lin I IV I V)
           (h/align-contexts :s)
           ;; on each chord
           (each (chans
               ;; rythmn
               [(patch :woodblock) C0 (dupt 4)]
               [(patch :tinkle-bell) C0 (r/gen-tup 12 5 {:durations [1 2 3]})]
               ;; comping
               [(patch :marimba) o1- (r/gen-tup 12 5 :euclidean) (each (par s0 s2)) (each (one-of s0 s1 s1-))]
               [(patch :acoustic-bass) t2- vel10 (r/gen-tup 12 5 :euclidean :shifted)]
               ;; ornementation
               [vel12 (patch :music-box) o1
                (one-of s0 s1 s1-)
                (shuftup s0 s1 s3)
                (each (probs {[(par s0 s2) (maybe (tup s0 s1))] 3
                           [(tup s3 s1 (par s2 s0) s1-)] 2
                           [(tup d1- s0 d1 s0) (maybe (m/rotation 2))] 1}))]))
           ;; repeat one time
           (dup 2))

    ]

    ["epic lydian"

     (play {:description "epic lydian sequence by minor thirds"}

           (h/harmonic-zip
            [lydian sus47
             (tup* (map root [:C :Eb :F# :A]))
             (dupt 2)
             (h/align-contexts :s)]

            (par [(chan 1) (patch :choir-aahs) vel3
                  (ntup 8 (par s0 s1 s2))]

                 [vel4
                  (let [s? (one-of s2- s1- s1 s2)]
                    (m/simple-tupline (* 16 16)
                                      (any-that (within-pitch-bounds? :C-1 :C2)
                                                (lin s? s?)
                                                [(shuflin s1 s2 s3 s4) (maybe rev)]
                                                (lin d1 d1- s0 s?)
                                                (lin d1- d1 s0 s?))))

                  (par [(chan 2) (patch :french-horn)]
                       [(chan 3) vel5 o2 (patch :flute)])]

                 [(chan 4) (patch :taiko-drum)
                  vel2 (ntup 16 (lin dur3 [o1 vel4 dur2] dur3))]

                 [(chan 5) (patch :acoustic-bass)
                  o2- (ntup 32 t0)]))

           #_(sub {:channel 5} (each tonic-round))

           (adjust 32)
           (nlin 4 (s-shift -1)))

    ]

    ["Tritonal experiment"

     ;; A rich harmonic sequence using V I progressions over tritonal modulation cycle (like Giant step).

     (play {:description "tritonal chord sequence shifts by minor thirds"}

           (let [I (one-of [lydian+ (structure [2 3 4 5 6])] [melodic-minor (structure [1 2 4 5 6])])
                 V (one-of [V mixolydian (structure [1 3 4 5 6])] [V phrygian6 (structure [0 1 3 5 6])])
                 [B G Eb] (map root [:B :G :Eb])]

             [(tup [B V] [B I] [G V] [G I] [Eb V dur2] [Eb I dur2])
              (rup 4 (transpose d2-))
              (h/align-contexts :s :static)

              (chans

               [(patch :choir-aahs)
                vel3
                (each (par s0 s1 s2 s3 s4))]

               [(patch :vibraphone)
                vel5
                (each (probs {(par s0 s1 s2 s3 s4) 1
                              (shuftup [dur2 (par s0 s2 s4)] [(one-of dur2 dur3) (par s1- s1 s3)]) 3}))]

               [(patch :acoustic-bass)
                vel5
                (each [tetrad o2- t0 (maybe (tup (one-of dur2 dur3) [dur2 o1-]))])]

               [(patch :taiko-drum)
                vel3
                (each (shuftup s0 s1 s2 s3 s4))
                (each (probs {vel0 3 same 1 (one-of o1 o1-) 1 (tup t0 t1) 1}))]

               [vel6
                (h/grid-zipped
                 [(chans (patch :flute) [o1 (patch :piccolo)])
                  (ntup> (* 32 10)
                         (any-that (within-pitch-bounds? :C-2 :C2)
                                   s1 s2 s1- s2- s3 s3-))]
                 (each (probs {vel0 1
                               same 4
                               (superpose (one-of s1 s2 s3)) 0})))])

              (adjust 48)]))

    ]

    ["Elliot smith chords"

     (play dur2
           (lin [VI seventh]
                [IV add2]
                [I]
                [III seventh (inversion 2)]
                [VI seventh]
                [IV add2]
                (tup I [III seventh phrygian3])
                [IV])
           (h/align-contexts :d)
           (each (chans [(patch :acoustic-bass) o1- t-round]
                        h/simple-chord)))


     (play (chans [(patch :electric-piano-1) (tup (shuftup s0 s1 s2 s3) (shuftup s2 s3 s4 s5))]
                  [(patch :acoustic-bass) o1- t-round])
           (dupt 8)
           (h/grid
            [(tup [VI seventh]
                  [IV add2]
                  [I]
                  [III seventh (inversion 2)]
                  [VI seventh]
                  [IV add2]
                  (tup I [III seventh phrygian3])
                  [IV])
             (h/align-contexts :d)])
           (adjust 8)
           (dup 2))

    ]

    ["Minor progression"

     (play (lin [I melodic-minor] [V phrygian3] [V phrygian3] [I melodic-minor]
                [I phrygian3] [IV dorian] [II locrian] [IIb lydianb7])
           (dup 2)
           (lin {:section :a}
                [{:section :b} (transpose c6)])
           (h/align-contexts :d)
           (parts {:section :a} (each (chans [(patch :vibraphone) (shuftup s0 s1 s2 s3 s4 s5)]
                                             [(patch :flute) o1 (shuftup s0 s1 s2 s3 s4 s5)]
                                             [(patch :acoustic-bass) o1- t-round]))
                  {:section :b} (each (chans [(patch :choir-aahs) vel4 (par s0 s1 s2)]
                                             [(patch :ocarina) vel4 s2- (shuftup s0 s2 s4)]
                                             [(patch :music-box) vel6 o1 (shuftup s0 s1 s2 s3 s4 s5 s6 s7 s8)]
                                             [(patch :acoustic-bass) o1- t-round])))
           (dup 2))

    ]

    ["I V"
     ;; A chord sequence based on I V progressions in major and minor.

     (play dur3

           ;; base I V in minor using melodic minor and superlocrian modes
           (lin [I (scale :melm) (structure :tetrad)]
                [V (scale :alt) (structure :sus47)])

           ;; repeat it one time shifting one structural degree down
           (append s1-)

           ;; repeat this 4 bars sequence modulating it a major third up
           ;; degree I becomes lydian and V mixolydianb2
           (append [(transpose c4-)
                    (parts (scale :melm) (scale :lydian)
                           (scale :alt) [(scale :mixolydianb2) (structure [1 5 9 10])])])

           ;; the whole sequence is repeated 2 times
           (dup 2)

           ;; align all harmonic contexts so the melody can come over without skips between chords
           (h/align-contexts :s)

           ;; on each chord we apply some content
           ;; melody is built using several techniques
           ;; - passing notes
           ;; - randomized diatonic steps
           (let [below (one-of d1- s1-)
                 above (one-of d1 s1)
                 contours [[0 -1 1 0]
                           [0 1 -1 0]
                           [-1 0 1 0]
                           [1 0 -1 0]
                           [1 0 -1 0]
                           [-1 0 1 0]]
                 passings (mapv (partial mapv {0 _ -1 below 1 above}) contours)
                 rand-passing (one-of* (map tup* passings))
                 below-step (one-of d1- d3- d4-)
                 above-step (one-of d1 d3 d4)
                 rand-line (rup 4 (one-of below-step above-step))
                 rand-vel (fn [min max] {:velocity (fn [_] (+ min (rand/rand-int (- max min))))})]


             (each (chans
                    ;; simple choir structural chords
                    [(patch :choir-aahs) vel4 (par s0 s1 s2 s3)
                     (h/drop 1)]
                    ;; simple bass
                    [(patch :acoustic-bass) t-round o1-]
                    ;; melody, composing a line using shuftup rand-passing and rand-line
                    ;; playing it a the vibraphone
                    ;; add some flute and glockenspeil decorations
                    [(shuftup s0 s1 s2 s3)
                     (each (one-of rand-passing rand-line))
                     (chans [(patch :vibraphone) (each (rand-vel 40 70)) (each (maybe vel0))]
                            [(patch :flute)
                             (each (rand-vel 60 80))
                             o1
                             (each (maybe vel0 [(chan inc) (patch :glockenspiel) vel4]))])]))))

    ]

    ["Not too happy birthday"

     ;; A simple experiment on happy birthday chords turned into minor.

     (play

       ;; setting up the main scale
       harmonic-minor

       ;; the chord sequence
       (lin I
            V
            VII
            I
            ;; this notation for the secondary dominant of fourth degree
            ;; is not satisfaying, I would like to be able to write `(Vof IV)` maybe...
            [IV melodic-minor VII]
            IV
            I
            VII)

       ;; aligning harmonic contexts to get voice leading more easily
       (h/align-contexts :s)

       ;; simple chord plus arpegio on each chord.
       (each (par (par s0 s1 s2)
                  [o1 (shuftup s0 s1 s2)]))

       ;; loop 4 times
       (dup 4))

    ]

    ["I.m.M7 VI.alt bVI.7.#11 bII.7.sus4"

     ;; A very artificial sounding chord sequence using quartal voicings and ninuplets (tup of size 9)

     (play (lin [I melodic-minor] [VI superlocrian] [VIb lydianb7] [IIb mixolydian])
           (h/align-contexts :s)
           (dup 2)
           (each (chans [(patch :vibraphone) vel6 t0 (par> d0 d3 d3 d3 d3)]
                        [(patch :acoustic-bass) vel6 t2-]
                        [(patch :taiko-drum) (shuftup vel3 vel5 [vel4 (dupt 2)])]
                        [(ntup> 9 (any-that (within-pitch-bounds? :G-1 :C2)
                                            d1- d1 d3 d3- d4 d4-))
                         vel9
                         (chans (patch :flute)
                                [o1- vel4 (patch :vibraphone)])]))
           (lin _ c6)
           (dup 2) )


    ]]

   ["Melodic"

    ["Target notes"
     ;; Building good rythmic melodies is not easy.
     ;; Here, I will try to start from target notes and fill the holes between them.

     (play aeolian
           (lin s0 s2 s1 s0))

     ;; How to fill between the notes of this simple line?

     (def fill-diatonically
       "A very low level way to connect subsequent notes diatonically using `noon.harmony` directly.
        It feels too complicated for such a simple thing..."
       (sf_ (let [sorted (sort-by :position _)
                  couples (partition 2 1 sorted)]
              (-> (reduce (fn [ret [a b]]
                            (let [va (events/pitch-value a)
                                  vb (events/pitch-value b)
                                  direction (if (> va vb) :down :up)
                                  cnt (loop [cnt 0 current (:pitch a)]
                                        (case direction
                                          :up (if (>= (hc/hc->chromatic-value current) vb)
                                                cnt
                                                (recur (inc cnt) (hc/upd current (hc/d-step 1))))
                                          :down (if (<= (hc/hc->chromatic-value current) vb)
                                                  cnt
                                                  (recur (inc cnt) (hc/upd current (hc/d-step -1))))))]
                              (score/concat-score ret
                                            (score/update-score #{(assoc a :position 0)}
                                                          (rup cnt (case direction :up d1 :down d1-))))))
                          #{}
                          couples)
                  (conj (last sorted))))))

     ;; trying it on a basic structural line
     (play aeolian
           (lin s0 s2 s1 s0)
           fill-diatonically)

     ;; Let's generalise to other layers:

     (defn fill-line
       "This evolution of fill-diatonically let the user specify the harmonic layer.
        It is still relying on `noon.harmony` which is not great."
       [layer]
       (sf_ (let [sorted (sort-by :position _)
                  couples (partition 2 1 sorted)]
              (-> (reduce (fn [ret [a b]]
                            (let [va (events/pitch-value a)
                                  vb (events/pitch-value b)
                                  direction (if (> va vb) :down :up)
                                  [check increment] (case direction :up [>= 1] :down [<= -1])
                                  cnt (loop [cnt 0 current (:pitch a)]
                                        (if (check (hc/hc->chromatic-value current) vb)
                                          cnt
                                          (recur (inc cnt) (hc/upd current (hc/layer-step layer increment)))))]
                              (score/concat-score ret
                                            (score/update-score #{(assoc a :position 0)}
                                                          (rup cnt (ef_ (update _ :pitch (hc/layer-step layer increment))))))))
                          #{}
                          couples)
                  (conj (last sorted))))))

     ;; The same as in previous example
     (play aeolian
           (lin s0 s2 s1 s0)
           (fill-line :c))

     ;; A more elaborated example using structural filling
     (play dur:2
           harmonic-minor
           tetrad
           (patch :orchestral-harp)
           (lin s0 s2 s2- s4 s4- s2 s2- s5-)
           (lin _ [(transpose c6) s2 rev])
           (lin _ s2 s2-)
           (fill-line :s))

     ;; Next step will be to have control over the number of notes between targets.

     (defn target
       [layer size direction duration]
       (sfn score
         (->> score
              (map (fn [e]
                     (->> (range size)
                          (map (fn [i]
                                 (-> (update e :pitch
                                             (hc/layer-step
                                              layer
                                              (case direction
                                                :up (inc i)
                                                :down (- (inc i)))))
                                     (update :position - (* (inc i) duration))
                                     (assoc :duration duration))))
                          (into #{e}))))
              (score/merge-scores))))

     ;; It is a step in the right direction but it overlaps passing notes
     (play (lin _
                [s2 (target :c 3 :up 1/4)]
                [s1- (target :d 3 :down 1/4)]
                [_ (target :c 3 :up 1/4)])
           (out/options {:filename "test/trash/target"}))

     ;; The problem here is that the precedent note overlaps the targeting notes.

     ;; Using the =noon.harmony/simplest-connections= we can connect two notes in a given amount of steps using.
     ;; Let's build a function that leverage that to fill subsequent notes in a melodic way.

     (defn connect [& sizes]
       (sf_ (let [sorted (sort-by :position _)]
              (reduce (fn [s [n1 n2]]
                        (let [hcs (loop [sizes sizes]
                                    (if-let [[s & sizes] (seq sizes)]
                                      (or (hc/simplest-connection s (:pitch n1) (:pitch n2))
                                          (recur sizes))))
                              duration (/ (:duration n1) (dec (count hcs)))]

                          (into s (map-indexed (fn [idx pitch]
                                                 (assoc n1
                                                        :pitch pitch
                                                        :position (+ (* idx duration) (:position n1))
                                                        :duration duration))
                                               (butlast hcs)))))
                      #{(last sorted)} (partition 2 1 sorted)))))

     (play harmonic-minor
           (lin I [VI lydianb7] V IV [II phrygian3] [V aeolian] [IIb lydian])
           (h/align-contexts :s)
           (m/$lin [(lin s0 s2 s2- s4) (maybe [rev s2])])
           (lin _ s1 s1- _)
           (chans [(patch :tango) (connect 5 3 2 1 0)]
                  [(patch :ocarina) vel6 s2 (connect 2 1 0)]
                  [(patch :acoustic-bass) o1- s2- (connect 1 0)]))


     ;; The =connect= function is now available in =noon.lib.melody=

     (play harmonic-minor
           (lin I [VI lydianb7] V IV [II phrygian3] [V aeolian] [IIb lydian])
           (h/align-contexts :s)
           (m/$lin [(lin s0 s2 s2- s4) (maybe [rev s2])])
           (lin _ s1 s1- _)
           (chans [(patch :tango) (m/connect 5 3 2 1 0)]
                  [(patch :ocarina) vel6 s2 (m/connect 2 1 0)]
                  [(patch :acoustic-bass) o1- s2- (m/connect 1 0)]))

    ]

    ["Passing notes"
     ["simple"

      ;; A bunch of simplistic passing note examples

      (play dorian
            (rep 4 s1)
            (each (tup c1- s2 s1 s0))
            (tup _ rev)
            (rep 4 (transpose c3))
            (append rev))

      (play dorian
            (rep 4 s1)
            (each (tup _ s2))
            (each (tup c1- d2 d1 d0)))

      (play melodic-minor
            dur4
            (append (transpose c3) (transpose c6) (transpose c3))
            (dup 2)
            (each (shuftup s0 s1 s2 s3 s4))
            (each (tup _ (one-of s1 s2 s1- s2- s3 s3-)))
            (each (one-of (tup c1- d2 d1 d0)
                          (tup c1- s1- s0 s2))))

      (play dur4
            (append (transpose c3) (transpose c6) (transpose c3))
            (each (one-of phrygian6 lydian melodic-minor))
            (dup 2)
            (each (chans [(patch :acoustic-bass) t2- (tup _ s2 s1- _)]
                         [(patch :flute) vel8]
                         [(patch :vibraphone) vel4 (par s0 d4 d6 d8 d10 d12)]
                         [(patch :taiko-drum)
                          (r/gen-tup 10 4 :euclidean)
                          (each [(one-of s0 s1 s1-) (one-of vel1 vel3 vel5)])]))
            (parts (chan 1)
                   [(each (shuftup s0 s1 s2 s3 s4))
                    (each (tup _ (one-of s1 s2 s1- s2- s3 s3-)))
                    (each (one-of (tup c1- d2 d1 d0)
                                  (tup c1- s1- s0 s2)
                                  (tup c1- s1- s2- s0)))
                    (each (one-of vel5 vel6 vel7 vel9))]))

      (play melodic-minor
            (shuflin s0 s1 s2 s3)
            (each (let [step (one-of s1 s2 s3 s1- s2- s3-)
                        ap (lin c1- d1 s1-)]
                    (tup [_ ap] [step ap] _ step)))
            (append c2- c2-))

      (play melodic-minor
            (lin (shuflin s0 s1 s2 s3)
                 [{:passing true} (shuflin s0 s1 s2 s3)])
            (each (let [step (one-of s1 s2 s3 s1- s2- s3-)
                        ap (lin c1- d1)]
                    (tup [_ ap] [step ap] _ step (par s2- s2))))
            (append c4-)
            (dup 2))

      (play melodic-minor
            dur:3
            (shuflin s0 s2 s4)
            (each (one-of (shuftup _ c1- d1)
                          (shuftup _ d1 d1-)))
            (m/permutation :rand)
            (rep 3 (one-of (s-shift 1) (s-shift -1)))
            (rep 3 (transpose c3))
            (dup 2))

      ;; this one is more interesting
      (play dorian+4
            (lin I IV)
            (m/$lin
             [;; a simple tup using open triad
              (shuftup s0 s2 s4)
              ;; adding chromatic inferior triad and diatonic superior triads
              (tup c1- _ d1)
              ;; mixing all !
              ;; this is the interesting part:
              ;; a passing is often occuring before the note it targets
              ;; but actually the order can be reversed and we can even interpose
              ;; other notes between the passing tone and the targetted one.
              ;; this way to do it is radical but it somehow works (being quite dissonant of course)
              (m/permutation :rand)
              (rep 4 (one-of (s-shift 1) (s-shift -1)))])
            (append (transpose c3))
            (append (s-shift -1)))

     ]

     ["intermediate"

      (defn chromatic-double-passing [side]
        (sf_
          (assert (= 1 (count _))
                  (str `chromatic-double-passing
                       "works only on single note scores"))
          (let [target (first _)
                d-suroundings (hc/diatonic-suroundings (:pitch target))
                c-space (get d-suroundings (case side :up 1 :down 0))
                step (case side :up 1 :down -1)]
            (score/update-score _
                          (if (= c-space 2)
                            (tup (d-step step) (c-step step) same)
                            (tup (d-step step) (case side :up c1- :down d1) same))))))

      (play dur4
            (rup 6 (one-of d4 d3-))
            (each (tup (chromatic-double-passing :down)
                       [d6 (chromatic-double-passing :up)])))

      (let [c-d+ (efn e (if-let [p- (get-in (hc/neibourhood (:pitch e)) [:down :c])]
                          (assoc e :pitch p-)
                          (d1 e)))]
        (play dur:4
              (rep 14 d1)
              (each (tup c-d+ _))))

     ]

     ["interleaving"

      ;; Experimenting interleaving passing notes

      (defn interpose-with [f]
        (sf_ (if (m/line? _)
               (set (mapcat (fn [[a b]] (if b ((f a b)) a))
                            (partition 2 1 nil (sort-by :position _)))))))

      (defn interleaved [& xs]
        (sf_ (let [scores (map (partial score/update-score _) xs)
                   counts (map count scores)
                   durations (map score/score-duration scores)]
               (assert (apply = counts)
                       "interleaved scores should have same number of elements")
               (assert (apply = durations)
                       "interleaved scores should have same duration")
               (assert (apply = (mapcat (partial map :duration) scores))
                       "interleaved scores should have even durations")
               (let [duration (/ (first durations) (first counts))
                     shift (/ duration (count scores))]
                 (:score
                  (reduce (fn [{:as state :keys [at]} xs]
                            (-> state
                                (update :at + duration)
                                (update :score into (map-indexed (fn [i n] (assoc n :position (+ at (* i shift)) :duration shift)) xs))))
                          {:score #{} :at 0}
                          (apply map vector (map score/sort-score scores))))))))

      (play dur4
            (interleaved
             (rup 8 d1 :skip-first)
             (rup 8 d1- :skip-first)))

      (let [up (one-of d1 s1)
            down (one-of c1- d1- s1-)
            rand-double-passing
            (one-of (tup up _ down _)
                    (tup down _ up _)
                    (tup down up down _)
                    (tup up down up _))]
        (play harmonic-minor
              dur4
              (interleaved
               [(nlin 4 (shuftup s0 s1 s2 s3)) (each rand-double-passing)]
               [(nlin 4 (shuftup s0 s1 s2 s3)) s2 (each rand-double-passing)])))

      (defn interleaving [polarities a b]
        (loop [s [] ps polarities a a b b]
          (if-let [[p & ps] (seq ps)]
            (let [[nxt a' b'] (case p 0 [(first a) (next a) b] 1 [(first b) a (next b)])]
              (recur (conj s nxt) ps a' b'))
            s)))

      (defn rand-interleaving
        ([a b]
         (interleaving (rand/shuffle (concat (repeat (count a) 0) (repeat (count b) 1)))
                       a b))
        ([a b & xs]
         (reduce rand-interleaving
                 (rand-interleaving a b)
                 xs)))

      (require '[clojure.math.combinatorics :as combinatorics])

      (defn interleavings [a b]
        (reduce (fn [ret perm]
                  (conj ret (interleaving perm a b)))
                []
                (combinatorics/permutations (concat (repeat (count a) 0) (repeat (count b) 1)))))

      (u/defn* randomly-interleaved
        "randomly interleave the result of the given updates"
        [xs]
        (sf_ (:score
              (reduce (fn [state n]
                        (-> state
                            (update :score conj (assoc n :position (:at state)))
                            (update :at + (:duration n))))
                      {:at 0 :score #{}}
                      (apply rand-interleaving (map (fn [u] (sort-by :position (score/update-score _ u))) xs))))))

      (defn n-firsts [n]
        (sf_ (->> (group-by :position _)
                  (sort)
                  (take n)
                  (map second)
                  (reduce into #{}))))

      (let [up (one-of d1 s1)
            down (one-of c1- d1- s1-)
            rand-double-passing
            (one-of (tup _ up down _)
                    (tup _ down up _)
                    (tup up _ down _)
                    (tup down _ up _)
                    (tup down up down _)
                    (tup up down up _))]
        (play harmonic-minor
              dur2
              (randomly-interleaved
               [(chan 1) (nlin 4 (shuftup s0 s1 s2 s3)) (each rand-double-passing)]
               [(chan 2) (nlin 4 (shuftup s0 s1 s2 s3)) s4- (each rand-double-passing)]
               [(chan 3) (nlin 4 (shuftup s0 s1 s2 s3)) s4 (each rand-double-passing)])))

     ]

     ["experience 1"

      ;; Try to implement diverse melodic passing notes things.

      ;; Mono harmony passing notes

      (play (rep 6 s1)
            (m/connect 1))

      (play o1
            (rep 6 s1-)
            (m/connect 1))

      (defn connect-with
        "use `f` to connect subsequent notes of a score."
        [f]
        (connect-by :position
                    (fn [chunk1 chunk2]
                      ;; `noon.score/connect-by` is chunking the received score by :position
                      ;; the two chunks are sets of events
                      ;; but we assumes a monophonic scores so we only take care of first (and only) event of each chunk
                      (let [from (first chunk1)
                            to (first chunk2)]
                        (score/update-score #{(assoc from :position 0)}
                                      [(lin _ [(ef_ (assoc _ :pitch (:pitch to)))
                                               f])
                                       (adjust from)])))))

      (play (lin s0 s2 s4)
            (connect-with d1))

      (play (lin s0 s2 s4)
            (lin s0 s1 s2)
            (connect-with (tup d1- d1)))

      (play [aeolian dur:2]
            (lin s0 s2 s4)
            (lin s0 s1 s2)
            (connect-with (shuflin d1 c1-)))

      ;; With chord changes

      (play harmonic-minor
            (lin I VII)
            (nlin> 3 (transpose c3))
            (h/align-contexts :s)
            (dup 2)
            (each (ntup> 6 s1))
            (connect-with d1))


      ;; With parts and vsl


      (play
       ;; grid
       [harmonic-minor
        (lin I VII)
        (nlin> 3 (transpose c3))
        (h/align-contexts :s)
        (dup 4)]
       ;; parts
       (par
        ;; flute melody
        [(vsl :flute-1 :staccato)
         o1 vel4
         (each (shuftup s0 s2 s4))
         (connect-with (one-of d1- d1))]
        ;; bass
        [(vsl :solo-double-bass :pizzicato)
         o1- t-round]
        ;; viola comping
        [(vsl :chamber-violas :pizzicato)
         vel5
         (each (one-of (tup s1 (par s2 s3) vel0)
                       (tup vel0 s1 (par s2 s3))))]))

      ;; Targetting other chord/key

      (defn connect-with2 [f]
        (connect-by :position
                    (fn [chunk1 chunk2]
                      (let [from (first chunk1)
                            to (first chunk2)]
                        (score/update-score #{(assoc from :position 0)}
                                      [(lin _ [(repitch (events/event->pitch to)) f])
                                       (adjust from)])))))

      (comment

        (play (lin d0 [IIb mixolydian])
              (connect-with d1-))
        (play (lin d0 [IIb mixolydian])
              (connect-with2 d-floor))
        (score (lin d0 [IIb mixolydian])
            (connect-with2 _))
        (score (lin d0 [IIb mixolydian]))
        (play (lin d0 d-floor)))

      ;; This is difficult... to be continued

     ]

     ["polarity"

      ;; This morning I was playing modal melodies on the flute, and experimenting with different polarity cycles.

      [0 0 1 0]
      [0 1 1 0]

      ;; 0 can represent tonic and 1 dominant, whatever it means depending on the harmonic context.

      ;; let's take the phrygian mode as an example.

      ;; using this polarity sequence:
      [0 0 1 0 1 0 0 1]

      (play phrygian
            ;; the polarity are implemented using degrees
            ;; I is 0
            ;; VII is 1 (The VII degree is often good as dominant)
            (lin I I VII I VII I VII VII)

            (mixlin s0 s2)
            (each (chans [(patch :acoustic-bass) o2- (maybe t-round)]
                         [(patch :ocarina) s2 (shuftup s0 s2 s4)]))

            ;; adding a bunch of noise (feel free to remove the following updates)
            (lin _ [rev (transpose c3-)])
            (parts (chan 1) (connect-with (one-of (one-of d1 d1-)
                                                  (shuflin (one-of s1 s1-) (one-of d1 d1-))))
                   (chan 0) (each (probs {(tup (one-of s1 s1-) _) 1
                                          _ 4}))))

      ;; Let's experiment around creating those polarity sequences

      (comment
        (let [id identity
              rev (fn [x] (mapv {0 1 1 0} x))
              _dup (fn [x] (vec (concat x x)))
              cat (fn [& xs] (fn [x] (vec (mapcat (fn [f] (f x)) xs))))
              acc (fn [n f] (apply comp (repeat n f)))
              each (fn [f] (fn [x] (vec (mapcat (comp f vector) x))))
              _scan (fn [size step f] (fn [x] (vec (mapcat f (partition size step x)))))
              >> (fn [& xs] (fn [x] (reduce #(%2 %1) x xs)))
              upd (fn [x f] (f x))]
          (upd [1]
               (>> (acc 3 (cat id rev))
                   (each (cat id rev id))))))

      ;; to be continued...

     ]

     ["degree moves"

      ;; It seems that the degree that is under the current one can serve as kind of a dominant.

      (play dorian
            (nlin> 8 s1)
            [(patch :ocarina) (connect-with (degree -1))])

      (play dorian
            dur4 o1 (lin _ (nlin> 3 s1-))
            [(patch :ocarina) (connect-with (degree 1))]
            (each (tup s0 s2))
            (connect-with (degree 1)))

      (let [pol+ {:polarity 0}
            pol- {:polarity 1}
            invert-pol (each {:polarity (fn [x] (case x 0 1 1 0))})]
        (play lydianb7
              dur2
              (lin pol+ pol-)
              (lin _ invert-pol)
              (tup _ invert-pol)
              (rep 4 (transpose c3-))
              (h/align-contexts :s)
              (dup 2)
              (parts pol+ _
                     pol- (each (one-of (degree -1) (degree 1))))
              (chans [(patch :ocarina) (each [(one-of s0 s1) (shuftup s0 s1 s2 s3)]) (connect-with (one-of d1 d1-))]
                     [(patch :acoustic-bass) o1- (each (one-of s0 s1- s2-))])))

      (let [pol+ {:polarity 0}
            pol- {:polarity 1}
            invert-pol (each {:polarity (fn [x] (case x 0 1 1 0))})]
        (play (chans [(patch :ocarina)
                      s2- (ntup> 7 s1)
                      (shuftup [_ (connect-with d1)]
                               [rev s1- (connect-with d1-)])
                      (dupt 16)]
                     [(patch :acoustic-bass) (dupt 64) o2- t-round (each (maybe s2- s2))])
              (h/grid [phrygian3
                       (tup pol+ pol-)
                       (tup _ invert-pol)
                       (tup _ invert-pol)
                       (rup 4 (transpose c3-))
                       (h/align-contexts :s)
                       (dupt 2)
                       (parts pol+ _
                              pol- (each (degree -1)))])
              (adjust {:duration 64})))

     ]

     ["scanning"


      ;; as mentioned previously, in order to build or evolve a melody,
      ;; it can be handy to start with a squeleton and fill the hole between them with passing tones.
      ;; In order to be able to do so, it is necessary to scan the melodic line 2 by 2 in order
      ;; to determine the correct passing tones.

      ;; in this first example we are just using each to decorate our skeleton line
      ;; but it do not "connect" the subsequent notes, it just decorate them.
      (play (patch :electric-piano-1)
            aeolian
            (nlin> 6 s1)
            (each (tup _ c1- [s1 c1-] _)))

      ;; here we can get a glimpse at what we are trying to achieve
      ;; Our skeleton line: (nlin> 4 s1) is regular,
      ;; so we know that every next note will be one structural step above
      ;; therefore we can decorate each note and finish the decoration with
      ;; one passing note toward the next skeleton note.
      ;;
      ;; It works well but it lacks flexibility
      ;; (we want to be able to achieve similar result regardless of the skeleton line)
      (play (patch :electric-piano-1)
            dur2 aeolian
            (nlin> 4 s1)
            (each (tup
                   ;; decoration
                   _ [s2 c1-] c1- _ s2
                   ;; anticipating of the next note
                   [s1 d1])))

      ;; it could make sense to have some sort of scan/partition mapping operator

      '(defn scan
        {:doc (str "Chunk the score using the `by` function. "
                   "Chunks are partitioned by `size` and stepped by `step`. "
                   "`f` is applied to each chunks partition and should return a single score. "
                   "Resulting scores are merged together.")}
        [by size step f]
        (sf_ (->> (chunk-score _ by)
                  (partition size step)
                  (map f)
                  (score/merge-scores))))

      (play (patch :electric-piano-1)
            dur2
            aeolian (nlin> 4 s3)
            (scan :position 2 1
                  (fn [[a b]]
                    (let [start (first a)
                          {target-pitch :pitch} (first b)]
                      (score/update-score #{start}
                                    (each (tup _ [s2 c1-] c1- _ s2
                                               [(ef_ (assoc _ :pitch target-pitch)) d1])))))))

      '(defn in-place
        {:doc (str "Turn the given update `u` into an update that reposition received score to position zero before applying `u` to it. "
                   "The resulting score is then adjusted to its initial duration and shifted to its original position. "
                   "This is useful when you need to scan update a score. "
                   "It is similar to what the `noon.score/each` function is doing.")}
        [u]
        (sf_ (let [score-origin (score-origin _)
                   score-duration (- (score/score-duration _) score-origin)]
               (score/update-score (score/shift-score _ (- score-origin))
                             [u (adjust {:position score-origin :duration score-duration})]))))

      ;; slight variation of the previous snippet, but using the 'in-place function which is more self explanatory.

      (play (patch :electric-piano-1)
            aeolian
            (nlin> 8 [(degree 4) s1-])
            (scan :position 2 1 (fn [[a b]]
                                  (let [start (first a)
                                        {target-pitch :pitch} (first b)]
                                    (score/update-score #{start}
                                                  (in-place (tup _ [s2 c1-] c1- _ s2
                                                                 [(ef_ (assoc _ :pitch target-pitch)) d1])))))))



      ;; This approach seems more flexible even if it is still a bit verbose

      ;; This scan operation can be built following another approach
      ;; by implementing an update builder that limits the effect of an update to a given time span,
      ;; we can scan the score by a given step, applying the update only to a limited time span.
      ;; Accumulating the score along the way, subsequent framed updates will "see" the effect of previous framed updates.

      '(defn only-between
        {:doc (str "Use `f` to update the subscore delimited by `beg` and `end` positions. "
                   "Leave other events unchanged.")}
        [beg end f]
        (par [(trim beg end) (in-place f)]
             (trim nil beg)
             (trim end nil)))

      (play (nlin> 8 d1)
            (only-between 4 6 o1))

      '(defn scan>
        {:doc (str "Accumulative scan. "
                   "Use `f` to accumulatively update time slices of given `size` of the score, stepping by `step`.")
         :tags [:temporal :accumulative :iterative]}
        ([size f]
         (scan> size size f))
        ([size step f]
         (sfn s (reduce (fn [s from]
                          (score/update-score s (only-between from (+ from size) f)))
                        s (range 0 (score-duration s) step)))))

      (play (nlin> 8 d1)
            (scan> 4 3 (tup _ d1 d1-)))

      ;; The `scan` and `scan>` function are now part of `noon.score` incubation subsection.
      ;; `only-between` and `in-place` are part of `noon.score`

     ]]

    ["Canon"
     ;; First thing would be to come up with a simple melodic motiv.
     ;; It will be based on a triad, with some decorating tones.

     ;; The skeleton could be something like

     (play (shuftup s0 s1 s2))

     ;; We can start in 3/4.
     ;; The next step will be to decorate it.

     ;; Previously we've discussed the connect function that can do something like this

     (play (shuftup s0 s1 s2)
           (m/connect 1))

     ;; But it is not really what we want.


     (def decorate
       (sf_ (let [sorted (sort-by :position _)]
              (reduce (fn [s [n1 n2]]
                        (into s (score/update-score #{n1 n2} (maybe (m/connect 1)))))
                      #{(last sorted)} (partition 2 1 sorted)))))

     (play dur2
           (lin (shuftup s0 s1 s2 s3)
                [(one-of s1 s1-) (shuftup s0 s1 s2 s3)])
           decorate
           (lin _ (s-shift 1) (s-shift -1) _)
           (lin _ (s-shift 2))
           (chans [(patch :ocarina) o1 (s-shift -1)]
                  [(sf_ (score/shift-score _ 2))]
                  [(patch :acoustic-bass) o2- (s-shift 1) (sf_ (score/shift-score _ 5))])
           (h/grid dur2
                   harmonic-minor
                   (lin I IV VII I [IV melodic-minor VII] IV [V harmonic-minor VII] VII)
                   (dup 4)
                   (h/align-contexts :s)))


    ]]

   ["Techniques"

    ;; Some attempts to implement or illustrate various musical ideas using noon.

    ["Barry Harris"

     (def barry-harris (scale [0 2 4 5 7 8 9 11]))


     (play barry-harris
           (tup d0 d3 d4 d7)
           (tup d0 d2)
           (rep 4 d1))


     (let [chord-tones [d0 d2 d4 d7]]
       (play barry-harris
             (lin d0 d3)
             (rep 8 (one-of d1- d1))
             (each [(chans [(patch :pad-1-new-age) o1- vel3 (par* chord-tones)]
                           [(patch :ocarina) vel4 (shuftup* chord-tones) (each (maybe (tup (one-of d1 d1-) d0)))]
                           [(patch :vibraphone) vel5 o1 (ntup 6 [(one-of* chord-tones) (maybe o1) (maybe (tup d1- d0))])])
                    (maybe rev)])))


     (def barry-harris2 [barry-harris (structure [0 2 4 7])])


     (play barry-harris2
           (lin I VI VII IV)
           (h/align-contexts :d)
           (each (chans [(patch :brass) (par s0 s1 s2 s3)]
                        [(patch :acoustic-bass) o1- t-round]
                        [(patch :ethnic) o1 (shuftup s0 s1 s2 s3 s4 s5 s6)]))
           (rep 2 s1)
           (append (transpose c3)))


     (play barry-harris2
           (lin IV I)
           (h/align-contexts :d)
           (each (par s0 s1 s2 s3))
           (rep 4 (transpose c3))
           h/voice-led)

    ]

    ["Symetric modes"

     (def symetric-modes {:half-whole (scale [0 1 3 4 6 7 9 10])
                          :whole-half (scale [0 2 3 5 6 8 9 11])
                          :whole (scale [0 2 4 6 8 10])
                          :augm-half (scale [0 3 4 7 8 11])
                          :half-augm (scale [0 1 4 5 8 9])
                          :messian3 (scale [0 2 3 4 6 7 8 10 11])
                          :messian4 (scale [0 1 2 5 6 7 8 11])
                          :messian5 (scale [0 1 5 6 7 11])
                          :messian6 (scale [0 2 4 5 6 8 10 11])
                          :messian7 (scale [0 1 2 3 5 6 7 8 9 11])})


     (play (symetric-modes :augm-half)
           (:two {:one (rup 8 (one-of d1 d1- d2 d2- d3 d3-))
                  :two (shuftup d1 d2 d3 d4 d5 d6 d7)})

           (patch :electric-piano-1)
           (rep 32 (one-of (each d3)
                           (each d3-)
                           (m/rotation 1/2)
                           (m/permutation :rand {:grade 2})
                           (m/contour :similar {:delta 0 :layer :d}))))


     (defn rand-structure [size]
       (ef_ (let [degree-count (-> _ :pitch :scale count)
                  degrees (take size (rand/shuffle (range degree-count)))]
              ((structure (vec (sort degrees))) _))))


     (def rand-degree
       (ef_ (let [scale-size (-> _ :pitch :scale count)
                  deg (rand/rand-nth (range 1 scale-size))]
              ((degree (rand/rand-nth [(- deg) deg])) _))))


     (defn rand-tup [size]
       (e->s event
             (let [degree-count (-> event :pitch :scale count)
                   degrees (take size (rand/shuffle (range degree-count)))]
               (score/update-score #{event} (tup* (mapv d-step degrees))))))



     (play (symetric-modes :half-whole)
           (rand-structure 3)
           (rep 3 rand-degree)
           (each (chans [vel4 h/simple-chord]
                        [(patch :music-box) o1 (rand-tup 6) (each (one-of vel0 vel4 vel6 vel7))]))
           (append [rev s2])
           (append (transpose c5))
           (append (between 0 1/3)))

    ]

    ["Arvo Part"

     (let [m-line (fn [size]
                    (rand/rand-nth (vals {:up-to [(rep size d1-) rev]
                                        :up-from (rep size d1)
                                        :down-to [(rep size d1) rev]
                                        :down-from (rep size d1-)})))
           base (rand/shuffle (map vector
                                 [s0 s1 s2 (one-of s0 s1 s2)]
                                 (map m-line (rand/shuffle (rand/rand-nth (u/sums 12 4 [2 3 4 5]))))))]
       (play lydianb7
             (lin* base)
             (each (chans [(patch :piccolo) vel6 o1]
                          [(patch :flute) vel3 o1 d5-]
                          [(patch :accordion) vel4 d0]
                          [(patch :choir-aahs) s-floor (vel-humanize 7 [40 80])]
                          [(patch :choir-aahs) s-floor o1 s1 (vel-humanize 7 [40 80])]
                          [(patch :acoustic-bass) C-2 t-floor]))

             m/connect-repetitions
             (append [rev (transpose c3-)])
             (append dorian)
             (dup 2)))


    ]

    ["Bartok harmony axis"

     (let [L- (transpose c5)
           L+ (transpose c5-)
           R- (transpose c3)
           R+ (transpose c3-)
           M (transpose c6)]
       (play (rep 8 [(one-of L- L+) (maybe R- R+ M) (one-of ionian aeolian)])
             (h/align-contexts :d)
             (chans [(patch :aahs) (each (par s0 s1 s2))]
                    [(patch :ocarina) o1 (each (shuftup s2- s1- s0 s1 s2 s3))]
                    [(patch :acoustic-bass) o1-
                     t-round
                     (maybe s1 s1-)])
             (lin _ s1 s1- _)))

     (let [L- (transpose c5)
           L+ (transpose c5-)
           R- (transpose c3)
           R+ (transpose c3-)
           M (transpose c6)
           tup1 (mixtup s2- s1- s0 s1 s2 s3)
           tup2 (mixtup s2- s1- s0 s1 s2 s3)]
       (play (rep 8 [(one-of L- L+) (maybe R- R+ M) (one-of ionian aeolian)
                     (maybe dur2 dur:2)])
             (h/align-contexts :d)
             (chans [(patch :aahs)
                     (each [add2 (par s0 s1 s2 s3)])
                     m/connect-repetitions]
                    [(patch :ocarina) o1 add2 (each [(one-of tup1 tup2) (maybe rev)])]
                    [(patch :acoustic-bass) o1-
                     t-round
                     (maybe s1 s1-)])
             (lin _ s1 s1- _)))

     (let [L- (transpose c5)
           _L+ (transpose c5-)
           R- (transpose c3)
           R+ (transpose c3-)
           M (transpose c6)

           base [(rand/rand-nth [R- R+ M]) (rand/rand-nth [ionian aeolian])]
           rand-color [(maybe R- R+ M) (one-of ionian aeolian)]
           tup1 (mixtup s2- s1- s0 s1 s2 s3)
           tup2 (mixtup s2- s1- s0 s1 s2 s3)]
       (play base
             (lin _ [L- rand-color] rand-color [L- rand-color] _)
             (lin _ M rev)
             (h/align-contexts :d)
             (chans [(patch :aahs)
                     (each [add2 (par s0 s1 s2 s3)])
                     m/connect-repetitions]
                    [(patch :ocarina) o1 add2 (each [(one-of tup1 tup2) (maybe rev)])]
                    [(patch :acoustic-bass) o1-
                     t-round
                     (maybe s1 s1-)])
             (lin _ s1 [rev s1-] _)))

     (let [initial [{:harmonic-coords [0 0]} melodic-minor sixth]
           up [{:harmonic-coords (fn [[x y]] [x (mod (inc y) 3)])} (transpose c5)]
           down [{:harmonic-coords (fn [[x y]] [x (mod (dec y) 3)])} (transpose c5-)]
           left [{:harmonic-coords (fn [[x y]] [(mod (dec x) 4) y])} (transpose c3)]
           right [{:harmonic-coords (fn [[x y]] [(mod (inc x) 4) y])} (transpose c3-)]]
       (play initial
             (lin> _ up left down)
             (lin _ up)
             (lin _ [rev left])
             (lin _ [right right])
             (h/align-contexts :d)
             (chans [(patch :aahs) (structure [1 2 5 6]) (each (par s0 s1 s2 s3))]
                    (let [tup1 (mixtup s2- s1- s0 s1 s2 s3)
                          tup2 (mixtup s2- s1- s0 s1 s2 s3)]
                      [(patch :ocarina) o1 add2 (each [(one-of tup1 tup2) (maybe rev)])])
                    [(patch :acoustic-bass) o1-
                     t-round
                     (maybe s1 s1- s2-)])
             (lin _ s1 [up s1-] up)))


     (let [initial [lydian seventh]
           up (transpose c5)
           down (transpose c5-)
           left (transpose c3)
           right (transpose c3-)]
       (play ;; grid
        [initial
         (lin> _ up left down)
         (each (maybe (degree 2) (degree -2)))
         (lin _ up)
         (lin _ [rev left])
         (lin _ [right right])
         (h/align-contexts :d)]
        ;; voices
        (chans [(patch :aahs) (each (par s0 s1 s2 s3))]
               #_[(patch :aahs) t-round (each (par d0 d3 d6 d9)) #_h/voice-led]
               (let [tup1 [(structure [2 3 4 6]) (mixtup s3- s2- s1- s0 s1 s2 s3 s4)]
                     tup2 (mixtup d3- d2- d1- d0 d1 d2 d3 d4)]
                 [(patch :ocarina) o1 (each [(one-of tup1 tup2) (maybe rev)])])
               [(patch :acoustic-bass) o2-
                t-round
                (each (probs {_ 3
                              (one-of s1- s2) 3
                              (tup _ (one-of s1- s2)) 1
                              (tup (one-of s1- s2) _) 1}))])
        ;; why not ?
        (lin _ s1 [up s1-] up)
        (out/options :bpm 40 :xml true)))

    ]

    ["Simple counterpoint"

     (require '[clojure.math.combinatorics :as combinatorics])

     (let [perms (combinatorics/permutations [0 1 2 3])
           complementary-map
           (reduce (fn [acc p]
                     (assoc acc p
                            (filter (fn [p']
                                      (every? (fn [[a b]] (not= (mod a 3) (mod b 3)))
                                              (map vector p p')))
                                    perms)))
                   {} perms)

           [base complements] (rand/rand-nth (seq complementary-map))
           voice1 (rand/rand-nth complements)
           voice2 (map (fn [a b]
                         (first (filter (complement (set (map #(mod % 3) [a b])))
                                        [0 1 2])))
                       base
                       voice1)]

       (play (patch :electric-piano-1)
             (chans (lin* (map s-step base))
                    [o1- (lin* (map s-step voice1))]
                    [o1 (lin* (map s-step voice2))])
             [aeolian
              (lin _ (degree -1))
              (lin _ s1)
              (lin _ [(degree 3) s1-])
              (lin _ (transpose c3-))]
             ($by :channel (connect-with (probs {void 5 d1 1 d1- 1})))))

     ;; this complementary util is interesting, but the way I get the third voice is not pretty.
     ;; How about introducing another level ?

     (defn complementarity-tree

       ([structure-size sequence-size]
        (let [elements (range structure-size)
              q (quot sequence-size structure-size)
              r (rem sequence-size structure-size)
              base (apply concat (repeat q elements))
              partials (filter (fn [s] (= r (count s))) (combinatorics/subsets elements))
              permutations (mapcat (fn [p] (combinatorics/permutations (concat base p))) partials)]
          (complementarity-tree [] structure-size (set permutations))))

       ([at structure-size perms]
        (if-let [perms
                 (some-> (if (seq at)
                           (filter (fn [p']
                                     (every? (fn [xs]
                                               (apply distinct?
                                                      (map #(mod % structure-size) xs)))
                                             (apply map vector p' at)))
                                   perms)
                           perms)
                         seq
                         set)]
          (->> perms
               (map (fn [child]
                      [child
                       (complementarity-tree
                        (conj at child)
                        structure-size
                        (disj perms child))]))
               (into {})))))

     (defn leaves-paths
       ([m] (leaves-paths m []))
       ([x at]
        (if (and (map? x) (not-empty x))
          (mapcat (fn [[k v]] (leaves-paths v (conj at k))) x)
          [at])))

     (let [[v1 v2 v3] (->> (complementarity-tree 3 3)
                           (leaves-paths)
                           (filter #(= 3 (count %)))
                           (rand/rand-nth))]
       (play [dur3
              aeolian
              (lin _ (degree -1))
              (lin _ s1)
              (lin _ [(degree 3) s1-])
              (lin _ [s1 (transpose c3-)])]
             (patch :electric-piano-1)
             (each (! (let [[v1 v2 v3] (rand/shuffle [v1 v2 v3])]
                        (chans (tup* (map s-step v1))
                               [o1- (tup* (map s-step v2))]
                               [o1 (tup* (map s-step v3))]))))
             ($by :channel (connect-with (probs {void 5 d1 1 d1- 1})))))

     ;; could this complementarity-tree be used to for rythmn ?

     (let [[[r1 r2 r3] [l1 l2 l3]] (->> (complementarity-tree 3 3)
                                        (leaves-paths)
                                        (filter #(= 3 (count %)))
                                        (rand/shuffle))
           f (fn [r l] (tup* (map (fn [r l]
                                   [(s-step l)
                                    (case r
                                      0 _
                                      1 (tup [dur2 _] (one-of d1 d1-) _)
                                      2 (one-of (tup _ d1 d1- _)
                                                (tup _ d1- d1 _)))])
                                 r l)))]
       (vsl/noon {:play true}
                 (score [dur3
                      aeolian
                      (lin _ (degree -1))
                      (lin _ s1)
                      (lin _ [(degree 3) s1-])
                      (lin _ [s1 (transpose c3-)])]
                     (each (! (let [[a b c] (rand/shuffle [(f r1 l1) (f r2 l2) (f r3 l3)])]
                                (chans
                                             ;[(vsl :flute-1 :staccato) vel3 o1 (s-shift 1) c]
                                 [(vsl :solo-violin-1 :pizzicato) o1 b]
                                 [(vsl :solo-viola :pizzicato) c]
                                 [(vsl :solo-cello-1 :pizzicato) o1- a])))))))

     ;; We miss meaninful connections between triad degrees, here we only do ornementation.
     ;; This is also a bit too monotonic.

     (let [[arpegios ornamentations harmonic-sequences]
           (->> (complementarity-tree 3 3)
                (leaves-paths)
                (filter #(= 3 (count %)))
                (rand/shuffle))

           choices {:harmony {0 _
                              1 [(degree 3) (s-shift -1)]
                              2 [(degree 4) (s-shift -2)]}
                    :arpegio {0 s0 1 s1 2 s2}
                    :ornamentation {0 void
                                    1 d1
                                    2 d1-}
                    :instruments {0 [vel8 (vsl :chamber-violins-1 :legato) o1]
                                  1 [vel7 (vsl :chamber-violas :legato)]
                                  2 [vel6 (vsl :chamber-cellos :legato) o1-]}}

           degrees (mapcat (fn [s]
                             (map (choices :harmony) s))
                           harmonic-sequences)
           lines (map (fn [offset]
                        [(get-in choices [:instruments offset])
                         (lin* (map (fn [d a c]
                                      [d (tup* (map (fn [step orn]
                                                      [(get-in choices [:arpegio step]) {:connection orn}])
                                                    a c))])
                                    degrees
                                    (drop offset (cycle arpegios))
                                    (drop offset (cycle ornamentations))))])
                      (range 3))]
       (vsl/noon {:pdf true
                  :play true}
                 (score dur8
                     harmonic-minor
                     (par* lines)
                     (lin _ (transpose c3-))
                     ($by :channel (connect-with
                                    (sf_ (->> (get-in choices [:ornamentation (:connection (first _))])
                                              (score/update-score _))))))))

     '(let [[arpegios ornamentations harmonic-sequences articulations]
           (->> (complementarity-tree 3 3)
                (leaves-paths)
                (filter #(= 3 (count %)))
                (rand/shuffle))

           choices {:harmony {0 _
                              1 [lydian (transpose c4) (s-shift -1)]
                              2 [(transpose c2-)]}
                    :arpegio {0 s0 1 s1 2 s2}
                    :ornamentation {0 void
                                    1 (lin vel0 d1)
                                    2 (lin d1- vel0)}
                    :instruments {0 [(vsl/instrument :chamber-violins-1) o1]
                                  1 [(vsl/instrument :chamber-violas)]
                                  2 [(vsl/instrument :chamber-cellos) o1-]}
                    :articulations {0 (vsl/patch :pizzicato)
                                    1 (vsl/patch :pizzicato)
                                    2 (vsl/patch :pizzicato)}}

           degrees (mapcat (fn [s]
                             (map (choices :harmony) s))
                           harmonic-sequences)
           lines (map (fn [offset]
                        [(get-in choices [:instruments offset])
                         (lin* degrees)
                         (each (tup* (map (fn [d a c p]
                                            [d (tup* (map (fn [step orn p]
                                                            [(one-of vel3 vel6 vel9)
                                                             (get-in choices [:articulations p])
                                                             (get-in choices [:arpegio step])
                                                             {:connection orn}])
                                                          a c p))])
                                          degrees
                                          (drop offset (cycle arpegios))
                                          (drop (* 2 offset) (cycle (concat ornamentations arpegios)))
                                          (drop offset (cycle articulations)))))])
                      (range 3))]
       (vsl/noon {:pdf true
                  :play true}
                 (score dur8
                     dur2
                     dorian
                     (par* (cons bass lines))
                     (lin _ (transpose c3-))
                     ($by :channel (connect-with
                                    (sf_ (->> (get-in choices [:ornamentation (:connection (first _))])
                                              (score/update-score _))))))))


    ]]

   ["Tunes"

    ;; Trying to make music on top of some known jazz standards

    ["Autumn leaves"

     ;; Simple experiment on the first part of autumn leaves grid:

     (play {:title "Autumn Leaves"}

           vel3
           [tetrad
            (lin II V I IV VII [III phrygian3] [VI (lin [melodic-minor sixth] phrygian3)])
            (h/align-contexts :s)
            (dup 2)]

           (h/grid-zipped
            (nlin 16 (chans [(patch :acoustic-bass)
                             o1- t-round]

                            [(patch :vibraphone)
                             (par s0 s1 s2 s3)]

                            [(patch :electric-piano-1) vel2
                             o2 (par s0 s2 s4) (shuftup s0 s2)]

                            [(patch :whistle) o1 vel5
                             (each [(shuftup s0 s1 s2 s3)
                                    (tup same (one-of s1 s1- s2 s2-))])]))))

    ]

    ["Giant steps (John Coltrane)"

     ;; An experiment using Giant steps harmony.

     ;; Ocarina runs over simplistic bass and piano comping.


     (def GIANT_STEPS
       (let [II [II {:degree :II}]
             V [V {:degree :V}]
             I [I {:degree :I}]
             t1 same
             t2 (transpose c4-)
             t3 (transpose c4)
             s1 (lin [t1 I] [t2 (lin V I)] [t3 (lin V [dur2 I])] [t2 (lin II V)])
             II-V-I (lin II V [I dur2])]
         [tetrad
          (tup s1
               [t2 s1]
               [t3 I dur2] [t2 II-V-I] II-V-I [t3 II-V-I] [t1 (lin II V)])
          (h/align-contexts :structural :static)]))

     (play vel3
           (h/harmonic-zip
            [GIANT_STEPS (dupt 2)]
            (chans
             [(patch :acoustic-bass) o2- (each t-round)]
             [(patch :electric-piano-1) (each (par s0 s1 s2 s3))]
             [(patch :ocarina)
              vel5
              (each (parts {:degree :II} (structure [0 3 4 6])
                        {:degree :V} (structure [1 2 5 6])
                        {:degree :I} (structure :tetrad)))
              (ntup 32 [(one-of o1 o2)
                        (! (rup (rand/rand-nth [5 6 7]) s1))
                        (tup (maybe (m/permutation 1/4))
                             [(maybe rev) (one-of s1 s2 s2- s1-)])])]))
           m/connect-repetitions
           (adjust 32))

    ]

    ["ESP (Wayne Shorter)"

     ;; first try:

     (play

       {:title "ESP"
        :composer "Wayne Shorter"}

       (h/harmonic-zip
        ;; grid
        [tetrad
         (tup [VII superlocrian dur2] [I lydian dur2]
              [VII superlocrian dur2] [VIIb lydian dur2]
              [VI superlocrian] [VIIb lydian] [VII superlocrian] (tup [I lydian] [VIIb lydianb7])
              [VI dorian] [II lydianb7] [II dorian] [IIb lydianb7])
         (h/align-contexts :s)
         (dupt 2)]

        ;; parts
        [vel4
         (chans [(patch :acoustic-bass) o2-
                 t-round]

                [(patch :electric-piano-1) vel3 o1-
                 (par> d0 d3 d3 d3 d3)]

                [(patch :flute) vel6
                 (fill> (/ 1 (* 2 32 6)) (any-that (within-pitch-bounds? :C0 :C3) d4- d3- d1- d1 d3 d4))])])

       ;; repeat
       (adjust 32)
       (dup 2))

     ;; full grid:

     (def ESP_fullgrid
       (let [common (lin [VII superlocrian dur2] [I lydian dur2]
                         [VII superlocrian dur2] [VIIb lydian dur2]
                         [VI superlocrian] [VIIb lydian] [VII superlocrian] (tup [I lydian] [VIIb lydianb7]))]
         (tup common
              (lin [VI dorian] [II lydianb7] [II dorian] [IIb lydianb7])
              common
              (lin [VIb lydianb7] [II dorian] (tup [VIb dorian] [IIb lydianb7]) I))))

     (play
       (h/harmonic-zip
        [ESP_fullgrid
         (dupt 2)
         (h/align-contexts :s)]

        (chans

         [(patch :electric-piano-1) o1- vel3
          (voices> d0 d3 d3 d3 d3)]

         [(patch :acoustic-bass) vel2 C-2 t-round]

         [(patch :flute)
          (fill> (/ 1 (* 6 64))
                 (maybe
                  (any-that* (within-pitch-bounds? :G-1 :C2)
                             [d4- d3- d1- d1 d3 d4])))
          (each (probs {void 1
                        same 5}))
          m/connect-repetitions
          (vel-humanize 10 [30 70])]))

       (adjust 48))

    ]

    ["Cyclic episode (Sam Rivers)"
     ;; One more shredding experiment

     (def CYCLIC_EPISODE
       (let [a1 [dorian (rep 4 (transpose c3))]
             a2 [dorian (rep 4 (transpose c3-))]
             b (lin [IV dorian] [V superlocrian (structure [2 3 5 6])])
             c (lin [V mixolydian sus47] [V phrygian sus27])
             d [dorian (append (transpose c3))]]
         [tetrad
          (tup [(root :Bb) a1]
               [(root :G) b] [(root :D) b]
               [(root :D) a2]
               [(root :G) c] [(root :Eb) d])
          (dupt 4)
          (h/align-contexts :s :static)]))

     (let [n-bars (* 4 16)

           bass [(patch :acoustic-bass) (each t2-)]
           vibe [(patch :vibraphone) vel5 t1 (each (par s0 s1 s2 s3)) h/voice-led]

           ;; alternate leads

           _lead1 (ntup> (* n-bars 12)
                        (any-that (within-pitch-bounds? :C0 :C3)
                                  d1 d1- d3 d3- d4 d4-))

           _lead2 [(repeat-while (within-time-bounds? 0 (* n-bars 10))
                    (append [start-from-last
                             (any-that (within-pitch-bounds? :C-1 :C2)
                                       (rep 3 d3 :skip-first)
                                       (rep 3 d3- :skip-first)
                                       d1 d1-)]))
                  (adjust 1)]

           lead4 [(tup (mixtup s0 s1 s2 s3)
                       (mixtup s2 s3 s4 s5))
                  (rup n-bars
                       (probs {(m/permutation [0 1/2]) 2
                               (m/rotation :rand) 3
                               rev 1
                               (any-that* (within-pitch-bounds? :C0 :C3)
                                          (map s-step (range -2 3))) 5}))]]

       (play CYCLIC_EPISODE
             (chans bass
                    vibe
                    [(h/grid-zipped lead4)
                     (chans [(patch :flute) vel8 s2]
                            [(patch :electric-piano-1) vel5])
                     (each (probs {vel0 1
                                same 2}))])
             (vel-humanize 0.15)
             (adjust 64)))

    ]

    ["Inner urge (Joe Henderson)"


     (defn last-n-positions
                "Limit the score to the n latest positions found."
                [n]
                (sf_ (let [_ (->> (group-by :position _)
                                  seq (sort-by key)
                                  reverse (take n)
                                  (map second) (reduce into #{}))]
                       (score/update-score _ (start-from (score/score-origin _))))))

     (let [n-bars 24
           choir [(patch :choir-aahs) vel5 (par> d3 d3 d3)]
           bass [(patch :acoustic-bass) C-2 t-round]
           lead-line (any-that (within-pitch-bounds? :C0 :C3)
                               (rep 2 d3 :skip-first)
                               (rep 2 d3- :skip-first)
                               d4 d4-
                               d1 d1-
                               (rep 3 d2 :skip-first)
                               (rep 3 d2- :skip-first))]
       (play (h/harmonic-zip
              [(tup (lin (nlin 4 [(root :F#) locrian2])
                         (nlin 4 [(root :F) lydian])
                         (nlin 4 [(root :Eb) lydian])
                         (nlin 4 [(root :Db) lydian]))
                    [lydian
                     (lin* (map root [:E :Db :D :B :C :A :Bb :G]))])
               (h/align-contexts :s)
               (dupt 4)]
              (tup (chans choir
                          bass
                          [(patch :music-box)
                           vel5 C1
                           (m/simple-tupline (* n-bars 10) lead-line)])
                   (chans choir
                          bass
                          [(patch :ocarina)
                           vel4 C1
                           (m/simple-tupline (* n-bars 24) lead-line)])
                   (chans choir
                          bass
                          [(patch :sawtooth)
                           (dur (/ 1 n-bars))
                           vel4 C1
                           (tup d0 d3 d6)
                           (tup d0 d4 d8)
                           (m/line (one-of (last-n-positions 10) (last-n-positions 7))
                                   (any-that (within-pitch-bounds? :C0 :C3)
                                             (m/permutation {:grade 3})
                                             #_(one-of (m/contour :rotation {:layer :d})
                                                       (m/contour :mirror {:layer :d})
                                                       (m/contour :similar {:delta 0 :layer :d}))
                                             (one-of d1 d1-)
                                             (one-of d2 d2-))
                                   (sf_ (> (score/score-duration _) 1))
                                   (trim 0 1))
                           (vel-humanize 5 [40 80])])
                   (chans [choir
                           (ntup (/ n-bars 2) same)
                           ($by :position [(! (one-of (r/gen-tup 8 3 :euclidean)
                                                      (r/gen-tup 8 3 :durations [2 3 4 5])))
                                           (sf_ (let [xs (-> (group-by :position _) seq sort vals)]
                                                  (reduce into #{} (map score/update-score xs (rand/shuffle [d0 d1 d1-])))))])]
                          bass)))
             (adjust 180)))



    ]]

   ["Snippets"

    ["textures 1"

     ;; Trying to produce vibrating textures by playing very fast note sequences.

     (play dur2
           lydian
           (patch :flute)
           (chans _ d3 d6 d9)
           (each [(dupt 24) (each (one-of vel1 vel3 vel6)
                            (probs {_ 6 d1 1}))])
           ($by :channel (maybe rev))
           (append (transpose c3-))
           (append (transpose c1-)))


     (play dur3
           lydian
           (chans [(patch :marimba) (lin _ c1)]
                  [(patch :vibraphone) (lin d3 d2)]
                  [(patch :celesta) (lin d6 d6)]
                  [(patch :orchestral-harp) (lin d9 d9)])
           (append (transpose c2-))
           (dup 2)

           (each [(dupt 34)
               (each (one-of vel0 vel3 vel6 vel9)
                  (probs {_ 4 o1 1}))]))


     (play dur8
           o2
           (dupt 128)
           (each (par> d4 d4 d4)
              (one-of vel0 vel1 vel2 vel3 vel4 vel5)))

    ]

    ["Sparkling waves"

     (play dur:4
           vel4
           (scale :lydian)
           (patch :music-box)
           (par s0 s2 s4)
           (rep 3 (each [{:mark (rand/rand)} s1 {:velocity (div 1.1) :duration (mul 1.3)} (shuftup s2- s0 s2)])
                :skip-first)
           (lin I [rev III] [o1- V] [rev o1- VII])
           (append [rev (transpose c3)]))

    ]

    ["infinite climb"

     (play dur6 dur2
           (patch :ocarina)
           (rup 36 c1)
           (sf_ (set (map-indexed (fn [i n] (let [vel (* 60 2 (/ (inc i) (count _)))
                                                 vel (if (> vel 60) (- 60 (- vel 60)) vel)]
                                             (assoc n :velocity vel)))
                                  (sort-by :position _))))
           (par _
                (m/rotation 1/3)
                (m/rotation 2/3))
           (dup 4))

    ]

    ["violin fast arpegio"

     (play (dur 3/2)
           dorian
           (patch :violin)
           (lin I IV V I)
           (h/align-contexts :s)
           (each (ntup 2 (tup s0 s2 s4 s4 s2 s0)))
           (each (! (vel (mul (+ 0.9 (* (rand/rand) 0.2))))))
           (append s1-))

    ]

    ["zip rythmn"

     (play lydianb7
           (h/modal-structure 5)
           (chans
            [(patch :vibraphone)
             (shuflin s0 s1 s2 s3 s4)
             (nlin 4 (one-of s1 s2 s1- s2-))
             (sf_ (let [rythmn (score (nlin 2 (! (r/gen-tup 12 5 :shifted))) (append rev))]
                    (set (map (fn [r n]
                                (merge n (select-keys r [:position :duration])))
                              (sort-by :position rythmn)
                              (sort-by :position _)))))]
            [(patch :woodblock) (r/gen-tup 12 5 :euclidean) (dup 4)]
            [(patch :tinkle-bell) (dup 4)]
            [(patch :metallic) (shuflin s0 s1 s2 s3) (each (par s0 s1 s2))]
            [(patch :acoustic-bass) t2- (dup 4)])
           (adjust 8)
           (append [(transpose c3-) s1 rev] _))

    ]

    ["Gradual melodic transformation"

     ;; Random harmonic seq using IV II and VI degrees on vibraphone, ocarina melody derives using transposition, rotation and permutation.

     (play (chans

            [(patch :vibraphone)
             vel3
             (ntup 4 [(one-of IV II VI) tetrad (par [t2- vel5] s0 s1 s2 s3)])]

            [(patch :ocarina)
             vel5
             (shuftup d1 d2 d3 d4 d5)
             (each (maybe (par d0 d3)))
             (rup 16
                  (probs {(m/permutation :rand) 1
                          (m/rotation :rand) 3
                          (one-of* (map d-step (range -3 4))) 5}))])

           (adjust 10)
           (append [d2- (transpose c3)]
                   [d2 (transpose c3-)]
                   same))

    ]

    ["Bach prelude Cm melodic pattern"

     (play harmonic-minor
           (m/$lin (lin I IV I V))
           (h/align-contexts :s)
           (lin _ s1)
           (each (chans (tup s2 [s1 (lin _ d1- _)] s0 [s1 (lin _ d1- _)])
                        (tup s3- [s2- (lin _ d1 _)] s1- [s2- (lin _ d1 _)])))
           (lin _ [(transpose c3) rev])
           (dup 2))

     (play harmonic-minor
           (m/$lin (lin I IV I V))
           (h/align-contexts :s)
           (lin _ s1)
           (let [pat1 (tup s2 [s1 (lin _ d1- _)] s0 [s1 (lin _ d1- _)])
                 pat2 [pat1 (m/contour :mirror {:layer :s})]]
             (each (chans [o1 pat1]
                          [s1- pat2]))))


     (play harmonic-minor
           dur2
           (lin _ (transpose c3) _)
           (m/$lin (lin I IV I V))
           (h/align-contexts :s)
           (let [br (lin _ (one-of d1 d1-) _)
                 pat1 (one-of (tup s2 [s1 br] s0 [s1 br])
                              (tup [s1 br] s2 [s1 br] s0)
                              (tup s0 [s1 br] s2 [s1 br])
                              (tup [s1 br] s0 [s1 br] s2))
                 pat2 (one-of (tup s3- [s2- br] s1- [s2- br])
                              (tup s1- [s2- br] s3- [s2- br]))]
             (each (chans [o1 (patch :ocarina) vel8 pat1]
                          [(patch :vibraphone) pat2])))
           (dup 2))

    ]

    ["Modal chords"

     (let [rand-color (fn []
                        (let [k (rand/rand-nth [:lydian+ :lydian :ionian :dorian
                                              :melodic-minor :mixolydian :phrygian6])]
                          [(scale k)
                           (h/modal-structure 4)]))]
       (play dur2
             (lin* (map (comp transpose c-step) (rand/shuffle (range 12))))
             (each (! (rand-color)))
             (h/align-contexts :d :static)
             (chans [(patch :aahs) (each (par s0 s2 s3 s5)) #_h/voice-led]
                    [(patch :vibraphone) o1 (each (par s0 s2 s3) (shuftup s0 s3) (tup s0 s1 s1-))
                     ($by :position (probs {vel0 2
                                            (one-of vel3 vel5 vel7) 8
                                            [vel3 (ntup> 4 [s1 (vel+ 15)])] 1}))]
                    [(patch :acoustic-bass) o1- t-round])))



     (defn possible-modes
       "given a chromatic degree (int between 0 an 11)
        return possible modes"
       [cd modal-lvl least-priority]
       (let [modes (constants/lvl->mode->degree-priority modal-lvl)
             candidates (filter (fn [[_ s]] (-> (take least-priority s)
                                               (set) (contains? cd)))
                                modes)]
         candidates))

     (play (patch :aahs)
           dur4
           (shuflin c0 c1 c2 c3)
           (m/contour :similar {:delta 4 :layer :c})
           (par o1 [c6- (m/contour :mirror {:layer :c})])
           ($by :position
                (sfn score
                     (let [modal-lvl 1
                           chord-size 4
                           [min-pitch-val max-pitch-val] (h/pitch-values score)
                           interval (mod (- max-pitch-val min-pitch-val) 12)
                           [mode-kw prio] (rand/rand-nth (possible-modes interval modal-lvl (dec chord-size)))
                           partial-scale (cons 0 (take (dec chord-size) prio))
                           structure' (constants/partial-scale->structure mode-kw partial-scale)
                           closed (score/score (dissoc (first score) :pitch)
                                      (origin min-pitch-val)
                                      (scale mode-kw)
                                      (structure structure')
                                      (par* (map s-step (range chord-size))))
                           drops (filter (fn [drop] (= max-pitch-val (last (h/pitch-values drop))))
                                         (h/drops closed))]
                       (rand/rand-nth drops))))
           ($by :position (chans _
                                 [(patch :contrabass) vel3 min-pitch o1-]
                                 [max-pitch
                                  (patch :ocarina)
                                  (mixtup s0 s1- s2- s3- s4- s5-)
                                  (tup _ s2- s1)
                                  #_(each (probs {_ 4 (tup _ [vel4 (maybe s2- s3-)]) 1}))]))
           (lin _ [rev c3])
           (lin _ [rev c3-])
           (out/options :bpm 30 :xml true :preview true))

    ]

    ["melodic development"

     (play

       ;; use dorian mode
       dorian

       ;; random tup holding the 7 degrees of the scale
       (shuftup d0 d1 d2 d3 d4 d5 d6)

       ;; iterating while within time span 0-8
       (repeat-while (within-time-bounds? 0 8)
                     ;; each time we will append to the current score
                     (append
                      ;; taking care to stay within pitch bounds
                      (any-that (within-pitch-bounds? :C0 :C3)
                                ;; those 4 expression are randomly picked (if they respect the pitch bound condition)
                                ;; each one is operating on the nth last elements of the melody using the
                                ;; `noon.score/start-from-nth-last` update builder.
                                [(start-from-nth-last 1) (one-of d1- d1)]
                                ;; takes a random permutation in the 0-1/4 complexity range
                                [(start-from-nth-last 8) (m/permutation [0 1/4])]
                                ;; reverse the 4 last four notes of the score
                                [(start-from-nth-last 4) rev]
                                ;; change the contour of the last four notes
                                [(start-from-nth-last 4) (m/contour :similar {:extent [-2 2] :layer :d})]))
                     ;; at the end of the loop we adjust the score duration
                     (trim 0 8))

       ;; each note will have some chance to spawn some extra voice (vibraphone or flute)
       ;; or a velocity humanisation
       (each
        (probs {(one-of vel3 vel5 vel7 vel9) 6
                (superpose [(chan 2) (patch :vibraphone) vel8 (one-of d3 d4)]) 1
                (superpose [(chan 7) (patch :flute) vel8 o1]) 5
                }))

       ;; adding a simple bass
       (superpose (k (nlin 4 [(chan 5) (patch :acoustic-bass) t2- vel8 dur2])))

       ;; a randomized harmonic modulation sequence
       (rep 4 (one-of [(d-shift -2) (transpose c3)]
                      [(d-shift 2) (transpose c3-)]
                      [(d-shift 1) (transpose c1-)]
                      [(d-shift -3) (transpose c6)]))

       ;; the whole score is repeated superposing a taiko drum
       (append (superpose (k (nlin 4 [(patch :taiko-drum) (chan 3) (! [vel4 (maybe o1- d1) (r/gen-tup 7 3)])])
                             (dup 8)))))


    ]]

   ["Usage"

    ["=noon.lib.rythmn/bintup="

     ;; An experiment around =noon.lib.rythmn/gen-bintup=
     ;; The =gen-bintup= function is used to produce a bass line and a fast rythmic texture alternating between electric-piano and marimba.

     (play dur6
           (lin [I dorian]
                [III mixolydian]
                [VIb lydian]
                [I lydian])
           (append> (transpose c1-) (transpose c1-) (transpose c1-))
           (dup 2)
           (h/align-contexts)
           (each (chans [(patch :new-age) vel3  o1- (par s0 s1 s2 s3 [o1 (par> d3 d3 d3 d3)])]
                        [(patch :taiko-drum) (r/gen-tup 9 3 :durations [2 3 4]) (each (one-of vel4 vel3) (maybe d3 d3-))]
                        [(patch :acoustic-bass)
                         t-floor o1-
                         (r/gen-bintup 9 4 :euclidean :shifted)
                         vel4 (vel-humanize 1/5)
                         (parts {:bintup 0} (each (vel+ 20) (one-of s0 s1))
                                {:bintup 1} (each (probs {vel0 2 (one-of d3- d4) 1})))]
                        [(r/gen-bintup 54 11  :shifted :euclidean)
                         (parts {:bintup 0} [(patch :electric-piano-1)
                                             sus4
                                             (each vel3
                                                   (vel-humanize 1/10)
                                                   (one-of d2 d4 d6)
                                                   (probs {_ 3 [(one-of s0 s1 s2) (par s0 s1 s2)] 1}))]
                                {:bintup 1} [(patch :marimba)
                                             vel4
                                             (vel-humanize 1/5)
                                             (chan+ 1)
                                             (each [(one-of d3 d5 d7) (maybe o1 (par _ d4))])])])))

    ]

    ["=noon.lib.harmony/grid="

     (play dur3
           (nlin> 48 (one-of d1 d1-))
           (each (chans [(patch :aahs) vel5 (par s0 s1 s2 s3)]
                        [(patch :ocarina) (shuftup s0 s2 s4 s6) (shuftup d0 d3 d6) (tup _ rev)]
                        [(patch :acoustic-bass) t2-]))
           (h/grid dur3 tetrad
                   (lin [I lydian (structure [2 3 5 6])]
                        [IIb dorian (structure [1 2 3 6])]
                        [V mixolydian (structure [2 3 5 6])]
                        [Vb melodic-minor (structure [1 2 5 6])])
                   (rep 6 (transpose c2-))
                   (dup 2)
                   (h/align-contexts :d :static)))

     (play (ntup> 24 (one-of d1 d1-))
           (each (chans [(patch :aahs) vel5 (par s0 s1 s2 s3)]
                        [(patch :ocarina)
                         (one-of (mixtup s0 s2 s4 s6)
                                 (mixtup s0 s2 s4 s6))
                         (one-of (mixtup d0 d3 d6)
                                 (mixtup d0 d3 d6))
                         (vel-humanize 10 [40 80])
                         (tup _ rev)]
                        [(patch :acoustic-bass) t2-]))
           (h/grid tetrad
                   (tup [I lydian]
                        [IIb dorian]
                        [V mixolydian]
                        [Vb melodic-minor])
                   (each (h/modal-structure 4))
                   (rup 4 (transpose c2-))
                   (dupt 2)
                   (h/align-contexts :d :static))
           (adjust 60))

     (play (chans [(patch :aahs) vel5 (par s0 s1 s2 s3)]
                  [(patch :acoustic-bass) t2-])
           (h/grid (lin [I lydian (structure [2 3 5 6])]
                        [IIb dorian (structure [1 2 3 6])]
                        [V mixolydian (structure [2 3 5 6])]
                        [Vb melodic-minor (structure [1 2 5 6])])
                   (rep 2 (transpose c2-))
                   (dup 2)
                   (h/align-contexts :d :static)
                   (adjust 1))
           (parts (patch :acoustic-bass)
                  (each (tup (maybe o1) (one-of d4 d3-))))
           (adjust 32))

     (play (chans [(patch :aahs)
                   vel6
                   (rup 24 (any-that (within-pitch-bounds? :G-1 :G1)
                                     s2 s2- s3 s3-))
                   (each (par s0 s1 s2 s3))]
                  [(patch :acoustic-bass) t2-])
           (h/grid tetrad
                   (lin [I lydian (structure [2 3 5 6])]
                        [IIb dorian (structure [1 2 3 6])]
                        [V mixolydian (structure [2 3 5 6])]
                        [Vb melodic-minor (structure [1 2 5 6])])
                   (rep 2 (transpose c2-))
                   (dup 2)
                   (h/align-contexts :d :static)
                   (adjust 1))
           (parts (patch :acoustic-bass)
                  (each (tup (maybe o1) (one-of d4 d3-))))
           (adjust 32))

     (play (rup 128 (any-that (within-pitch-bounds? :C1 :C3)
                              s1 s2 s3 s1- s2- s3-))
           (chans (each (probs {_ 2
                                vel0 1
                                (shuftup s1- s0 s1 s2) 1}))
                  (each s1- (probs {_ 2
                                    vel0 1
                                    (shuftup s1- s0 s1) 1}))
                  (each [s2- o1- (probs {_ 2
                                         (shuftup s0 s2) 1})]))
           (h/grid harmonic-minor
                   (tup I IV VII I [IV melodic-minor VII] IV [V harmonic-minor VII] VII)
                   (dupt 4)
                   (h/align-contexts :s))
           (adjust {:duration 64}))

    ]

    ["Vienna symphonic library"

     ;; Using the awesome Vienna Symphonic Library for playback.
     ;; The Vienna Ensemble setup file is provided =vsl/setup1.vep64=
     ;; It assumes VSL special edition (volumes 1+ and 2+)

     (vsl/noon {:play true}
               (score (par [(vsl :chamber-violins-1 :detache)
                         (lin s0
                              [(vsl/patch :legato) (tup s1 s2 s3)]
                              [(vsl/patch :pizzicato)
                               (par [(vsl/patch :snap-pizzicato) _]
                                    [(vsl :solo-double-bass :pizzicato) o2- (tup s2 s1)])])]
                        [(vsl :flute-1 :portato) o1 s- (lin s0 [(vsl/patch :legato) (tup s1 s2 s3)])])
                   (lin s0 s2 s1-)
                   (dup 4)))

     (vsl/noon {:play true}
               (score vel10
                   (vsl/instrument :chamber-cellos)
                   (vsl/patch :pizzicato)
                   o1-
                   (shuftup d0 d3 d6)
                   (shuftup d0 d3 d6)
                   (dup 8)))

    ]

    ["the end"]]]]