package com.phonepe.intent.sdk.api;

import android.content.Context;
import android.content.Intent;
import android.content.pm.ApplicationInfo;
import android.content.pm.PackageManager;
import android.content.pm.ResolveInfo;
import android.os.Handler;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;

import com.phonepe.intent.sdk.BuildConfig;
import com.phonepe.intent.sdk.comunication.SDKtoAppConnection;
import com.phonepe.intent.sdk.contracts.iDeviceIdListener;
import com.phonepe.intent.sdk.core.DataConfig;
import com.phonepe.intent.sdk.core.ObjectFactory;
import com.phonepe.intent.sdk.models.Event;
import com.phonepe.intent.sdk.models.IntentSDKConfig;
import com.phonepe.intent.sdk.models.SDKConfig;
import com.phonepe.intent.sdk.networking.NetworkUtils;
import com.phonepe.intent.sdk.ui.PreCacheService;
import com.phonepe.intent.sdk.ui.TransactionActivity;
import com.phonepe.intent.sdk.utils.AnalyticsManager;
import com.phonepe.intent.sdk.utils.Config;
import com.phonepe.intent.sdk.utils.Constants;
import com.phonepe.intent.sdk.utils.DeviceIdGenerator;
import com.phonepe.intent.sdk.utils.SdkLogger;
import com.phonepe.intent.sdk.utils.Utils;

import org.json.JSONObject;

import java.util.List;
import java.util.concurrent.atomic.AtomicBoolean;

import static com.phonepe.intent.sdk.utils.AnalyticsManager.EventConstants.ERROR_MESSAGE;
import static com.phonepe.intent.sdk.utils.AnalyticsManager.Events.SDK_INITIALIZATION_FAILED;
import static com.phonepe.intent.sdk.utils.Constants.GenericConstants.PHONEPE_APP_NOT_INSTALLED;
import static com.phonepe.intent.sdk.utils.Constants.GenericConstants.SUCCESS;
import static com.phonepe.intent.sdk.utils.Constants.GenericConstants.UPI_NOT_REGISTERED;

/**
 * @author TheEternalWitness
 * @since 20/03/18.
 */
public class PhonePe {

    public static final String TAG = "PhonePe";
    private static ObjectFactory objectFactory;
    private AnalyticsManager analyticsManager;

    //region private members

    private PhonePe(Context applicationContext) {

        cacheMerchantMeta(applicationContext);

        SdkLogger.v(TAG, "checking for pre-caching service can be started or not ..");
        try {
            if (Utils.isTrue(getObjectFactory().<Boolean>get(Constants.MerchantMeta.PRE_CACHE_ENABLED))
                    && Utils.isBelowAndroidOreo()) {

                SdkLogger.d(TAG, "starting pre caching service ...");
                Intent intent = new Intent(applicationContext, PreCacheService.class);
                intent.putExtra(Constants.BundleConstants.DATA_FACTORY, getObjectFactory());
                applicationContext.startService(intent);
            }
        } catch (PhonePeInitException e) {
            SdkLogger.e(TAG, e.getMessage(), e);
        }
    }

    private static PhonePe getPhonePeInstance(Context context) {

        SdkLogger.init(context);
        SdkLogger.v(TAG, "PhonePe SDK initializing ...");
        PhonePe.objectFactory = new ObjectFactory(context);
        PhonePe phonePe = new PhonePe(context);
        try {
            phonePe.analyticsManager = getObjectFactory().<AnalyticsManager>get(AnalyticsManager.class);
            getObjectFactory().cache(PhonePe.class.getCanonicalName(), phonePe);
            NetworkUtils.Companion.syncSDKConfig();
        } catch (PhonePeInitException e) {
            SdkLogger.e(TAG, e.getMessage(), e);
        }
        SdkLogger.v(TAG, "PhonePe SDK initialized");
        return phonePe;

    }

    private static Intent getTransactionIntentInternal(@NonNull final TransactionRequest transactionRequest) throws PhonePeInitException {

        SdkLogger.v(TAG, String.format("transactionRequest {%s} is submitted", transactionRequest.toString()));
        return TransactionActivity.getIntent(getObjectFactory().getApplicationContext(), transactionRequest, getObjectFactory());
    }


    private void cacheMerchantMeta(Context context) {

        try {

            SdkLogger.v(TAG, "trying to get app metadata ...");
            ApplicationInfo applicationInfo = context.getPackageManager().getApplicationInfo(context.getPackageName(), PackageManager.GET_META_DATA);

            if (applicationInfo != null && applicationInfo.metaData != null) {
                try {
                    getObjectFactory().cache(Constants.MerchantMeta.MERCHANT_ID, applicationInfo.metaData.getString(Constants.MerchantMeta.MERCHANT_ID));
                    getObjectFactory().cache(Constants.MerchantMeta.APP_ID, applicationInfo.metaData.getString(Constants.MerchantMeta.APP_ID));
                    getObjectFactory().cache(Constants.MerchantMeta.DEBUGGABLE, applicationInfo.metaData.getBoolean(Constants.MerchantMeta.DEBUGGABLE));
                    getObjectFactory().cache(Constants.MerchantMeta.IS_UAT, applicationInfo.metaData.getBoolean(Constants.MerchantMeta.IS_UAT));
                    getObjectFactory().cache(Constants.MerchantMeta.PRE_CACHE_ENABLED, applicationInfo.metaData.getBoolean(Constants.MerchantMeta.PRE_CACHE_ENABLED));
                    getObjectFactory().cache(Constants.MerchantMeta.JUSPAY_ENABLED, applicationInfo.metaData.getBoolean(Constants.MerchantMeta.JUSPAY_ENABLED));
                } catch (PhonePeInitException e) {
                    SdkLogger.e(TAG, e.getMessage(), e);
                }

                SdkLogger.v(TAG, "app metadata found");
            } else {

                String errorMessage = "failed to get application applicationInfo or applicationMetaData";
                SdkLogger.wtf(TAG, errorMessage);
                sendEvent(SDK_INITIALIZATION_FAILED, errorMessage);

                throw new IllegalArgumentException(errorMessage);
            }
        } catch (PackageManager.NameNotFoundException e) {

            SdkLogger.e(TAG, String.format("failed to cache merchant meta with exception message = {%s}. failed to initialized SDK", e.getMessage()));

            sendEvent(SDK_INITIALIZATION_FAILED, e.getMessage());

            throw new IllegalArgumentException(Constants.ExceptionMessage.PHONE_PE_INIT_EXCEPTION);
        }
    }

    private void sendEvent(String eventName, String errorMessage) {

        Event event = analyticsManager.getEvent(eventName)
                .putInEventBody(ERROR_MESSAGE, errorMessage);
        analyticsManager.submit(event);
    }

    //endregion

    //region public members

    public static void init(Context context) {
        getPhonePeInstance(context);
//        try {
//            isUPIAccountRegistered();
//        } catch (Exception e) {
//
//        }
    }

    /**
     * API to get intent which can handle Debit request.
     * Using this intent you should be calling Activity#startActivityForResult to initiate the debit flow.
     * See documentation regarding interpreting result.
     * https://github.com/PhonePe/MerchantAndroidSDKDemo/wiki/API-Usage
     *
     * @param transactionRequest Instance of {@link TransactionRequest} which carries information about debit request.
     * @return Intent to complete the debit transaction.
     */
    @Nullable
    public static Intent getTransactionIntent(@NonNull final TransactionRequest transactionRequest) throws PhonePeInitException {
        return getTransactionIntentInternal(transactionRequest);
    }

    @Nullable
    public static Intent getImplicitIntent(Context context, @NonNull final TransactionRequest transactionRequest) throws PhonePeInitException {
        return TransactionActivity.getImplicit(context, transactionRequest, getObjectFactory());
    }

    public static void logout() throws PhonePeInitException {
        getObjectFactory().<DataConfig>get(DataConfig.class).clearAllData();
        getObjectFactory().flushCache();
        getPhonePeInstance(getObjectFactory().getApplicationContext());
    }

    public static ObjectFactory getObjectFactory() throws PhonePeInitException {

        if (PhonePe.objectFactory == null) {
            throw new PhonePeInitException("SDK is not initialized");
        }

        return PhonePe.objectFactory;

    }

    /**
     * Tells if logging is enabled or not
     *
     * @return <code>true</code> if debuggable or <code>false</code> otherwise
     */
    public boolean isDebuggable() {

        try {
            return Utils.isTrue(getObjectFactory().<Boolean>get(Constants.MerchantMeta.DEBUGGABLE));
        } catch (PhonePeInitException e) {
            SdkLogger.e(TAG, e.getMessage(), e);
        }
        return false;
    }

    /**
     * Tells if we are in UAT mode
     *
     * @return <code>true</code> if in UAT mode or <code>false</code> otherwise
     */
    public boolean isInUATMode() {
        try {
            return Utils.isTrue(getObjectFactory().<Boolean>get(Constants.MerchantMeta.IS_UAT));
        } catch (PhonePeInitException e) {
            SdkLogger.e(TAG, e.getMessage(), e);
        }
        return false;
    }

    public static String getSDKVersion() {
        return BuildConfig.SDK_VERSION;
    }

    public static void getDeviceId(Context context, iDeviceIdListener deviceIdListener) throws PhonePeInitException {
        getObjectFactory().<DeviceIdGenerator>get(DeviceIdGenerator.class).generateDeviceId(context, true, deviceIdListener);
    }

    public static boolean isAppInstalled() throws PhonePeInitException {
        return Utils.isPhonePeAppInstalled(getObjectFactory());
    }

    public static void isUPIAccountRegistered(final ShowPhonePeCallback callback) throws PhonePeInitException {
        SdkLogger.d("UPIRegistered", "Checking");
        if (isAppInstalled()) {
            try {
                final ObjectFactory objectFactory = getObjectFactory();
                shouldShow(new ShowPhonePeCallback() {
                    @Override
                    public void onResponse(boolean show) {
                        try {
                            boolean isRegistered = false;
                            isRegistered = false;
                            Intent upiUriIntent = new Intent();
                            upiUriIntent.setData(objectFactory.<Config>get(Config.class).getUpiUri());
                            List<ResolveInfo> resolveInfoList = Utils.getResolveInfo(objectFactory, upiUriIntent);
                            String phonepePackageName = Utils.getPhonePePackageName(objectFactory);
                            for (ResolveInfo resolveInfo : resolveInfoList) {
                                String packageName = resolveInfo.activityInfo.packageName;
                                if (packageName != null && !packageName.isEmpty() && phonepePackageName.matches(packageName)) {
                                    isRegistered = true;
                                }
                            }
                            SdkLogger.d("UPIRegistered", "Sending Response "+show+" "+isRegistered);
                            callback.onResponse(isRegistered && show);
                        } catch (Exception e) {
                            SdkLogger.d("UPIRegistered", "Sending Response "+e.getMessage());
                            callback.onResponse(false);
                        }

                    }
                });

            } catch (Exception e) {
                SdkLogger.d("UPIRegistered", "Sending Response "+e.getMessage());
                callback.onResponse(false);
            }
        } else {
            callback.onResponse(false);
        }
    }

    public static void checkAvailability(final AvailabilityCheckRequest availabilityCheckRequest,final CheckPhonePeAvailabilityCallback callback) throws PhonePeInitException {
        SdkLogger.d("checkAvailability", "Checking");
        shouldShowV2(new CheckPhonePeAvailabilityCallback() {
            @Override
            public void onResponse(final boolean isUPIAccountRegistered, @NonNull final String responseCode) {
                SdkLogger.d("checkAvailability", "Received isUPIAccountRegistered Response " + isUPIAccountRegistered + " " + responseCode);
                final AtomicBoolean callBackSent = new AtomicBoolean(false);
                IntentSDKConfig intentSDKConfig = objectFactory.get(IntentSDKConfig.class);
                boolean isAvailabilityCheckEnabled = intentSDKConfig.isAvailabilityCheckEnabled();
                long checkAvailabilityTimeoutMs = intentSDKConfig.getAvailabilityCheckTimeoutMs();
                if (isUPIAccountRegistered && isAvailabilityCheckEnabled) {
                    SdkLogger.d("checkAvailability", "Starting checkAvailability n/w call with timeout value " + checkAvailabilityTimeoutMs + " & enabled status " + isAvailabilityCheckEnabled);
                    startCheckAvailabilityCall(callBackSent, checkAvailabilityTimeoutMs, availabilityCheckRequest, callback);
                } else {
                    sendCallBackResponse(isUPIAccountRegistered, responseCode, callBackSent, callback);
                }
            }
        });
    }

    private static void startCheckAvailabilityCall(final AtomicBoolean callBackSent, final long checkAvailabilityTimeoutMs, final AvailabilityCheckRequest availabilityCheckRequest, final CheckPhonePeAvailabilityCallback callback) {
        new Handler().postDelayed(new Runnable() {
            @Override
            public void run() {
                SdkLogger.d("checkAvailability", "timeout");
                sendCallBackResponse(true, SUCCESS, callBackSent, callback);
            }
        }, checkAvailabilityTimeoutMs);
        NetworkUtils.Companion.checkAvailability(availabilityCheckRequest, new CheckPhonePeAvailabilityInternalCallback() {
            @Override
            public void onSuccess(final boolean isAvailable, @NonNull final String responseCode) {
                SdkLogger.d("checkAvailability", "Received checkAvailability Response " + isAvailable + " " + responseCode);
                sendCallBackResponse(isAvailable, responseCode, callBackSent, callback);
            }

            @Override
            public void onFailure(final boolean isAvailable, @NonNull final String errorCode) {
                //even if the response fails, we allow the payment in this case & send true with the responseCode sent by SDK<->App shouldShowV2
                SdkLogger.d("checkAvailability", "Received checkAvailability Response " + isAvailable + " " + errorCode);
                sendCallBackResponse(true, SUCCESS, callBackSent, callback);
            }
        });
    }

    private static void sendCallBackResponse(boolean response, String responseCode, AtomicBoolean callbackSent, CheckPhonePeAvailabilityCallback checkPhonePeAvailabilityCallback){
        if(!callbackSent.get()) {
            checkPhonePeAvailabilityCallback.onResponse(response,responseCode);
            callbackSent.set(true);
        }
    }

    public static void shouldShow(final ShowPhonePeCallback callback) throws PhonePeInitException {
        if (isAppInstalled()) {
            SDKConfig sdkConfig = getObjectFactory().get(SDKConfig.class);
            if (sdkConfig.shouldShowPhonePeEnabled()) {
                JSONObject constraints = new JSONObject();
                try {
                    constraints.put(SDKtoAppConnection.VALUE_LOGGED_IN, true);
                    constraints.put(SDKtoAppConnection.VALUE_PAYMENTINSTRUMENT, true);
                    constraints.put(SDKtoAppConnection.VALUE_MERCHANT_APP_ID, getObjectFactory().get(Constants.MerchantMeta.APP_ID));
                } catch (Exception e) {
                    SdkLogger.e(TAG, e.getMessage(), e);
                }
                request(SDKtoAppConnection.VALUE_SHOULDSHOWPHONEPE, constraints.toString(), new RequestCallback() {
                    @Override
                    public void onResponse(String result) {
                        SdkLogger.d("UPIRegistered", result);
                        if (callback != null) {
                            if (result == null || result.isEmpty()) {
                                callback.onResponse(true);
                            } else {
                                try {
                                    JSONObject jsonObject = new JSONObject(result);
                                    boolean didPhonePeRespond = jsonObject.optBoolean(SDKtoAppConnection.KEY_PHONEPE_RESPONDED, true);
                                    if (didPhonePeRespond)
                                        callback.onResponse(jsonObject.optBoolean(SDKtoAppConnection.KEY_RESULT, true));
                                    else
                                        callback.onResponse(true);
                                } catch (Exception e) {
                                    callback.onResponse(true);
                                }
                            }
                        }
                    }
                });
            } else {
                callback.onResponse(true);
            }

        } else {
            callback.onResponse(false);
        }
    }

    public static void shouldShowV2(final CheckPhonePeAvailabilityCallback callback) throws PhonePeInitException {
        if (isAppInstalled()) {
            SDKConfig sdkConfig = getObjectFactory().get(SDKConfig.class);
            if (sdkConfig.shouldShowPhonePeEnabled()) {
                JSONObject constraints = new JSONObject();
                try {
                    constraints.put(SDKtoAppConnection.VALUE_LOGGED_IN, true);
                    constraints.put(SDKtoAppConnection.VALUE_PAYMENTINSTRUMENT, true);
                    constraints.put(SDKtoAppConnection.VALUE_MERCHANT_APP_ID, getObjectFactory().get(Constants.MerchantMeta.APP_ID));
                } catch (Exception e) {
                    SdkLogger.e(TAG, e.getMessage(), e);
                }
                request(SDKtoAppConnection.VALUE_SHOULDSHOWPHONEPE, constraints.toString(), new RequestCallback() {
                    @Override
                    public void onResponse(String result) {
                        SdkLogger.d("shouldShowV2", result);
                        if (callback != null) {
                            if (result == null || result.isEmpty()) {
                                callback.onResponse(true, SUCCESS);
                            } else {
                                try {
                                    JSONObject jsonObject = new JSONObject(result);
                                    boolean shouldShowResult = jsonObject.optBoolean(SDKtoAppConnection.KEY_RESULT, true);
                                    String responseCode = shouldShowResult ? SUCCESS : UPI_NOT_REGISTERED;
                                    callback.onResponse(shouldShowResult,responseCode);
                                } catch (Exception e) {
                                    callback.onResponse(true,SUCCESS);
                                }
                            }
                        }
                    }
                });
            } else {
                callback.onResponse(true,SUCCESS);
            }
        } else {
            callback.onResponse(false, PHONEPE_APP_NOT_INSTALLED);
        }
    }

    public static void prime() throws Exception {
        shouldShow(null);
    }

    public static void request(String request, String constraints, RequestCallback callback) {
        ObjectFactory.InitializationBundle initializationBundle = objectFactory.<ObjectFactory.InitializationBundle>get(ObjectFactory.InitializationBundle.class);
        initializationBundle.put(SDKtoAppConnection.KEY_REQUEST, request);
        initializationBundle.put(SDKtoAppConnection.KEY_CONSTRAINTS, constraints);
        initializationBundle.put(SDKtoAppConnection.KEY_CALLBACK, callback);
        objectFactory.get(SDKtoAppConnection.class, initializationBundle);
    }

    //endregion
}