package com.phonepe.intent.sdk.core;

import android.Manifest;
import android.app.Activity;
import android.content.Intent;
import android.content.IntentFilter;
import android.content.pm.PackageManager;
import android.webkit.JavascriptInterface;

import com.phonepe.intent.sdk.bridges.PermissionsHandler;
import com.phonepe.intent.sdk.contracts.SmsListener;
import com.phonepe.intent.sdk.contracts.iActivityDestroyedHook;
import com.phonepe.intent.sdk.contracts.iBridgeCallback;
import com.phonepe.intent.sdk.contracts.iPermissionsReceived;
import com.phonepe.intent.sdk.models.Body;
import com.phonepe.intent.sdk.models.Error;
import com.phonepe.intent.sdk.models.ListenOtpBody;
import com.phonepe.intent.sdk.models.ListenOtpData;
import com.phonepe.intent.sdk.models.PermissionsBody;
import com.phonepe.intent.sdk.models.PermissionsObject;
import com.phonepe.intent.sdk.ui.BaseWebActivity;
import com.phonepe.intent.sdk.utils.Constants;
import com.phonepe.intent.sdk.utils.SdkLogger;

import java.util.regex.Matcher;
import java.util.regex.Pattern;


/**
 * @author Sharath Pandeshwar
 * @since 26/05/17.
 */

public class SMSManager implements SmsListener, iPermissionsReceived, iActivityDestroyedHook, ObjectFactoryInitializationStrategy {

    public static final String TAG = "SMSManager";
    public static final int PERMISSIONS_REQUEST_READ_SMS = 456;
    public static final int PERMISSIONS_REQUEST_SEND_SMS = 457;
    public static final int PERMISSIONS_REQUEST_READ_PHONE_STATE = 458;
    private static final String ACTION = "android.provider.Telephony.SMS_RECEIVED";
    private Activity activity;
    private SmsReceiver smsReadReceiver;

    private String otpRegexPattern;
    private String jsCallback;
    private String context;

    private String jsData;

    private iBridgeCallback bridgeCallback;
    private ObjectFactory objectFactory;


    @JavascriptInterface
    public void startListeningToOTP(String context, String data, String callback) {
        this.jsCallback = callback;
        this.context = context;
        this.jsData = data;

        SdkLogger.d(TAG, String.format("listening to otp : jsCallback = {%s}, context = {%s}, jsData = {%s}", callback, context, data));
        if (PermissionUtils.hasPermission(this.activity, Manifest.permission.SEND_SMS)
                && PermissionUtils.hasPermission(this.activity, Manifest.permission.RECEIVE_SMS)
                && PermissionUtils.hasPermission(this.activity, Manifest.permission.READ_SMS)) {
            handleSendOtp();
        } else {

            SdkLogger.e(TAG, "calling permission error call back for SEND_SMS...");
            onPermissionErrorCallback(Manifest.permission.SEND_SMS);
        }
    }

    @JavascriptInterface
    public void stopListeningToOTP() {
        try {
            SdkLogger.d(TAG, "trying to unregister sms receiver...");
            this.activity.unregisterReceiver(this.smsReadReceiver);
            SdkLogger.d(TAG, "sms receiver unregistered successfully");
        } catch (IllegalArgumentException e) {

            SdkLogger.e(TAG, String.format("sms receiver un-registration failed with message {%s} . ignore if sms receiver was not registered in first place", e.getMessage()));
        }
    }

    @Override
    public void onPermissionReceived(String[] permission, int[] isPermissionReceived) {

        if (isPermissionReceived.length > 0) {

            boolean isGranted = isPermissionReceived[0] == PackageManager.PERMISSION_GRANTED;
            if (!isGranted) {
                onErrorCallback(PermissionUtils.getPermissionDeniedCode(permission[0]));
            } else {
                switch (permission[0]) {
                    case Manifest.permission.READ_SMS:
                        handleSendOtp();
                        break;
                }
            }
        }

    }

    private String getErrorJson(String code) {

        Error errorStatus = this.objectFactory.<Error>get(Error.class);
        errorStatus.setCode(code);
        return errorStatus.toJsonString();
    }

    private String getPermissionJson(String permission) {
        PermissionsObject permissionsObject = this.objectFactory.<PermissionsObject>get(PermissionsObject.class);
        permissionsObject.setPermissionType(permission);
        PermissionsBody permissionsBody = this.objectFactory.<PermissionsBody>get(PermissionsBody.class);
        permissionsBody.addPermission(permissionsObject);
        return getBodyString(permissionsBody);
    }

    private String getBodyString(PermissionsBody permissionsBody) {

        Body<PermissionsBody> body = this.objectFactory.<Body>get(Body.class);
        body.setData(permissionsBody);
        return body.toJsonString();
    }

    private void onPermissionErrorCallback(String permission) {
        String error = getErrorJson(PermissionsHandler.PERMISSION_NOT_GRANTED);
        String permissionBodyString = getPermissionJson(permission);
        SdkLogger.d(TAG, String.format("calling onBridgeCallBack  callback = {%s}, error = {%s}, response={%s}, context={%s}, body={%s}.", this.jsCallback, error, null, this.context,
                permissionBodyString));
        this.bridgeCallback.onBridgeCallBack(this.jsCallback, error, null, this.context,
                permissionBodyString);

    }


    private void onErrorCallback(String errorCode) {
        String error = getErrorJson(errorCode);
        PermissionsBody permissionsBody = this.objectFactory.<PermissionsBody>get(PermissionsBody.class);
        String bodyString = getBodyString(permissionsBody);
        SdkLogger.d(TAG, String.format("calling onBridgeCallBack  callback = {%s}, error = {%s}, response={%s}, context={%s}, body={%s}.", this.jsCallback, error, null, this.context, bodyString));
        this.bridgeCallback.onBridgeCallBack(this.jsCallback, error, null, this.context, bodyString);
    }


    private void handleSendOtp() {
        this.smsReadReceiver.setListener(this);
        final IntentFilter theFilter = new IntentFilter();
        theFilter.addAction(ACTION);
        theFilter.setPriority(999);
        theFilter.addCategory(Intent.CATEGORY_DEFAULT);
        this.activity.registerReceiver(this.smsReadReceiver, theFilter);
    }

    @Override
    public void onSmsReceived(String sender, String sms) {
        ListenOtpData listenOtpData = ListenOtpData.fromJsonString(this.jsData, this.objectFactory, ListenOtpData.class);
        this.otpRegexPattern = listenOtpData.<String>get(ListenOtpData.KEY_REGEX);
        Pattern p = Pattern.compile(this.otpRegexPattern);
        Matcher m = p.matcher(sms);

        if (m.find() && m.groupCount() > 0) {
            String otp = m.group(1);
            ListenOtpBody listenOtpBody = this.objectFactory.<ListenOtpBody>get(ListenOtpBody.class);
            listenOtpBody.setOtp(otp);
            Body<ListenOtpBody> bodyResponse = this.objectFactory.<Body>get(Body.class);
            String response = this.objectFactory.getResponse(Constants.GenericConstants.SUCCESS).toJsonString();
            bodyResponse.setData(listenOtpBody);
            String body = bodyResponse.toJsonString();
            SdkLogger.d(TAG, String.format("calling onBridgeCallBack  callback = {%s}, error = {%s}, response={%s}, context={%s}, body={%s}.", this.jsCallback, null, response, this.context, body));
            this.bridgeCallback.onBridgeCallBack(this.jsCallback, null, response, this.context, body);
        } else {
            SdkLogger.d(TAG, "message not matched");
        }
    }


    @Override
    public void onActivityDestroyed() {
        stopListeningToOTP();
    }

    @Override
    public void init(ObjectFactory objectFactory, ObjectFactory.InitializationBundle initializationBundle) {

        this.activity = initializationBundle.<Activity>get(BaseWebActivity.ACTIVITY, null);
        this.smsReadReceiver = objectFactory.<SmsReceiver>get(SmsReceiver.class);
        this.bridgeCallback = (iBridgeCallback) initializationBundle.get(BaseWebActivity.BRIDGE_CALLBACK);
        this.objectFactory = objectFactory;
    }

    @Override
    public boolean isCachingAllowed() {
        return false;
    }
}
