package com.phonepe.intent.sdk.core;

import android.content.BroadcastReceiver;
import android.content.Context;
import android.content.Intent;
import android.os.Build;
import android.os.Bundle;
import android.telephony.SmsMessage;

import com.phonepe.intent.sdk.contracts.SmsListener;
import com.phonepe.intent.sdk.utils.SdkLogger;
import com.phonepe.intent.sdk.utils.Utils;

/**
 * Broadcast Receiver for receiving sms
 *
 * @author Sharath Pandeshwar
 * @since 26/05/2017
 */
public class SmsReceiver extends BroadcastReceiver implements ObjectFactoryInitializationStrategy {

    private static final String TAG = "SmsReceiver";
    private static final String FORMAT = "3gpp";
    private static final String PDUS = "pdus";
    private static final String BUNDLE = "bundle";
    private SmsListener smsListener;

    @Override
    public void onReceive(Context context, Intent intent) {
        SdkLogger.v(TAG, "Received triggered in SmsReceiver");
        try {
            onSmsReceived(context, intent);
        } catch (Exception e) {
            SdkLogger.e(TAG, String.format("sms parsing failed with exception = {%s}, intent = {%s}.", e.getMessage(), intent.toString()), e);
        }
    }

    private void onSmsReceived(Context context, Intent intent) throws Exception {

        final Bundle bundle = intent.getExtras();

        if (Utils.isNull(bundle, SmsReceiver.TAG, BUNDLE)) {

            return;
        }

        final Object[] pdusObj = (Object[]) bundle.get(PDUS);

        if (Utils.isNull(pdusObj, SmsReceiver.TAG, PDUS)) {

            return;
        }


        for (Object aPdusObj : pdusObj) {

            SmsMessage currentMessage;
            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.M) {
                currentMessage = SmsMessage.createFromPdu((byte[]) aPdusObj, FORMAT);
            } else {
                currentMessage = SmsMessage.createFromPdu((byte[]) aPdusObj);
            }
            String senderNum = currentMessage.getDisplayOriginatingAddress();
            String message = currentMessage.getDisplayMessageBody();

            if (smsListener != null) {
                SdkLogger.d(TAG, "calling sms listener ...");
                smsListener.onSmsReceived(senderNum, message);
            }
        }
    }

    /**
     * Set the listener for SMS Receiver
     *
     * @param smsListener Listener, using which callbacks will be provided
     */
    public void setListener(SmsListener smsListener) {
        this.smsListener = smsListener;
    }

    @Override
    public void init(ObjectFactory objectFactory, ObjectFactory.InitializationBundle initializationBundle) {

    }

    @Override
    public boolean isCachingAllowed() {
        return true;
    }
}