package com.phonepe.intent.sdk.networking;

import android.os.AsyncTask;
import android.support.annotation.NonNull;
import android.util.Pair;

import com.phonepe.intent.sdk.core.ObjectFactory;
import com.phonepe.intent.sdk.core.ObjectFactoryInitializationStrategy;
import com.phonepe.intent.sdk.utils.Config;
import com.phonepe.intent.sdk.utils.SdkLogger;

import java.io.InputStream;
import java.io.UnsupportedEncodingException;
import java.util.Map;
import java.util.concurrent.Executor;

/**
 * @author TheEternalWitness
 * @since 10/04/18.
 */
public class APIManager implements ObjectFactoryInitializationStrategy {

    private static final String TAG = "APIManager";
    private ObjectFactory objectFactory;
    private Executor executor;

    public void asyncGetRequest(final String endPoint, final Map<String, String> headers, final String body, @NonNull final INetworkResponseListener listener, final boolean useCache) {


        asyncRequest(endPoint, false, useCache, headers, body, listener);
    }

    private void asyncRequest(final String endPoint, final boolean isPost, final boolean useCache, final Map<String, String> headers, final String body, @NonNull final INetworkResponseListener listener) {

        new AsyncTask<Void, Void, NetworkResponse>() {
            @Override
            protected NetworkResponse doInBackground(Void... params) {
                HttpClient httpClient = createHttpClient(endPoint, isPost, useCache, false, headers, body, objectFactory);
                return httpClient.makeRequest();
            }

            @Override
            protected void onPostExecute(NetworkResponse networkResponse) {
                networkResponse.announceResult(listener);
            }
        }.executeOnExecutor(executor);
    }

    public void asyncPostRequest(final String endPoint, final Map<String, String> headers, final String body, @NonNull final INetworkResponseListener listener, final boolean useCache) {

        asyncRequest(endPoint, true, useCache, headers, body, listener);
    }

    public NetworkResponse makeGetRequest(final String endPoint, final Map<String, String> headers, final String body, final boolean useCache) {
        return makeRequest(endPoint, false, useCache, headers, body);
    }

    public NetworkResponse makePostRequest(final String endPoint, final Map<String, String> headers, final String body, final boolean useCache) {
        return makeRequest(endPoint, true, useCache, headers, body);
    }

    public Pair<Integer, InputStream> getInputStream(final String endPoint) {
        HttpClient httpClient = createHttpClient(endPoint, false, false, false, null, null, objectFactory);
        try {
            return httpClient.getResponse();
        } catch (UnsupportedEncodingException e) {
            SdkLogger.e(TAG, e.getMessage(), e);
            return new Pair<>(-1, null);
        }
    }

    private NetworkResponse makeRequest(final String endPoint, final boolean isPost, final boolean useCache, final Map<String, String> headers, final String body) {

        HttpClient httpClient = createHttpClient(endPoint, isPost, useCache, false, headers, body, objectFactory);
        return httpClient.makeRequest();
    }

    //*********************************************************************
    // Private methods
    //*********************************************************************


    @Override
    public void init(ObjectFactory objectFactory, ObjectFactory.InitializationBundle initializationBundle) {

        this.objectFactory = objectFactory;
        Config config = objectFactory.<Config>get(Config.class);
        this.executor = config.getExecutor();
    }

    @Override
    public boolean isCachingAllowed() {
        return true;
    }

    //*********************************************************************
    // Private classes
    //*********************************************************************

    public static class NetworkResponse implements ObjectFactoryInitializationStrategy {
        public int statusCode;
        public String response;
        public boolean isSuccess;

        public void updateNetworkResponse(int statusCode, String response, boolean isSuccess) {

            this.statusCode = statusCode;
            this.response = response;
            this.isSuccess = isSuccess;
        }

        public void announceResult(INetworkResponseListener listener) {

            if (listener != null) {

                if (this.isSuccess) {
                    listener.onSuccess(this.response);
                } else {
                    listener.onFailure(this.statusCode, this.response);
                }
            }
        }

        @Override
        public void init(ObjectFactory objectFactory, ObjectFactory.InitializationBundle initializationBundle) {

        }

        @Override
        public boolean isCachingAllowed() {
            return false;
        }
    }

    //*********************************************************************
    // Private utility methods
    //*********************************************************************


    private HttpClient createHttpClient(String url, boolean isPost, boolean useCache, boolean useDefaultCace, Map<String, String> headers, String body, ObjectFactory objectFactory) {

        headers = headers == null ? objectFactory.<String, String>getHashMap() : headers;
        ObjectFactory.InitializationBundle initializationBundle = objectFactory.<ObjectFactory.InitializationBundle>get(ObjectFactory.InitializationBundle.class);
        initializationBundle.put(HttpClient.KEY_URL, url);
        initializationBundle.put(HttpClient.KEY_IS_POST, isPost);
        initializationBundle.put(HttpClient.KEY_USE_CACHE, useCache);
        initializationBundle.put(HttpClient.KEY_USE_DEFAULT_CACHE, useDefaultCace);
        initializationBundle.put(HttpClient.KEY_HEADERS, headers);
        initializationBundle.put(HttpClient.KEY_BODY, body);
        return objectFactory.<HttpClient>get(HttpClient.class, initializationBundle);
    }

    //*********************************************************************
    // End of class
    //*********************************************************************
}
