package com.phonepe.intent.sdk.networking;


import android.util.Base64;
import android.util.Pair;

import com.phonepe.intent.sdk.core.ObjectFactory;
import com.phonepe.intent.sdk.core.ObjectFactoryInitializationStrategy;
import com.phonepe.intent.sdk.utils.AnalyticsManager;
import com.phonepe.intent.sdk.utils.RuntimeExceptionManager;
import com.phonepe.intent.sdk.utils.SdkLogger;

import java.io.BufferedReader;
import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.io.UnsupportedEncodingException;
import java.net.HttpURLConnection;
import java.net.ProtocolException;
import java.net.URL;
import java.nio.charset.StandardCharsets;
import java.security.KeyManagementException;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.security.cert.Certificate;
import java.security.cert.X509Certificate;
import java.util.HashMap;
import java.util.Map;

import javax.net.ssl.HostnameVerifier;
import javax.net.ssl.HttpsURLConnection;
import javax.net.ssl.SSLPeerUnverifiedException;

import static com.phonepe.intent.sdk.utils.AnalyticsManager.EventConstants.ERROR_MESSAGE;
import static com.phonepe.intent.sdk.utils.AnalyticsManager.Events.SDK_NETWORK_ERROR;

public class HttpClient implements ObjectFactoryInitializationStrategy {

    private static final String TAG = "APIManager";

    private boolean isSecure;
    private ObjectFactory objectFactory;
    private HttpsURLConnection httpsURLConnection;
    private HttpURLConnection httpURLConnection;
    private String body;
    public static final String KEY_URL = "url";
    public static final String KEY_IS_POST = "isPost";
    public static final String KEY_USE_DEFAULT_CACHE = "defaultCache";
    public static final String KEY_USE_CACHE = "useCache";
    public static final String KEY_HEADERS = "headers";
    public static final String KEY_BODY = "body";

    private boolean isSecure() {
        return isSecure;
    }

    private HttpURLConnection getConnection() {
        if (isSecure) {
            return httpsURLConnection;
        } else {
            return httpURLConnection;
        }
    }

    private void setUpClient(HttpURLConnection connection, boolean isPOST) throws ProtocolException {
        connection.setReadTimeout(NetworkConstants.CONNECTION_READ_TIMEOUT);
        connection.setConnectTimeout(NetworkConstants.CONNECTION_CONNECTION_TIMEOUT);
        if (isPOST) {
            connection.setRequestMethod("POST");
            connection.setDoOutput(true);
        } else {
            connection.setRequestMethod("GET");
        }
    }

    private void setHeaders(Map<String, String> headers) {
        StringBuilder stringBuilder = new StringBuilder();
        URL url = getConnection().getURL();
        if(url != null) {
            stringBuilder.append(url.toString()+"\n");
        }
        for (Map.Entry<String, String> entry : headers.entrySet()) {
            getConnection().setRequestProperty(entry.getKey(), entry.getValue());
            stringBuilder.append("Header :" + entry.getKey() + ":" + entry.getValue());
            stringBuilder.append('\n');
        }
        if(body != null) {
            stringBuilder.append("Body: "+body);
        }
        SdkLogger.d(TAG, "=============================================================================");
        SdkLogger.d(TAG, stringBuilder.toString());
    }

    public Certificate[] getServerCertificates() throws SSLPeerUnverifiedException {
        if (isSecure) {
            return httpsURLConnection.getServerCertificates();
        } else {
            return new Certificate[]{};
        }
    }

    private String readResponse(InputStream inputStream) throws IOException {

        StringBuilder sb = new StringBuilder();
        BufferedReader in = new BufferedReader(new InputStreamReader(inputStream));
        String line;
        while ((line = in.readLine()) != null) {
            sb.append(line);
        }
        in.close();
        return sb.toString();
    }

    public Pair<Integer, InputStream> getResponse() throws UnsupportedEncodingException {
        try {
            getConnection().connect();
            validatePinning();
            if (this.body != null) {
                OutputStream os = getConnection().getOutputStream();
                os.write(body.getBytes());
                os.flush();
                os.close();
            }
            int statusCode = getConnection().getResponseCode();

            if (200 <= statusCode && statusCode < 300) {
                return new Pair<>(statusCode, getConnection().getInputStream());
            } else {
                return new Pair<>(statusCode, getConnection().getErrorStream());
            }
        } catch (Exception e) {
            SdkLogger.e(TAG, e.getMessage(), e);
            InputStream stream = new ByteArrayInputStream(e.getMessage().getBytes("UTF-8"));
            return new Pair<>(-1, stream);
        }
    }

    public APIManager.NetworkResponse makeRequest() {
        SdkLogger.d(TAG, "=============================================================================");
        APIManager.NetworkResponse networkResponse = this.objectFactory.<APIManager.NetworkResponse>get(APIManager.NetworkResponse.class);

        if (getConnection() == null) {

            networkResponse.isSuccess = false;
            networkResponse.response = String.format("http(s)  url connection is null, please check logs from same {%s}", TAG);
            networkResponse.statusCode = -1;
            return networkResponse;
        }

        try {
            //Pair<Integer, InputStream> responsePair = getResponse();
            getConnection().connect();
            validatePinning();
            if (this.body != null) {
                OutputStream os = getConnection().getOutputStream();
                os.write(body.getBytes());
                os.flush();
                os.close();
            }
            int statusCode = getConnection().getResponseCode();
            URL url = getConnection().getURL();
            String urlString = "";
            if(url != null) {
                urlString = url.toString();
            }
            String response = "";
            if (200 <= statusCode && statusCode < 300) {
                response = readResponse(getConnection().getInputStream());
                networkResponse.updateNetworkResponse(statusCode, response, true);
            } else {
                response = readResponse(getConnection().getErrorStream());
                networkResponse.updateNetworkResponse(statusCode, response, false);
                if (!getConnection().getURL().getPath().contains(NetworkConstants.APIEndPoints.API_EVENT_INJESTION)) {
                    AnalyticsManager analyticsManager = this.objectFactory.<AnalyticsManager>get(AnalyticsManager.class);
                    analyticsManager.submit(analyticsManager.getEvent(SDK_NETWORK_ERROR).putInEventBody(ERROR_MESSAGE, networkResponse.response));
                }
            }
            SdkLogger.d(TAG, "Url: "+urlString+"\nStatus Code: "+statusCode+"\nResponse: "+response);
        } catch (Exception e) {
            SdkLogger.e(TAG, e.getMessage(), e);
            networkResponse.updateNetworkResponse(-1, e.getMessage(), false);
        }
        SdkLogger.d(TAG, "=============================================================================");
        return networkResponse;
    }

    private boolean validatePinning() throws SSLPeerUnverifiedException, NoSuchAlgorithmException {
        // https://medium.com/@appmattus/android-security-ssl-pinning-1db8acb6621e
        // https://www.paypal-engineering.com/2015/10/14/key-pinning-in-mobile-applications/

        if (!isSecure()) {

            return true;
        }

        Certificate[] certs = getServerCertificates();
        MessageDigest md = MessageDigest.getInstance("SHA-256");
        for (Certificate cert : certs) {
            X509Certificate x509Certificate = (X509Certificate) cert;
            byte[] key = x509Certificate.getPublicKey().getEncoded();
            md.update(key, 0, key.length);
            byte[] hashBytes = md.digest();
            String base64String = Base64.encodeToString(hashBytes, Base64.NO_WRAP);
            if (NetworkConstants.validCertificates.contains(base64String)) {
                return true;
            }
        }

        throw new SSLPeerUnverifiedException("Unidentified server.");
    }

    @Override
    public void init(ObjectFactory objectFactory, ObjectFactory.InitializationBundle initializationBundle) {

        this.objectFactory = objectFactory;
        try {

            URL url = new URL(initializationBundle.<String>get(KEY_URL, null));
            boolean isPostCall = initializationBundle.<Boolean>get(KEY_IS_POST, false);
            this.isSecure = url.toString().startsWith(NetworkConstants.HTTPS);
            if (isSecure) {

                httpsURLConnection = (HttpsURLConnection) url.openConnection();
                setUpClient(httpsURLConnection, isPostCall);
                httpsURLConnection.setSSLSocketFactory(new TLSSocketFactory());
                HostnameVerifier hostnameVerifier = this.objectFactory.<CustomHostNameVerifier>get(CustomHostNameVerifier.class);
                httpsURLConnection.setHostnameVerifier(hostnameVerifier);

            } else {
                httpURLConnection = (HttpURLConnection) url.openConnection();
                setUpClient(httpURLConnection, isPostCall);
            }

            HashMap<String, String> headers = this.objectFactory.<String, String>getHashMap();
            getConnection().setUseCaches(initializationBundle.<Boolean>get(KEY_USE_CACHE, false));
            getConnection().setDefaultUseCaches(initializationBundle.<Boolean>get(KEY_USE_DEFAULT_CACHE, false));
            this.body = initializationBundle.<String>get(KEY_BODY, null);
            setHeaders(initializationBundle.<HashMap<String, String>>get(KEY_HEADERS, headers));
        } catch (NoSuchAlgorithmException e) {

            String message = String.format("NoSuchAlgorithmException caught with message = {%s}", e.getMessage());
            this.objectFactory.getRuntimeExceptionManager().submit(TAG, message, RuntimeExceptionManager.Severity.LOW);
        } catch (KeyManagementException e) {
            String message = String.format("KeyManagementException caught with message = {%s}", e.getMessage());
            this.objectFactory.getRuntimeExceptionManager().submit(TAG, message, RuntimeExceptionManager.Severity.LOW);
        } catch (ProtocolException e) {
            String message = String.format("ProtocolException caught with message = {%s}", e.getMessage());
            this.objectFactory.getRuntimeExceptionManager().submit(TAG, message, RuntimeExceptionManager.Severity.LOW);
        } catch (IOException e) {
            String message = String.format("IOException caught with message = {%s}", e.getMessage());
            this.objectFactory.getRuntimeExceptionManager().submit(TAG, message, RuntimeExceptionManager.Severity.LOW);
        }
    }

    @Override
    public boolean isCachingAllowed() {
        return false;
    }
}
