package com.phonepe.intent.sdk.ui;

import android.annotation.SuppressLint;
import android.os.Bundle;
import android.support.annotation.NonNull;
import android.support.v7.app.AppCompatActivity;
import android.util.Base64;
import android.view.View;
import android.webkit.WebSettings;
import android.webkit.WebView;
import android.widget.ProgressBar;

import com.phonepe.intent.sdk.R;
import com.phonepe.intent.sdk.api.TransactionRequest;
import com.phonepe.intent.sdk.bridges.BridgeHandler;
import com.phonepe.intent.sdk.bridges.DataStore;
import com.phonepe.intent.sdk.bridges.PermissionsHandler;
import com.phonepe.intent.sdk.contracts.iBridgeCallback;
import com.phonepe.intent.sdk.contracts.iNativeCardCallback;
import com.phonepe.intent.sdk.core.ObjectFactory;
import com.phonepe.intent.sdk.core.SMSManager;
import com.phonepe.intent.sdk.models.SDKConfig;
import com.phonepe.intent.sdk.utils.AnalyticsManager;
import com.phonepe.intent.sdk.utils.Config;
import com.phonepe.intent.sdk.utils.Constants;
import com.phonepe.intent.sdk.utils.SdkLogger;
import com.phonepe.intent.sdk.utils.Utils;

import org.json.JSONArray;
import org.json.JSONObject;

import static com.phonepe.intent.sdk.utils.AnalyticsManager.EventConstants.ERROR_MESSAGE;
import static com.phonepe.intent.sdk.utils.AnalyticsManager.Events.SDK_BASE_WEB_ACTIVITY_CREATED;
import static com.phonepe.intent.sdk.utils.AnalyticsManager.Events.SDK_ERROR_TO_USER;

/**
 * @author TheEternalWitness
 * @since 31/03/18.
 */
public abstract class BaseWebActivity extends AppCompatActivity implements iBridgeCallback, iNativeCardCallback {

    public static final String TAG = "BaseWebActivity";
    public static final String ACTIVITY = "activity";
    public static final String BRIDGE_CALLBACK = "bridgeCallback";
    public static final String NATIVE_CARD_CALL_BACK = "nativeCardCallback";
    private static final String KEY_PAYMENT_SOURCES = "paymentSources";
    private static final String DEFAULT_LOADER = "default";
    private static final String KEY_PAYMENT_SOURCE_MODE = "mode";
    private SMSManager smsManager;
    private PermissionsHandler permissionHandler;
    private DataStore dataStore;
    private BridgeHandler bridgeHandler;

    protected WebView webView;
    private ObjectFactory objectFactory;
    private AnalyticsManager analyticsManager;
    private ProgressBar progressBar;
    private String loaderType;

    @Override
    public void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        setContentView(R.layout.activity_web);
        initViews();
        this.objectFactory = getIntent().<ObjectFactory>getParcelableExtra(Constants.BundleConstants.DATA_FACTORY);
        ObjectFactory.applicationContext = getApplicationContext();
        ObjectFactory.InitializationBundle initializationBundle = this.objectFactory.<ObjectFactory.InitializationBundle>get(ObjectFactory.InitializationBundle.class);
        TransactionRequest transactionRequest = getIntent().getParcelableExtra(Constants.BundleConstants.KEY_REQUEST);
        loaderType = DEFAULT_LOADER;
        if (transactionRequest != null) {
            try {
                String data = transactionRequest.getData();
                String decodedData = new String(Base64.decode(data, Base64.DEFAULT));
                JSONObject transactionRequestData = new JSONObject(decodedData);
                if (transactionRequestData.has(KEY_PAYMENT_SOURCES)) {
                    JSONArray paymentSources = transactionRequestData.getJSONArray(KEY_PAYMENT_SOURCES);
                    JSONObject paymentSource = paymentSources.getJSONObject(0);
                    if (paymentSource.has(KEY_PAYMENT_SOURCE_MODE)) {
                        loaderType = paymentSource.getString(KEY_PAYMENT_SOURCE_MODE);
                    }
                }
            } catch (Exception e) {
                SdkLogger.e(TAG, e.getMessage(), e);
            }
        }
        initializationBundle.put(ACTIVITY, this);
        initializationBundle.put(BRIDGE_CALLBACK, this);
        initializationBundle.put(NATIVE_CARD_CALL_BACK, this);
        initializationBundle.put(ObjectFactory.TAG, this.objectFactory);
        this.smsManager = this.objectFactory.<SMSManager>get(SMSManager.class, initializationBundle);
        this.bridgeHandler = this.objectFactory.<BridgeHandler>get(BridgeHandler.class, initializationBundle);
        this.permissionHandler = this.objectFactory.<PermissionsHandler>get(PermissionsHandler.class, initializationBundle);
        this.dataStore = this.objectFactory.<DataStore>get(DataStore.class, initializationBundle);
        this.analyticsManager = this.objectFactory.<AnalyticsManager>get(AnalyticsManager.class);
        this.analyticsManager.submit(this.analyticsManager.getEvent(SDK_BASE_WEB_ACTIVITY_CREATED));
        Utils.setUpResponseCache(this.objectFactory);
        initWebView();
    }

    protected ObjectFactory getObjectFactory() {

        return this.objectFactory;
    }

    @Override
    protected void onStop() {
        super.onStop();
        progressBar.setVisibility(View.GONE);
    }

    @Override
    public void onDestroy() {
        super.onDestroy();
        Utils.flushHttpResponseCache(this.objectFactory);
        this.webView.removeJavascriptInterface(SMSManager.TAG);
        this.webView.removeJavascriptInterface(PermissionsHandler.TAG);
        this.webView.removeJavascriptInterface(BridgeHandler.TAG);
        this.webView.removeJavascriptInterface(DataStore.TAG);
        this.smsManager.onActivityDestroyed();
    }

    //*********************************************************************
    // View related
    //*********************************************************************

    private void initViews() {
        this.webView = findViewById(R.id.webviewId);
        this.progressBar = findViewById(R.id.progressBar);

    }

    @SuppressLint("AddJavascriptInterface")
    protected void initWebView() {
        this.webView.addJavascriptInterface(this.dataStore, DataStore.TAG);
        this.webView.addJavascriptInterface(this.bridgeHandler, BridgeHandler.TAG);
        this.webView.addJavascriptInterface(this.smsManager, SMSManager.TAG);
        this.webView.addJavascriptInterface(this.permissionHandler, PermissionsHandler.TAG);
    }

    protected WebView getWebView() {
        return this.webView;
    }

    protected void showLoading() {
        progressBar.setVisibility(View.VISIBLE);
        webView.setVisibility(View.INVISIBLE);
    }

    protected void hideLoading() {
        runOnUiThread(new Runnable() {
            @Override
            public void run() {
                progressBar.setVisibility(View.GONE);
                webView.setVisibility(View.VISIBLE);
            }
        });
    }

    protected void hideProgressBar() {
        runOnUiThread(new Runnable() {
            @Override
            public void run() {
                progressBar.setVisibility(View.GONE);
            }
        });
    }

    public void showError(String errorMessage, boolean showRetry) {
        progressBar.setVisibility(View.GONE);
        String errorStatusMessageFormat = this.objectFactory.<Config>get(Config.class).getErrorStatusMessageFormat(showRetry);
        String applicationName = Utils.getAppNameFor(this.objectFactory, getPackageName());
        String statusMessage = String.format(errorStatusMessageFormat, errorMessage, applicationName);
        this.analyticsManager.submit(this.analyticsManager.getEvent(SDK_ERROR_TO_USER).putInEventBody(ERROR_MESSAGE, statusMessage));
    }

    protected String getLoaderType() {
        return loaderType;
    }

    //*********************************************************************
    // Permission Handling
    //*********************************************************************

    @Override
    public void onRequestPermissionsResult(int requestCode, @NonNull String permissions[], @NonNull int[] grantResults) {
        switch (requestCode) {
            case SMSManager.PERMISSIONS_REQUEST_SEND_SMS:
            case SMSManager.PERMISSIONS_REQUEST_READ_SMS:
            case SMSManager.PERMISSIONS_REQUEST_READ_PHONE_STATE:

                this.smsManager.onPermissionReceived(permissions, grantResults);
                return;
            case PermissionsHandler.JS_PERMISSIONS:
                if (this.permissionHandler != null) {
                    this.permissionHandler.onPermissionReceived(permissions, grantResults);
                }

        }
        /* I do not care what was the result, as it's not critical permission */
    }

    //*********************************************************************
    // APIs
    //*********************************************************************

    @Override
    public void openUrlInWebView(String url) {
        int cacheMode = WebSettings.LOAD_DEFAULT;
        try {
            if (!getObjectFactory().get(SDKConfig.class).isEnableWebViewCache()) {
                cacheMode = WebSettings.LOAD_NO_CACHE;
            }
            this.webView.getSettings().setCacheMode(cacheMode);
        } catch (Exception e) {
            SdkLogger.e(TAG, e.getMessage(), e);
        }
        cacheMode = this.webView.getSettings().getCacheMode();
        SdkLogger.d("CacheMode", "CacheMode: " + cacheMode);
        this.webView.loadUrl(url);
    }

    @Override
    public void onBridgeCallBack(String callback, String error, String response, String context, String body) {
        String webUrlFormat = this.objectFactory.<Config>get(Config.class).getWebUrlFormat();
        final String webUrl = String.format(webUrlFormat, callback, error, response, context, body);

        boolean isWebViewVisibile = this.webView != null && this.webView.getVisibility() == View.VISIBLE;
        if (!isFinishing() || isWebViewVisibile) {
            runOnUiThread(new Runnable() {
                @Override
                public void run() {
                    BaseWebActivity.this.webView.loadUrl(webUrl);
                }
            });
        }
    }

    //*********************************************************************
    // Abstract definitions
    //*********************************************************************

    //*********************************************************************
    // End of class
    //*********************************************************************
}
