package com.phonepe.intent.sdk.ui;

import android.app.IntentService;
import android.content.Intent;
import android.text.TextUtils;

import com.phonepe.intent.sdk.core.ObjectFactory;
import com.phonepe.intent.sdk.models.AbstractJson;
import com.phonepe.intent.sdk.models.SDKConfig;
import com.phonepe.intent.sdk.networking.APIManager;
import com.phonepe.intent.sdk.networking.INetworkResponseListener;
import com.phonepe.intent.sdk.networking.NetworkConstants;
import com.phonepe.intent.sdk.utils.Constants;
import com.phonepe.intent.sdk.utils.SdkLogger;
import com.phonepe.intent.sdk.utils.Utils;

import org.json.JSONArray;
import org.json.JSONObject;

import java.util.concurrent.CountDownLatch;

public class PreCacheService extends IntentService {

    private static final String TAG = "PreCacheService";
    private static final String KEY_ASSET_URL_LIST = "assetUrlList";
    private ObjectFactory objectFactory;

    public PreCacheService() {
        super(TAG);
        SdkLogger.v(TAG, "service is created");
    }


    @Override
    protected void onHandleIntent(Intent intent) {
        this.objectFactory = intent.<ObjectFactory>getParcelableExtra(Constants.BundleConstants.DATA_FACTORY);
        if(objectFactory == null) {
            return;
        }
        SDKConfig sdkConfig = objectFactory.get(SDKConfig.class);
        if(!sdkConfig.isPrecacheEnabled()) {
            SdkLogger.d(TAG, "Precache has been disabled by config");
            return;
        }
        final APIManager apiManager = this.objectFactory.<APIManager>get(APIManager.class);
        boolean isCacheSetUp = Utils.setUpResponseCache(this.objectFactory);
        if (!isCacheSetUp) {

            SdkLogger.e(TAG,"service failed to set up http response cache. returning ..");
            return;
        }

        SdkLogger.i(TAG,"fetching asset stats");
        String endPoint = sdkConfig.getPrecacheUrl() != null && sdkConfig.getPrecacheUrl() != "" ? sdkConfig.getPrecacheUrl() : NetworkConstants
                .getAssetStatsUrl(Utils.isTrue(this.objectFactory.<Boolean>get(Constants.MerchantMeta.IS_UAT)));
        APIManager.NetworkResponse networkResponse = apiManager.makeGetRequest(endPoint, null, null, false);

        if (!networkResponse.isSuccess) {

            SdkLogger.e(TAG,String.format("pre caching attempt failed, returning. network request failed, network response = {%s}.",networkResponse.response));
            return;
        }

        String assetStatsResponse = networkResponse.response;
        JSONObject assetStats = this.objectFactory.getJsonObject(assetStatsResponse);

        if (assetStats == null || !assetStats.has(KEY_ASSET_URL_LIST)) {

            SdkLogger.e(TAG, "either asset stats is null or does not have any asset url");
            return;
        }

        JSONArray assetUrlList = AbstractJson.<JSONArray>get(assetStats, KEY_ASSET_URL_LIST);
        int lengthOfList = assetUrlList.length();

        if (assetUrlList == null || lengthOfList == 0) {

            SdkLogger.e(TAG, "either assetUrlList is null or empty");
            return;
        }
        final CountDownLatch countDownLatch = new CountDownLatch(lengthOfList);

        for (int index = 0; index < lengthOfList; index++) {

            String url = AbstractJson.<String>get(assetUrlList, index);

            if (TextUtils.isEmpty(url)) {

                SdkLogger.e(TAG, "asset url is null or empty");
                countDownLatch.countDown();
                continue;
            }

            apiManager.asyncGetRequest(url, null, null, new INetworkResponseListener() {
                @Override
                public void onSuccess(String response) {
                    countDownLatch.countDown();
                }

                @Override
                public void onFailure(int responseCode, String error) {
                    countDownLatch.countDown();
                }
            }, true);
        }

        try {
            countDownLatch.await();
        } catch (InterruptedException e) {
            SdkLogger.e(TAG, String.format("thread got interrupted with message = {%s} , letch count = {%s}", e.getMessage(), Long.toString(countDownLatch
                    .getCount())), e);
        }
    }

}
