package com.phonepe.intent.sdk.utils;

import android.Manifest;
import android.annotation.SuppressLint;
import android.content.Context;
import android.content.pm.PackageInfo;
import android.content.pm.PackageManager;
import android.graphics.Point;
import android.net.ConnectivityManager;
import android.net.NetworkInfo;
import android.os.Build;
import android.support.v4.content.ContextCompat;
import android.telephony.CellIdentityGsm;
import android.telephony.CellIdentityLte;
import android.telephony.CellInfo;
import android.telephony.CellInfoGsm;
import android.telephony.CellInfoLte;
import android.telephony.CellSignalStrengthGsm;
import android.telephony.CellSignalStrengthLte;
import android.telephony.NeighboringCellInfo;
import android.telephony.SubscriptionInfo;
import android.telephony.SubscriptionManager;
import android.telephony.TelephonyManager;
import android.view.Display;
import android.view.WindowManager;

import com.phonepe.intent.sdk.models.MyLocation;
import com.phonepe.intent.sdk.models.NetworkType;
import com.phonepe.intent.sdk.models.TowerInfo;

import org.json.JSONArray;
import org.json.JSONObject;

import java.util.List;

/**
 * Class to provide all the details about the device
 * Created by viveksoneja on 31/03/16.
 */
@SuppressWarnings("unused")
public class SDKDeviceInfoProvider extends DeviceInfoProvider {

    private static final String TAG = "SDKDeviceInfoProvider";

    private static final String OPERATING_SYSTEM = "Android";
    private static final String UNKNOWN = "UNKNOWN";
    private static final String GET_CELLINFO_NOT_SUPPORTED = "GET_CELLINFO_NOT_SUPPORTED";
    private static final String CELL_ID = "cellId";
    private static final String LAC = "lac";
    private static final String RSSI = "rssi";
    private static final String DBM = "dbm";
    private static final String MNC = "mnc";
    private static final String MCC = "mcc";
    private static final String TAC = "tac";
    private LocationProvider mLocationProvider;

    /**
     * @return Returns the current OS
     */
    public String getOs() {
        return OPERATING_SYSTEM;
    }

    public String getProduct() {
        return Build.PRODUCT;
    }

    public String getBrand() {
        return Build.BRAND;
    }

    public String getDevice() {
        return Build.DEVICE;
    }

    public String getBuildVersion() {
        return Build.FINGERPRINT;
    }

    /**
     * @return The type of this device
     */
    public String getDeviceType() {
        return "Mobile";
    }

    public String getCapability() {
        return "capability";
    }

    /**
     * @return the current version code of the application
     */
    public int getAppVersionCode() {
        int appVersionCode = -1;
        try {
            PackageInfo pInfo = getApplicationContext().getPackageManager().getPackageInfo(getApplicationContext().getPackageName(), 0);
            appVersionCode = pInfo.versionCode;

        } catch (Exception e) {
            //Do nothing
        }

        return appVersionCode;
    }

    /**
     * @param context context
     * @return name of the application
     */
    public String getAppName(Context context) {
        return context.getString(context.getApplicationInfo().labelRes);
    }

    public Point getDeviceResolution() {
        WindowManager windowManager = (WindowManager) getApplicationContext().getSystemService(Context.WINDOW_SERVICE);
        Display display = windowManager.getDefaultDisplay();
        Point size = new Point();
        display.getSize(size);
        int width = size.x;
        int height = size.y;
        return new Point(width, height);
    }

    private boolean checkPermissions(final String permission) {
        int hasWriteContactsPermission = ContextCompat.checkSelfPermission(getApplicationContext(), permission);
        return hasWriteContactsPermission == PackageManager.PERMISSION_GRANTED;
    }

    @SuppressLint({"HardwareIds", "MissingPermission"})
    public String getIMEI() {
        String imie = null;
        try {
            if (getApplicationContext() != null) {
                if (checkPermissions(Manifest.permission.READ_PHONE_STATE)) {
                    try {
                        TelephonyManager mngr = (TelephonyManager) getApplicationContext().getSystemService(Context.TELEPHONY_SERVICE);
                        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.M) {
                            SubscriptionManager subscriptionManager = SubscriptionManager.from(getApplicationContext());
                            List<SubscriptionInfo> subscriptionInfos = subscriptionManager.getActiveSubscriptionInfoList();
                            if (subscriptionInfos != null && subscriptionInfos.size() > 0) {
                                boolean isFirst = true;
                                for (SubscriptionInfo sub : subscriptionInfos) {
                                    if (isFirst) {
                                        imie = mngr.getDeviceId(sub.getSimSlotIndex());
                                        isFirst = false;
                                    } else {
                                        imie += "," + mngr.getDeviceId(sub.getSimSlotIndex());
                                    }
                                }
                            }
                        } else {
                            imie = mngr.getDeviceId();
                        }
                    } catch (Exception e) {
                        // Do nothing
                    }
                } else {
                    imie = Constants.GenericConstants.PERMISSION_DENIED;
                }
            }
        } catch (Exception e) {
            // Do nothing
        }

        SdkLogger.d(TAG, "TEST FRAUD DETECTION IMIE " + imie);

        return imie;
    }

    @SuppressLint({"HardwareIds", "MissingPermission"})
    public String getICCID() {
        String iccId = null;
        try {
            if (getApplicationContext() != null) {
                if (checkPermissions(Manifest.permission.READ_PHONE_STATE)) {
                    if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP_MR1) {
                        // it returns a list with a SubscriptionInfo instance for each simcard
                        // there is other methods to retrieve SubscriptionInfos (see [2])
                        List<SubscriptionInfo> sis;
                        SubscriptionManager sm = SubscriptionManager.from(getApplicationContext());
                        sis = sm.getActiveSubscriptionInfoList();

                        if (sis != null && sis.size() > 0) {
                            boolean isFirst = true;
                            for (SubscriptionInfo s : sis) {
                                if (isFirst) {
                                    iccId = s.getIccId();
                                    isFirst = false;
                                } else {
                                    iccId += "," + s.getIccId();
                                }
                            }
                        }
                    } else {
                        TelephonyManager mngr = (TelephonyManager) getApplicationContext().getSystemService(Context.TELEPHONY_SERVICE);
                        iccId = mngr.getSimSerialNumber();
                    }
                } else {
                    iccId = Constants.GenericConstants.PERMISSION_DENIED;
                }
            }
        } catch (Exception e) {
            // Do nothing
        }


        SdkLogger.d("PhonePe", "TEST FRAUD DETECTION ICCID " + iccId);
        return iccId;
    }

    // Tower info
    public TowerInfo getCellInfo() {
        TowerInfo towerInfo = getObjectFactory().<TowerInfo>get(TowerInfo.class);
        JSONArray cellList = new JSONArray();
        if (checkPermissions(Manifest.permission.ACCESS_COARSE_LOCATION) || checkPermissions(Manifest.permission.ACCESS_FINE_LOCATION)) {
            try {
                TelephonyManager tel = (TelephonyManager) getApplicationContext().getSystemService(Context.TELEPHONY_SERVICE);
                // Type of the network
                int phoneTypeInt = tel.getPhoneType();
                String phoneType = null;
                phoneType = phoneTypeInt == TelephonyManager.PHONE_TYPE_GSM ? "gsm" : phoneType;
                phoneType = phoneTypeInt == TelephonyManager.PHONE_TYPE_CDMA ? "cdma" : phoneType;

                if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.JELLY_BEAN_MR1) {
                    @SuppressLint("MissingPermission") List<CellInfo> infos = tel.getAllCellInfo();
                    if (infos != null) {
                        for (int i = 0; i < infos.size(); ++i) {
                            JSONObject cellObj = new JSONObject();
                            CellInfo info = infos.get(i);
                            if (info instanceof CellInfoGsm) {
                                CellSignalStrengthGsm gsm = ((CellInfoGsm) info).getCellSignalStrength();
                                CellIdentityGsm identityGsm = ((CellInfoGsm) info).getCellIdentity();
                                cellObj.put(CELL_ID, identityGsm.getCid());
                                cellObj.put(LAC, identityGsm.getLac());
                                cellObj.put(DBM, gsm.getDbm());
                                cellObj.put(MNC, identityGsm.getMnc());
                                cellObj.put(MCC, identityGsm.getMcc());
                                cellList.put(cellObj);
                            } else if (info instanceof CellInfoLte) {
                                CellSignalStrengthLte lte = ((CellInfoLte) info).getCellSignalStrength();
                                CellIdentityLte identityLte = ((CellInfoLte) info).getCellIdentity();
                                cellObj.put(CELL_ID, identityLte.getCi());
                                cellObj.put(TAC, identityLte.getTac());
                                cellObj.put(DBM, lte.getDbm());
                                cellObj.put(MNC, identityLte.getMnc());
                                cellObj.put(MCC, identityLte.getMcc());
                                cellList.put(cellObj);
                            }
                        }
                    }
                    towerInfo.setSuccess(true);
                    towerInfo.setValues(cellList);
                } else {
                    // getNeighboringCellInfo() is deprecated in Android SDK version 29
                    towerInfo.setSuccess(false);
                    towerInfo.setErrorCode(GET_CELLINFO_NOT_SUPPORTED);
                }
            } catch (Exception e) {
                towerInfo.setSuccess(false);
                towerInfo.setErrorCode(UNKNOWN);
            }
        } else {
            towerInfo.setSuccess(false);
            towerInfo.setErrorCode(Constants.GenericConstants.PERMISSION_DENIED);
        }
        return towerInfo;
    }

    public MyLocation getMyLocation() {
        MyLocation myLocation = null;
        if (mLocationProvider != null) {
            myLocation = mLocationProvider.getLastKnownLocation(getApplicationContext());
        }

        MyLocation locationZero = getObjectFactory().<MyLocation>get(MyLocation.class);
        locationZero.setLongitude(0.0);
        locationZero.setLatitude(0.0);
        return myLocation != null ? myLocation : locationZero;
    }

    public String getLocation() {
        MyLocation myLocation = getMyLocation();
        if (myLocation.isZeroLocation()) {
            return myLocation.toJsonString();
        }
        return Constants.GenericConstants.PERMISSION_DENIED;
    }

    public String getHardware() {
        return Build.HARDWARE;
    }

    public String getNetWorkType() {

        // Get connect manger
        final ConnectivityManager connMgr = (ConnectivityManager)
                getApplicationContext().getSystemService(Context.CONNECTIVITY_SERVICE);
        NetworkInfo info = connMgr.getActiveNetworkInfo();

        // check for wifi
        if (info != null) {
            int netType = info.getType();
            int netSubtype = info.getSubtype();
            if (netType == ConnectivityManager.TYPE_WIFI) {
                return NetworkType.WIFI.getValue();
            } else if (netType == ConnectivityManager.TYPE_MOBILE) {
                return getNetworkClass();
            }
        }
        return NetworkType.NO_NETWORK.getValue();

    }

    private String getNetworkClass() {
        TelephonyManager mTelephonyManager = (TelephonyManager)
                getApplicationContext().getSystemService(Context.TELEPHONY_SERVICE);
        int networkType = mTelephonyManager.getNetworkType();
        switch (networkType) {
            case TelephonyManager.NETWORK_TYPE_GPRS:
            case TelephonyManager.NETWORK_TYPE_EDGE:
            case TelephonyManager.NETWORK_TYPE_CDMA:
            case TelephonyManager.NETWORK_TYPE_1xRTT:
            case TelephonyManager.NETWORK_TYPE_IDEN:
                return NetworkType.MOBILE_DATA_2G.getValue();
            case TelephonyManager.NETWORK_TYPE_UMTS:
            case TelephonyManager.NETWORK_TYPE_EVDO_0:
            case TelephonyManager.NETWORK_TYPE_EVDO_A:
            case TelephonyManager.NETWORK_TYPE_HSDPA:
            case TelephonyManager.NETWORK_TYPE_HSUPA:
            case TelephonyManager.NETWORK_TYPE_HSPA:
            case TelephonyManager.NETWORK_TYPE_EVDO_B:
            case TelephonyManager.NETWORK_TYPE_EHRPD:
            case TelephonyManager.NETWORK_TYPE_HSPAP:
                return NetworkType.MOBILE_DATA_3G.getValue();
            case TelephonyManager.NETWORK_TYPE_LTE:
                return NetworkType.MOBILE_DATA_4G.getValue();
            default:
                return NetworkType.MOBILE_DATA_UNKNOWN.getValue();
        }
    }
}
