package com.phonepe.intent.sdk.utils;

import android.content.Context;
import android.content.Intent;
import android.content.pm.ApplicationInfo;
import android.content.pm.PackageManager;
import android.content.pm.ResolveInfo;
import android.graphics.drawable.Drawable;
import android.net.Uri;
import android.net.http.HttpResponseCache;
import android.os.Build;
import android.support.annotation.NonNull;
import android.util.Base64;

import com.phonepe.intent.sdk.BuildConfig;
import com.phonepe.intent.sdk.api.PhonePe;
import com.phonepe.intent.sdk.api.PhonePeInitException;
import com.phonepe.intent.sdk.core.ObjectFactory;
import com.phonepe.intent.sdk.networking.NetworkConstants;

import org.json.JSONException;
import org.json.JSONObject;

import java.io.File;
import java.io.IOException;
import java.util.List;
import java.util.Locale;

import static com.phonepe.intent.sdk.utils.AnalyticsManager.EventConstants.TRANSACTION_ID;

/**
 * @author TheEternalWitness
 * @since 31/03/18.
 */
public class Utils {

    private static final String TAG = "Utils";

    /**
     * API to find if PhonePe app is installed and thus to show PhonePe payment option or not
     *
     * @param context Instance of {@linkplain Context}
     * @return <code>true</code> if PhonePe app is installed and <code>false</code> otherwise
     */
    public static boolean isAppExist(Context context, String packageName) {
        if(context == null) {
            return false;
        }
        PackageManager packageManager = context.getPackageManager();
        Intent launchIntent = packageManager.getLaunchIntentForPackage(packageName);

        if (launchIntent == null) {

            return false;
        }
        List<ResolveInfo> list = packageManager.queryIntentActivities(launchIntent, PackageManager.MATCH_DEFAULT_ONLY);
        return !list.isEmpty();
    }

    public static List<ResolveInfo> getResolveInfo(ObjectFactory objectFactory, Intent implicitIntent) {

        try {
            if (isNull(implicitIntent, Utils.TAG, "implicitIntent")) {

                objectFactory.getRuntimeExceptionManager().submit(Utils.TAG, "implicitIntent is null", RuntimeExceptionManager.Severity.LOW);
                return objectFactory.<ResolveInfo>getArrayList();
            }

            Context applicationContext = objectFactory.getApplicationContext();
            PackageManager packageManager = applicationContext.getPackageManager();
            return packageManager.queryIntentActivities(implicitIntent, PackageManager.MATCH_DEFAULT_ONLY);
        } catch (Exception e) {
            return null;
        }
    }

    public static void cacheTransactionId(String data){

        ObjectFactory objectFactory;
        try {
            objectFactory = PhonePe.getObjectFactory();
            String decodedData = new String(Base64.decode(data, Base64.DEFAULT));
            JSONObject transactionRequestData = new JSONObject(decodedData);
            objectFactory.cache(TRANSACTION_ID,transactionRequestData.get(TRANSACTION_ID));
        } catch (JSONException | PhonePeInitException e) {
            SdkLogger.e(TAG,"Error getting txnId from data");
        }

    }

    public static boolean isNull(Object object, String tag, String objectName) {

        if (object == null) {

            SdkLogger.d(tag, String.format(Constants.LogFormats.OBJECT_IS_NULL_OR_EMPTY, objectName));
            return true;
        }

        return false;
    }

    public static boolean isTrue(Boolean bool) {

        return bool != null && bool.booleanValue();
    }

    public static boolean isNullOrEmpty(String str) {
        return str == null || str.trim().isEmpty();
    }

    public static String getPhonePePackageName(ObjectFactory objectFactory) {

        Context context = objectFactory.getApplicationContext();
        boolean isUat = isTrue(objectFactory.<Boolean>get(Constants.MerchantMeta.IS_UAT));

        if (!isUat) {

            return objectFactory.<Config>get(Config.class).getPhonePeAppPackageName();
        }

        if(BuildConfig.BUILD_TYPE.matches(NetworkConstants.ENV_STAGE)) {
            String phonepeStagePackageName = objectFactory.<Config>get(Config.class).getPhonePeStagePackageName();
            if(isAppExist(context, phonepeStagePackageName))
                return phonepeStagePackageName;

            return objectFactory.<Config>get(Config.class).getPhonePeStageInternalPackageName();
        }

        String phonePeUatInternalPackageName = objectFactory.<Config>get(Config.class).getPhonePeUatInternalPackageName();
        if (isAppExist(context, phonePeUatInternalPackageName)) {

            return phonePeUatInternalPackageName;
        }

        return objectFactory.<Config>get(Config.class).getPhonePeUatPackageName();
    }

    public static boolean isUPIUrl(String data) {
        Uri uri = Uri.parse(data);
        return uri != null;
        // Sdk should allow to handle any scheme sent by server
        //return uri != null && (uri.getScheme().equals(Constants.GenericConstants.UPI) || uri.getScheme().equals(Constants.GenericConstants.PPE));
    }

    /**
     * Sets up the {@linkplain HttpResponseCache}
     *
     * @param objectFactory
     * @return true when the cache is successfully set up else false.
     */
    public static boolean setUpResponseCache(ObjectFactory objectFactory) {

        if(objectFactory.get(HttpResponseCache.class.getCanonicalName()) != null){

            SdkLogger.i(TAG,"http response cache is already installed, returning...");
            return true;
        }

        try {

            SdkLogger.d(TAG, "trying to setup http cache ...");
            HttpResponseCache cache = HttpResponseCache.getInstalled();
            if (cache == null) {
                Context context = objectFactory.getApplicationContext();
                String cacheDir = objectFactory.<Config>get(Config.class).getPhonePeSdkCacheDir();
                File httpCacheDir = new File(context.getCacheDir(), cacheDir);
                long httpCacheSize = objectFactory.<Config>get(Config.class).getPhonePeSdkCacheSize();
                SdkLogger.d(TAG, String.format("trying to setup http cache in dir = {%s}.", httpCacheDir));
                HttpResponseCache.install(httpCacheDir, httpCacheSize);
            }
            SdkLogger.d(TAG, "http response cache is installed");

            objectFactory.cache(HttpResponseCache.class.getCanonicalName(),cache);
            return true;
        } catch (Exception e) {

            SdkLogger.w(TAG, String.format("IOException caught , http response cache installation failed. exception message = {%s}", e.getMessage()));
            return false;
        }
    }


    public static boolean flushHttpResponseCache(@NonNull ObjectFactory objectFactory){

        HttpResponseCache httpResponseCache = objectFactory.<HttpResponseCache>get(HttpResponseCache.class.getCanonicalName());

        if(httpResponseCache == null) {

            SdkLogger.e(TAG,"http response cache is not installed, can not flush");
            return false;
        }

        httpResponseCache.flush();

        SdkLogger.v(TAG,"http response cache is flushed");
        return true;
    }

    public static String getValidUpiFormattedPhone(String phone) {
        if (phone.length() == 12 && phone.startsWith("91")) {
            return phone;
        } else if (phone.length() == 10) {
            return "91" + phone;
        }
        return null;
    }

    /**
     * Returns the amount in Rs, that can be sent to NPCI library
     *
     * @param amount amount in paise
     * @return converted amount
     */
    public static String getAmountInRs(long amount) {
        return String.format(Locale.ENGLISH, "%" + ".2f", ((double) amount) / 100.0);
    }

//    public static boolean isJusPaySDKIntegrated(ObjectFactory objectFactory) {
//
//        try {
//
//            SdkLogger.d(TAG, "checking for JusPay integration...");
//            String justPayClassName = objectFactory.<Config>get(Config.class).getJustPaySdkClassName();
//            Class.forName(justPayClassName);
//            SdkLogger.d(TAG, "JusPay is integrated");
//        } catch (ClassNotFoundException e) {
//
//            SdkLogger.d(TAG, String.format("JusPay app is not integrated"));
//            return false;
//        }
//
//        return isTrue(objectFactory.<Boolean>get(Constants.MerchantMeta.JUSPAY_ENABLED));
//    }

    public static boolean isPhonePeReferralPresent(String url) {
        return url.contains(Constants.GenericConstants.KEY_PHONEPE_CAMPAIGN_KEY);
    }

    public static boolean isBelowAndroidOreo() {
        return Build.VERSION.SDK_INT <= Build.VERSION_CODES.N_MR1;
    }

    public static String getAppNameFor(ObjectFactory objectFactory, String packageName) {

        try {
            SdkLogger.d(TAG, String.format("trying to get application name for the package = {%s}", packageName));
            Context context = objectFactory.getApplicationContext();
            PackageManager packageManager = context.getPackageManager();
            ApplicationInfo info = packageManager.getApplicationInfo(packageName, PackageManager.GET_META_DATA);
            String appName = packageManager.getApplicationLabel(info).toString();
            SdkLogger.d(TAG, String.format("application name for package name = {%s} is {%s}", packageName, appName));
            return appName;
        } catch (Exception e) {

            SdkLogger.e(TAG, String.format("Exception app name not found, exception message = {%s}", e.getMessage()));
            return objectFactory.<Config>get(Config.class).getDefaultAppNamePlaceHolder();
        }
    }

    public static Drawable getAppIcon(@NonNull String packageName, @NonNull ObjectFactory objectFactory){
        try {
            return objectFactory.getApplicationContext().getPackageManager().getApplicationIcon(packageName);
        }
        catch (Exception e) {
            SdkLogger.e(TAG, String.format("Exception app name not found, exception message = {%s}", e.getMessage()));
            return null;
        }
    }

    public static int getVersionCode(@NonNull String packageName, @NonNull ObjectFactory objectFactory) {

        Context applicationContext = objectFactory.getApplicationContext();

        try {
            return applicationContext.getPackageManager().getPackageInfo(packageName, 0).versionCode;
        } catch (Exception e) {
            SdkLogger.e(TAG, String.format("Exception app name not found, exception message = {%s}", e.getMessage()));
            return -1;
        }
    }

    public static boolean isPhonePeAppInstalled (ObjectFactory objectFactory) {

            return isAppExist(objectFactory.getApplicationContext(), getPhonePePackageName(objectFactory));

    }

    public static String getCheckSum(ObjectFactory objectFactory,String endPoint,String base64Payload) {
        try {
//            String signature = objectFactory.getPackageSignature();
            String signature = "YHyczmD0QO9iCWDbF9bm6T3Ysy0=";
            CryptLib cryptLib = new CryptLib();
            byte[] hash;
            if(base64Payload!=null) {
            hash = cryptLib.SHA256(base64Payload+endPoint+signature);
            } else  {
                hash = cryptLib.SHA256(endPoint+signature);
            }

            StringBuffer hashtext = new StringBuffer();

            for (int i = 0; i < hash.length; i++) {
                String hex = Integer.toHexString(0xff & hash[i]);
                if (hex.length() == 1) hashtext.append('0');
                hashtext.append(hex);
            }
            // Uncomment this when you want to debug Checksum logic
            //SdkLogger.d(TAG, "Base64Payload: "+base64Payload+"\nEndPoint: "+endPoint+"\nSignature: "+signature+"\nHashText:"+hashtext);
            return hashtext.toString();
        } catch (Exception e) {
            SdkLogger.e(TAG, e.getMessage(), e);
            return null;
        }
    }
}
