package com.phonepe.sdk.javasdk;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.phonepe.sdk.javasdk.config.ConfigurationFactory;
import com.phonepe.sdk.javasdk.config.models.HttpClientConfig;
import com.phonepe.sdk.javasdk.config.models.HystrixConfig;
import com.phonepe.sdk.javasdk.config.models.InitConfig;
import com.phonepe.sdk.javasdk.config.models.MerchantConfig;
import com.phonepe.sdk.javasdk.config.models.PhonePeSDKConfig;
import com.phonepe.sdk.javasdk.config.models.StatusConfig;
import com.phonepe.sdk.javasdk.http.PhonePeHttpClientFactory;
import com.phonepe.sdk.javasdk.transaction.client.TransactionClient;
import com.phonepe.sdk.javasdk.transaction.init.web.WebTransactionInitiator;
import com.phonepe.sdk.javasdk.transaction.status.TransactionStatusChecker;
import com.phonepe.sdk.javasdk.transaction.callback.CallbackHandler;
import com.phonepe.sdk.javasdk.transaction.init.models.InitResponse;
import com.phonepe.sdk.javasdk.transaction.init.TransactionInitiator;
import com.phonepe.sdk.javasdk.transaction.status.models.StatusResponse;
import lombok.extern.slf4j.Slf4j;
import okhttp3.OkHttpClient;

import javax.validation.Validation;
import javax.validation.Validator;

/**
 * Facade that wraps the {@link com.phonepe.sdk.javasdk.transaction.init.TransactionInitiator} and
 * {@link com.phonepe.sdk.javasdk.transaction.callback.CallbackHandler} and
 * {@link com.phonepe.sdk.javasdk.transaction.status.TransactionStatusChecker} for the
 * user.
 */
@Slf4j
public class PhonePeClient {

    private TransactionInitiator transactionInitiator;

    private CallbackHandler callbackHandler;

    private TransactionStatusChecker statusChecker;

    //Static member variable along with private constructor and
    // synchronised getInstance method ensures that only 1 instance of the client
    // is available.
    private static PhonePeClient phonePeClient;

    /*
     Default Constructor is private so that it cannot be called
     from outside.
     */
    private PhonePeClient(){

    }

    public static synchronized PhonePeClient getPhonePeClient(){
        if(phonePeClient == null){
            phonePeClient = new PhonePeClient();
            phonePeClient.init();
        }
        return phonePeClient;
    }

    //TODO: Check if this initialisation Process of PhonePeClient can be moved to a Factory Class and
    //TODO: made to return an fully configured object of PhonePeClient.
    public void init(){
        try{
        ObjectMapper objectMapper = new ObjectMapper();
        Validator validator = Validation.buildDefaultValidatorFactory()
                                        .getValidator();
        PhonePeSDKConfig phonePeSDKConfig = new ConfigurationFactory<PhonePeSDKConfig>(PhonePeSDKConfig.class, validator, objectMapper)
                .build();
        HttpClientConfig httpClientConfig = phonePeSDKConfig.getHttpClientConfig();
        StatusConfig statusConfig = phonePeSDKConfig.getStatusConfig();
        MerchantConfig merchantConfig = phonePeSDKConfig.getMerchantConfig();
        InitConfig initConfig = phonePeSDKConfig.getInitConfig();
        HystrixConfig hystrixConfig = phonePeSDKConfig.getHystrixConfig();
        OkHttpClient client = PhonePeHttpClientFactory.buildOkHttpClient(httpClientConfig);
        TransactionClient transactionClient = new TransactionClient(httpClientConfig,objectMapper,client, hystrixConfig);
        this.statusChecker  = TransactionStatusChecker.builder()
                                                      .merchantConfig(merchantConfig)
                                                      .statusConfig(statusConfig)
                                                      .transactionClient(transactionClient)
                                                      .build();

        this.callbackHandler =  CallbackHandler.builder()
                                               .merchantConfig(merchantConfig)
                                               .transactionStatusChecker(this.statusChecker)
                                               .build();
        this.transactionInitiator = WebTransactionInitiator.builder()
                                                           .initConfig(initConfig)
                                                           .merchantConfig(merchantConfig)
                                                           .transactionClient(transactionClient)
                                                           .build();

        } catch (Exception ex){
            log.error("Exception occurred during PhonePeClient init");
        }
    }

    public InitResponse initTransaction(final String transactionId,
                                        final int keyIndex,
                                        final String userId,
                                        final Long amount,
                                        final String orderId){
        return this.transactionInitiator.initiateTransaction(transactionId, keyIndex, userId, amount,orderId);
    }

    public StatusResponse handleCallback(final String responseReceived, final int keyIndex){
        return this.callbackHandler.handleCallback(responseReceived, keyIndex);
    }

    public StatusResponse checkTransactionStatus(final String transactionId,
                                                 final int keyIndex){
        return this.statusChecker.checkTransactionStatus(transactionId, keyIndex);
    }

}