package com.phonepe.sdk.javasdk.http;

import com.google.common.base.Preconditions;
import com.phonepe.sdk.javasdk.config.models.HttpClientConfig;
import lombok.extern.slf4j.Slf4j;
import okhttp3.ConnectionPool;
import okhttp3.ConnectionSpec;
import okhttp3.Dispatcher;
import okhttp3.OkHttpClient;

import javax.net.ssl.HostnameVerifier;
import javax.net.ssl.SSLSession;
import java.util.Collections;
import java.util.concurrent.TimeUnit;

@Slf4j
public class PhonePeHttpClientFactory {

  public static OkHttpClient buildOkHttpClient(final HttpClientConfig httpClientConfig) {

    Preconditions.checkNotNull(httpClientConfig);
    return createClient(httpClientConfig);
  }

  private static OkHttpClient createClient(final HttpClientConfig httpClientConfig) {

    int connections = httpClientConfig.getConnections();
    connections = connections == 0 ? 10 : connections;

    int idleTimeOutSeconds = httpClientConfig.getIdleTimeOutSeconds();
    idleTimeOutSeconds = idleTimeOutSeconds == 0 ? 30 : idleTimeOutSeconds;

    int connTimeout = httpClientConfig.getConnectTimeoutMs();
    connTimeout = connTimeout == 0 ? 10000 : connTimeout;

    int opTimeout = httpClientConfig.getOpTimeoutMs();
    opTimeout = opTimeout == 0 ? 10000 : opTimeout;

    Dispatcher dispatcher = new Dispatcher();
    dispatcher.setMaxRequests(connections);
    dispatcher.setMaxRequestsPerHost(connections);

    ConnectionSpec connectionSpec = httpClientConfig.isSecured() ?
                                    new ConnectionSpec.Builder(ConnectionSpec.MODERN_TLS)
                                                  .allEnabledTlsVersions()
                                                  .allEnabledCipherSuites()
                                                  .build() : null;

    OkHttpClient.Builder clientBuilder = new OkHttpClient.Builder()
            .followRedirects(false)
            .followSslRedirects(false)
            .retryOnConnectionFailure(true)
            .connectionPool(new ConnectionPool(connections, idleTimeOutSeconds, TimeUnit.SECONDS))
            .connectTimeout(connTimeout, TimeUnit.MILLISECONDS)
            .readTimeout(opTimeout, TimeUnit.MILLISECONDS)
            .writeTimeout(opTimeout, TimeUnit.MILLISECONDS)
            .dispatcher(dispatcher);

    if(connectionSpec != null){
      clientBuilder.connectionSpecs(Collections.singletonList(connectionSpec));
    }

    // TODO: Check if this can be or needs to be set to OkHostnameVerifier
    // TODO: or if we can skip setting this to use the default configuration.
    clientBuilder.hostnameVerifier(
            new HostnameVerifier() {
              @Override
              public boolean verify(String hostname, SSLSession session) {
                return true;
              }
            });

    return clientBuilder.build();
  }
}
