package com.phonepe.sdk.javasdk.transaction.init;

import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.phonepe.sdk.javasdk.config.models.APIKeyConfig;
import com.phonepe.sdk.javasdk.config.models.InitConfig;
import com.phonepe.sdk.javasdk.config.models.MerchantConfig;
import com.phonepe.sdk.javasdk.http.PhonePeException;
import com.phonepe.sdk.javasdk.http.models.HttpHeaderPair;
import com.phonepe.sdk.javasdk.http.utils.HttpUtils;
import com.phonepe.sdk.javasdk.transaction.client.TransactionClient;
import com.phonepe.sdk.javasdk.transaction.init.models.AllowedAccountConstraint;
import com.phonepe.sdk.javasdk.transaction.init.models.Base64Request;
import com.phonepe.sdk.javasdk.utils.ChecksumUtils;
import com.phonepe.sdk.javasdk.transaction.init.models.InitRequest;
import com.phonepe.sdk.javasdk.transaction.init.models.InitResponse;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;

import java.util.ArrayList;
import java.util.List;
import java.util.Set;

/**
 * Template Method class for initiating the transaction for any given type of flow.
 */
@Slf4j
public abstract class TransactionInitiator {

    private TransactionClient transactionClient;

    private InitConfig initConfig;

    private MerchantConfig merchantConfig;

    public TransactionInitiator(final TransactionClient transactionClient,
                                final InitConfig initConfig,
                                final MerchantConfig merchantConfig) {
        this.transactionClient = transactionClient;
        this.initConfig = initConfig;
        this.merchantConfig = merchantConfig;
    }


    public InitResponse initiateTransaction(final String transactionId,
                                            final Long amount){
        TransactionInitComponent transactionInitComponent = new SimpleTransactionComponent();
        InitRequest initRequest = transactionInitComponent.getInitRequest(this.merchantConfig.getMerchantId(), transactionId, amount);
        return initiateTransaction(initRequest, "","","","",1 );
    }

    public InitResponse initiateTransaction(final String transactionId,
                                            final String userId,
                                            final Long amount){
        SimpleTransactionComponent transactionInitComponent = new SimpleTransactionComponent();
        InitRequest initRequest = transactionInitComponent.getInitRequest(this.merchantConfig.getMerchantId(),transactionId, userId, amount);
        return initiateTransaction(initRequest, "","","","",1 );
    }

    public InitResponse initiateTransaction(final String transactionId,
                                            final int keyIndex,
                                            final String userId,
                                            final Long amount){
        SimpleTransactionComponent transactionInitComponent = new SimpleTransactionComponent();
        InitRequest initRequest = transactionInitComponent.getInitRequest(this.merchantConfig.getMerchantId(),transactionId, userId, amount);
        return initiateTransaction(initRequest, "","","","",keyIndex);
    }

    public InitResponse initiateTransaction(final String transactionId,
                                            final int keyIndex,
                                            final String userId,
                                            final Long amount,
                                            final String orderId){
        SimpleTransactionComponent transactionInitComponent = new SimpleTransactionComponent();
        InitRequest initRequest = transactionInitComponent.getInitRequest(this.merchantConfig.getMerchantId(),transactionId, userId, amount, orderId);
        return initiateTransaction(initRequest, "","","","",keyIndex);
    }

    public InitResponse initiateTransaction(final String transactionId,
                                            final Long amount,
                                            final Set<String> offerTags){
        SimpleTransactionComponent transactionInitComponent = new SimpleTransactionComponent();
        OfferTransactionComponent offerTransactionComponent = new OfferTransactionComponent(transactionInitComponent, offerTags);
        InitRequest offerInitRequest = offerTransactionComponent.getInitRequest(this.merchantConfig.getMerchantId(),transactionId,amount);
        return initiateTransaction(offerInitRequest, "","","","",1);
    }

    public InitResponse initiateTransaction(final String transactionId,
                                            final Long amount,
                                            final List<AllowedAccountConstraint> allowedAccountConstraints){
        SimpleTransactionComponent transactionInitComponent = new SimpleTransactionComponent();
        AccountConstraintsTransactionComponent accountConstraintsTransactionComponent = new AccountConstraintsTransactionComponent(transactionInitComponent, allowedAccountConstraints);
        InitRequest accountConstraintInitRequest = accountConstraintsTransactionComponent.getInitRequest(this.merchantConfig.getMerchantId(), transactionId,amount);
        return initiateTransaction(accountConstraintInitRequest, "","","","",1);
    }

    public InitResponse initiateTransaction(final InitRequest initRequest,
                                            final String redirectMode,
                                            final String redirectURL,
                                            final String callbackURL,
                                            final String callMode,
                                            final int keyIndex){


        InitResponse initResponse = null;
        try{

            // TODO: Move this into a separate utils class.
            final String jsonBody = new ObjectMapper().setSerializationInclusion(JsonInclude.Include.NON_EMPTY).writeValueAsString(initRequest);
            final String b64RequestBody = ChecksumUtils.encodeBase64(jsonBody);
            final String apiURL = getAPIURL();
            final String apiKey = getAPIKeyFromIndex(keyIndex);
            final String checksumBody = ChecksumUtils.generateChecksumBody(b64RequestBody, apiURL, apiKey);
            final String checksumHeader = ChecksumUtils.generateChecksumHeader(checksumBody, keyIndex);
            List<HttpHeaderPair> httpHeaders = getHttpHeaders(checksumHeader,
                                                              this.merchantConfig.getProviderId(),
                                                              redirectURL,
                                                              redirectMode,
                                                              callbackURL,
                                                              callMode);
            Base64Request base64Request = Base64Request.builder()
                                                       .request(b64RequestBody)
                                                       .build();
            initResponse =  performTransactionInit(base64Request,
                                                   httpHeaders,
                                                   apiURL);
        } catch (Exception ex){
            log.error("Exception occurred");
        }
        return initResponse;
    }

    protected abstract InitRequest buildInitRequest(final String merchantId,
                                                    final String transactionId,
                                                    final String userId,
                                                    final Long amount,
                                                    final String orderId);

    private InitResponse performTransactionInit(final Base64Request base64Request,
                                                final List<HttpHeaderPair> httpHeaders,
                                                final String apiURL) throws Exception{
        try{
            return this.transactionClient.initTransaction(base64Request,httpHeaders,apiURL);
        } catch (Exception ex){
            throw new PhonePeException("Could not perform transaction Initiation");
        }
    }

    private String getAPIURL(){
        return String.format("/%s/debit",
                             this.initConfig.getApiVersion().getApiVersion());
    }

    private List<HttpHeaderPair> getHttpHeaders(final String checksumHeader,
                                                final String providerId,
                                                final String redirectURL,
                                                final String redirectMode,
                                                final String callbackURL,
                                                final String callMode){
        List<HttpHeaderPair> headerPairs = new ArrayList<HttpHeaderPair>();
        HttpHeaderPair checksumHeaderPair = HttpUtils.getHeaderPair("X-VERIFY", checksumHeader);
        headerPairs.add(checksumHeaderPair);
        HttpHeaderPair contentTypeHeaderPair = HttpUtils.getContentTypeHeaderPair();
        headerPairs.add(contentTypeHeaderPair);
        if(StringUtils.isNotEmpty(providerId)){
            HttpHeaderPair providerIdHeader = HttpUtils.getHeaderPair("X-PROVIDER-ID", providerId);
            headerPairs.add(providerIdHeader);
        }
        if(StringUtils.isNotEmpty(redirectURL)){
            HttpHeaderPair redirectURLHeader = HttpUtils.getHeaderPair("X-REDIRECT-URL", redirectURL);
            headerPairs.add(redirectURLHeader);
        }
        if(StringUtils.isNotEmpty(redirectMode)){
            HttpHeaderPair redirectModeHeader = HttpUtils.getHeaderPair("X-REDIRECT-MODE", redirectMode);
            headerPairs.add(redirectModeHeader);
        }
        if(StringUtils.isNotEmpty(callbackURL)){
            HttpHeaderPair callbackURLHeader = HttpUtils.getHeaderPair("X-CALLBACK-URL", callbackURL);
            headerPairs.add(callbackURLHeader);
        }
        if(StringUtils.isNotEmpty(callMode)){
            HttpHeaderPair callbackModeHeader = HttpUtils.getHeaderPair("X-CALL-MODE", callMode);
            headerPairs.add(callbackModeHeader);
        }
        return headerPairs;
    }

    private String getAPIKeyFromIndex(int keyIndex) throws Exception{
        for(APIKeyConfig apiKey : this.merchantConfig.getApiKeys()){
            if(apiKey.getKeyIndex() == keyIndex)
                return apiKey.getKeyValue();
        }
        throw new PhonePeException();
    }


}
