package com.phonepe.sdk.javasdk.transaction.callback;

import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.base.Preconditions;
import com.phonepe.sdk.javasdk.config.models.MerchantConfig;
import com.phonepe.sdk.javasdk.http.PhonePeException;
import com.phonepe.sdk.javasdk.http.models.PhonePeHttpResponse;
import com.phonepe.sdk.javasdk.transaction.status.TransactionStatusChecker;
import com.phonepe.sdk.javasdk.transaction.status.models.StatusResponse;
import com.phonepe.sdk.javasdk.transaction.status.models.TransactionStatusResponse;
import com.phonepe.sdk.javasdk.utils.Base64Utils;
import com.phonepe.sdk.javasdk.utils.ChecksumUtils;
import com.phonepe.sdk.javasdk.utils.KeyUtils;
import lombok.Builder;
import lombok.extern.slf4j.Slf4j;

@Slf4j
public class CallbackHandler {

    private TransactionStatusChecker transactionStatusChecker;

    private MerchantConfig merchantConfig;

    @Builder
    public CallbackHandler(final TransactionStatusChecker transactionStatusChecker,
                           final MerchantConfig merchantConfig) {
        this.transactionStatusChecker = transactionStatusChecker;
        this.merchantConfig = merchantConfig;
    }

    public StatusResponse handleCallback(final String responseReceived, final String checksum) throws PhonePeException{
        Preconditions.checkNotNull(responseReceived);
        Preconditions.checkNotNull(checksum);
        try{
            final String responseBody = Base64Utils.decodeBase64(responseReceived);
            final PhonePeHttpResponse<TransactionStatusResponse> transactionStatusResponse = new ObjectMapper().readValue(responseBody, new
                    TypeReference<PhonePeHttpResponse<TransactionStatusResponse>>() {});
            final int keyIndex = KeyUtils.getKeyIndexFromChecksum(checksum);
            final TransactionStatusResponse statusResponseData = transactionStatusResponse.getData();
            final String merchantId = statusResponseData.getMerchantId();
            final Long callbackAmount = statusResponseData.getAmount();
            if(!isValidChecksum(responseReceived, checksum, keyIndex) || !isValidMerchantId(merchantId)){
                throw new PhonePeException("");
            }
            final String transactionId = transactionStatusResponse.getData().getTransactionId();
            StatusResponse statusResponse =  this.transactionStatusChecker.checkTransactionStatus(transactionId, keyIndex);
            if(!isValidTransactionAmount(callbackAmount, statusResponse.getAmount())){
                throw new PhonePeException("Transaction amount mismatch in callback response and status check response");
            }
            return statusResponse;
        }catch (Exception ex){
            log.error("Exception occurred while performing Callback handle");
            throw new PhonePeException(ex.getMessage(), ex);
        }
    }

    private boolean isValidMerchantId(final String merchantId){
        return this.merchantConfig.getMerchantId().equals(merchantId);
    }

    private boolean isValidChecksum(final String responseBody, final String checksum, final int keyIndex) throws Exception{
        final String apiKey = KeyUtils.getAPIKeyFromIndex(this.merchantConfig.getApiKeys(),keyIndex);
        final String generatedChecksum = ChecksumUtils.generateChecksumBody(responseBody,"", apiKey);
        return checksum.equalsIgnoreCase(generatedChecksum);
    }

    private boolean isValidTransactionAmount(final Long callbackAmount, final Long statusResponseAmount){
        return callbackAmount.equals(statusResponseAmount);
    }

}