package com.phonepe.sdk.javasdk;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.base.Preconditions;
import com.phonepe.sdk.javasdk.config.models.HttpClientConfig;
import com.phonepe.sdk.javasdk.config.models.HystrixConfig;
import com.phonepe.sdk.javasdk.config.models.InitConfig;
import com.phonepe.sdk.javasdk.config.models.MerchantConfig;
import com.phonepe.sdk.javasdk.config.models.SDKConfig;
import com.phonepe.sdk.javasdk.config.models.StatusConfig;
import com.phonepe.sdk.javasdk.http.PhonePeException;
import com.phonepe.sdk.javasdk.http.PhonePeHttpClientFactory;
import com.phonepe.sdk.javasdk.transaction.client.TransactionClient;
import com.phonepe.sdk.javasdk.transaction.init.models.AllowedAccountConstraint;
import com.phonepe.sdk.javasdk.transaction.init.web.WebTransactionInitiator;
import com.phonepe.sdk.javasdk.transaction.status.TransactionStatusChecker;
import com.phonepe.sdk.javasdk.transaction.callback.CallbackHandler;
import com.phonepe.sdk.javasdk.transaction.init.models.InitResponse;
import com.phonepe.sdk.javasdk.transaction.init.TransactionInitiator;
import com.phonepe.sdk.javasdk.transaction.status.models.StatusResponse;
import lombok.Builder;
import lombok.extern.slf4j.Slf4j;
import okhttp3.OkHttpClient;

import java.util.List;
import java.util.Set;

/**
 * Facade that wraps the {@link com.phonepe.sdk.javasdk.transaction.init.TransactionInitiator} and
 * {@link com.phonepe.sdk.javasdk.transaction.callback.CallbackHandler} and
 * {@link com.phonepe.sdk.javasdk.transaction.status.TransactionStatusChecker} for the
 * user.
 */
@Slf4j
public class PhonePeClient {

    private TransactionInitiator transactionInitiator;

    private CallbackHandler callbackHandler;

    private TransactionStatusChecker statusChecker;

    @Builder
    public PhonePeClient(final SDKConfig sdkConfig) {
        Preconditions.checkNotNull(sdkConfig);
        init(sdkConfig);
    }

    public void init(final SDKConfig sdkConfig) {
        ObjectMapper objectMapper = new ObjectMapper();
        HttpClientConfig httpClientConfig = sdkConfig.getHttpClientConfig();
        StatusConfig statusConfig = sdkConfig.getStatusConfig();
        MerchantConfig merchantConfig = sdkConfig.getMerchantConfig();
        InitConfig initConfig = sdkConfig.getInitConfig();
        HystrixConfig hystrixConfig = sdkConfig.getHystrixConfig();
        OkHttpClient client = PhonePeHttpClientFactory.buildOkHttpClient(httpClientConfig);
        TransactionClient transactionClient = new TransactionClient(httpClientConfig,objectMapper,client, hystrixConfig);
        this.statusChecker  =    TransactionStatusChecker.builder()
                                                      .merchantConfig(merchantConfig)
                                                      .statusConfig(statusConfig)
                                                      .transactionClient(transactionClient)
                                                      .build();

        this.callbackHandler =  CallbackHandler.builder()
                                               .merchantConfig(merchantConfig)
                                               .transactionStatusChecker(this.statusChecker)
                                               .build();
        this.transactionInitiator = WebTransactionInitiator.builder()
                                                           .initConfig(initConfig)
                                                           .merchantConfig(merchantConfig)
                                                           .transactionClient(transactionClient)
                                                           .build();
    }

    public InitResponse initTransaction(final String transactionId,
                                        final Long amount) throws PhonePeException{
        return initTransaction(transactionId, amount, null,null);
    }

    public InitResponse initTransaction(final String transactionId,
                                        final Long amount,
                                        final Set<String> offerTags) throws PhonePeException{
        return initTransaction(transactionId, amount, offerTags,null);
    }

    public InitResponse initTransaction(final String transactionId,
                                        final Long amount,
                                        final List<AllowedAccountConstraint> allowedAccountConstraints) throws PhonePeException{
        return initTransaction(transactionId, amount, null,allowedAccountConstraints);
    }

    public InitResponse initTransaction(final String transactionId,
                                        final Long amount,
                                        final Set<String> offerTags,
                                        final List<AllowedAccountConstraint> allowedAccountConstraints) throws PhonePeException{
        return initTransaction(transactionId, amount, "","","","","",
                               "","", offerTags, allowedAccountConstraints);
    }

    public InitResponse initTransaction(final String transactionId,
                                        final Long amount,
                                        final String orderId,
                                        final String userId,
                                        final String subMerchant,
                                        final String subMerchantId,
                                        final String mobileNumber,
                                        final String emailId,
                                        final String shortName,
                                        final Set<String> offerTags,
                                        final List<AllowedAccountConstraint> allowedAccountConstraints) throws PhonePeException {
        return initTransaction(transactionId, amount, orderId, userId,subMerchant, subMerchantId, mobileNumber,
                               emailId, shortName,offerTags, allowedAccountConstraints, "",
                               "", "", "");
    }


    public InitResponse initTransaction(final String transactionId,
                                        final Long amount,
                                        final String orderId,
                                        final String userId,
                                        final String subMerchant,
                                        final String subMerchantId,
                                        final String mobileNumber,
                                        final String emailId,
                                        final String shortName,
                                        final Set<String> offerTags,
                                        final List<AllowedAccountConstraint> allowedAccountConstraints,
                                        final String redirectMode,
                                        final String redirectURL,
                                        final String callbackURL,
                                        final String callMode) throws PhonePeException{
        return initTransaction(transactionId, amount, orderId, userId, subMerchant, subMerchantId, mobileNumber, emailId, shortName,
                               offerTags, allowedAccountConstraints, redirectMode,redirectURL,callMode,callbackURL,1);
    }

    public InitResponse initTransaction(final String transactionId,
                                        final Long amount,
                                        final String orderId,
                                        final String userId,
                                        final String subMerchant,
                                        final String subMerchantId,
                                        final String mobileNumber,
                                        final String emailId,
                                        final String shortName,
                                        final Set<String> offerTags,
                                        final List<AllowedAccountConstraint> allowedAccountConstraints,
                                        final String redirectMode,
                                        final String redirectURL,
                                        final String callMode,
                                        final String callbackURL,
                                        final int keyIndex) throws PhonePeException {
        return this.transactionInitiator.initiateTransaction(transactionId, amount, orderId, userId, subMerchant, subMerchantId, mobileNumber,
                                                             emailId, shortName, offerTags, allowedAccountConstraints,
                                                             redirectMode, redirectURL, callMode, callbackURL, keyIndex);
    }


    public StatusResponse handleCallback(final String responseReceived, final String checksum) throws PhonePeException{
        return this.callbackHandler.handleCallback(responseReceived, checksum);
    }

    public StatusResponse checkTransactionStatus(final String transactionId){
        return this.statusChecker.checkTransactionStatus(transactionId);
    }

}