package com.phonepe.sdk.javasdk.transaction.init;

import com.phonepe.sdk.javasdk.config.models.InitConfig;
import com.phonepe.sdk.javasdk.config.models.MerchantConfig;
import com.phonepe.sdk.javasdk.exception.PhonePeClientException;
import com.phonepe.sdk.javasdk.http.models.HttpHeaderPair;
import com.phonepe.sdk.javasdk.http.models.PhonePeHttpRequest;
import com.phonepe.sdk.javasdk.http.utils.HttpUtils;
import com.phonepe.sdk.javasdk.transaction.checksum.payload.ChecksumPayload;
import com.phonepe.sdk.javasdk.transaction.client.TransactionClient;
import com.phonepe.sdk.javasdk.transaction.init.models.AllowedAccountConstraint;
import com.phonepe.sdk.javasdk.transaction.request.PhonePeHttpRequestCreator;
import com.phonepe.sdk.javasdk.utils.ChecksumUtils;
import com.phonepe.sdk.javasdk.transaction.init.models.InitRequest;
import com.phonepe.sdk.javasdk.transaction.init.models.InitResponse;
import com.phonepe.sdk.javasdk.utils.KeyUtils;
import lombok.Data;
import lombok.extern.slf4j.Slf4j;

import java.lang.reflect.ParameterizedType;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * Template Method class for initiating the transaction for any given type of flow.
 */
@Data
@Slf4j
public abstract class TransactionInitiator {

    protected TransactionClient transactionClient;

    protected InitConfig initConfig;

    protected MerchantConfig merchantConfig;

    protected ChecksumPayload checksumPayload;

    protected PhonePeHttpRequestCreator<InitRequest> phonePeHttpRequestCreator;

    protected String apiURL;

    public TransactionInitiator(final TransactionClient transactionClient,
                                final InitConfig initConfig,
                                final MerchantConfig merchantConfig,
                                final ChecksumPayload checksumPayload,
                                final PhonePeHttpRequestCreator<InitRequest> phonePeHttpRequestCreator) {
        this.transactionClient = transactionClient;
        this.initConfig = initConfig;
        this.merchantConfig = merchantConfig;
        this.checksumPayload = checksumPayload;
        this.phonePeHttpRequestCreator = phonePeHttpRequestCreator;
        this.apiURL = String.format("/%s/debit", this.initConfig.getApiVersion());
    }

    public InitResponse initiateTransaction(final String transactionId,
                                            final Long amount,
                                            final String merchantOrderId,
                                            final String merchantUserId,
                                            final String subMerchant,
                                            final String subMerchantId,
                                            final String mobileNumber,
                                            final String emailId,
                                            final String shortName,
                                            final String message,
                                            final Set<String> offerTags,
                                            final List<AllowedAccountConstraint> allowedAccountConstraints,
                                            final String redirectMode,
                                            final String redirectURL,
                                            final String callbackMode,
                                            final String callbackURL,
                                            final int keyIndex) throws PhonePeClientException{

        try{
            final PhonePeHttpRequest<?> initRequest = buildInitRequest(this.merchantConfig.getMerchantId(),
                                                                       transactionId,
                                                                       amount,
                                                                       merchantOrderId,
                                                                       merchantUserId,
                                                                       subMerchant,
                                                                       subMerchantId,
                                                                       mobileNumber,
                                                                       emailId,
                                                                       shortName,
                                                                       message,
                                                                       offerTags,
                                                                       allowedAccountConstraints);
            final String apiKey = KeyUtils.getAPIKeyFromIndex(this.merchantConfig.getApiKeys(),keyIndex);
            final List<String> params = getParamsList(transactionId,amount);
            final String checksumPayload = this.checksumPayload.getChecksumPayload(initRequest, this.apiURL, params);
            final String checksumBody = ChecksumUtils.generateChecksumValue(checksumPayload,"",apiKey);
            final String checksumHeader = ChecksumUtils.generateChecksumHeader(checksumBody, keyIndex);
            return initiateTransaction(initRequest, redirectMode, redirectURL, callbackMode, callbackURL, checksumHeader);
        } catch (PhonePeClientException e){
            log.error("Error while making call to PhonePe init transaction API");
            throw e;
        } catch (Exception e) {
            log.error("Error in calling PhonePe init transaction API", e);
            Map<String, Object> objectMap = new HashMap<String, Object>();
            objectMap.put("MESSAGE", e.getMessage());
            throw new PhonePeClientException(PhonePeClientException.ErrorCode.EXECUTION_ERROR,
                                             "Error executing initiate transaction"
                                             + ": " + e.getMessage(), objectMap, e);
        }
    }

    private List<String> getParamsList(String transactionId, Long amount){
        List<String> paramsList =  new ArrayList<String>();
        paramsList.add(this.merchantConfig.getMerchantId());
        paramsList.add(transactionId);
        paramsList.add(Long.toString(amount));
        return paramsList;
    }
    private InitResponse initiateTransaction(final PhonePeHttpRequest<?> phonePeHttpRequest,
                                             final String redirectMode,
                                             final String redirectURL,
                                             final String callMode,
                                             final String callbackURL,
                                             final String checksumHeader) throws PhonePeClientException{

        try{
            final List<HttpHeaderPair> httpHeaders = HttpUtils.getHttpHeaders(checksumHeader,
                                                                              this.merchantConfig.getProviderId(),
                                                                              redirectURL,
                                                                              redirectMode,
                                                                              callbackURL,
                                                                              callMode);
            return this.transactionClient.initTransaction(phonePeHttpRequest,httpHeaders,this.apiURL);
        } catch (PhonePeClientException e){
            log.error("Error while making call to PhonePe init transaction API");
            throw e;
        } catch (Exception e) {
            log.error("Error in calling PhonePe init transaction API", e);
            Map<String, Object> objectMap = new HashMap<String, Object>();
            objectMap.put("MESSAGE", e.getMessage());
            throw new PhonePeClientException(PhonePeClientException.ErrorCode.EXECUTION_ERROR,
                                             "Error executing initiate transaction"
                                             + ": " + e.getMessage(), objectMap, e);
        }
    }

    protected abstract PhonePeHttpRequest<?> buildInitRequest(final String merchantId,
                                                              final String transactionId,
                                                              final Long amount,
                                                              final String merchantOrderId,
                                                              final String merchantUserId,
                                                              final String subMerchant,
                                                              final String subMerchantId,
                                                              final String mobileNumber,
                                                              final String emailId,
                                                              final String shortName,
                                                              final String message,
                                                              final Set<String> offerTags,
                                                              final List<AllowedAccountConstraint> allowedAccountConstraints) throws Exception;


}
