package com.phonepe.sdk.javasdk;

import com.google.common.base.Preconditions;
import com.phonepe.sdk.javasdk.config.models.Endpoint;
import com.phonepe.sdk.javasdk.config.models.HttpClientConfig;
import com.phonepe.sdk.javasdk.config.models.HystrixConfig;
import com.phonepe.sdk.javasdk.config.models.InitConfig;
import com.phonepe.sdk.javasdk.config.models.MerchantConfig;
import com.phonepe.sdk.javasdk.config.models.SDKConfig;
import com.phonepe.sdk.javasdk.config.models.StatusConfig;
import com.phonepe.sdk.javasdk.exception.PhonePeClientException;
import com.phonepe.sdk.javasdk.http.PhonePeHttpClientFactory;
import com.phonepe.sdk.javasdk.transaction.checksum.ChecksumGeneratorFactory;
import com.phonepe.sdk.javasdk.transaction.checksum.payload.ChecksumGenerator;
import com.phonepe.sdk.javasdk.transaction.client.TransactionClient;
import com.phonepe.sdk.javasdk.transaction.init.TransactionInitiatorFactory;
import com.phonepe.sdk.javasdk.transaction.init.models.AllowedAccountConstraint;
import com.phonepe.sdk.javasdk.transaction.init.models.InitRequest;
import com.phonepe.sdk.javasdk.transaction.request.PhonePeHttpRequestCreator;
import com.phonepe.sdk.javasdk.transaction.request.PhonePeHttpRequestCreatorFactory;
import com.phonepe.sdk.javasdk.transaction.status.TransactionStatusChecker;
import com.phonepe.sdk.javasdk.transaction.callback.CallbackHandler;
import com.phonepe.sdk.javasdk.transaction.init.models.InitResponse;
import com.phonepe.sdk.javasdk.transaction.init.TransactionInitiator;
import com.phonepe.sdk.javasdk.transaction.status.models.StatusResponse;
import lombok.Builder;
import lombok.Data;
import lombok.extern.slf4j.Slf4j;
import okhttp3.OkHttpClient;
import java.util.List;
import java.util.Set;

/**
 * Facade that wraps the {@link com.phonepe.sdk.javasdk.transaction.init.TransactionInitiator} and
 * {@link com.phonepe.sdk.javasdk.transaction.callback.CallbackHandler} and
 * {@link com.phonepe.sdk.javasdk.transaction.status.TransactionStatusChecker} for the
 * user.
 */
@Data
@Slf4j
public class PhonePeClient {

    private TransactionInitiator transactionInitiator;

    private CallbackHandler callbackHandler;

    private TransactionStatusChecker statusChecker;

    @Builder
    public PhonePeClient(final SDKConfig sdkConfig) {
        Preconditions.checkNotNull(sdkConfig);
        init(sdkConfig);
    }

    private void init(final SDKConfig sdkConfig) {
        HttpClientConfig httpClientConfig = sdkConfig.getHttpClientConfig();
        Endpoint endpoint = sdkConfig.getEndpoint();
        StatusConfig statusConfig = sdkConfig.getStatusConfig();
        MerchantConfig merchantConfig = sdkConfig.getMerchantConfig();
        InitConfig initConfig = sdkConfig.getInitConfig();
        HystrixConfig hystrixConfig = sdkConfig.getHystrixConfig();
        ChecksumGeneratorFactory checksumGeneratorFactory = new ChecksumGeneratorFactory();
        OkHttpClient client = PhonePeHttpClientFactory.buildOkHttpClient(httpClientConfig,endpoint.getEnvironment().isSecured());
        TransactionClient transactionClient = new TransactionClient(endpoint,client, hystrixConfig);
        ChecksumGenerator statusChecksumGenerator = statusConfig.getApiVersion().accept(checksumGeneratorFactory);
        ChecksumGenerator initChecksumGenerator = initConfig.getApiVersion().accept(checksumGeneratorFactory);

        PhonePeHttpRequestCreatorFactory<InitRequest> phonePeHttpRequestCreatorFactory = new PhonePeHttpRequestCreatorFactory<InitRequest>();
        PhonePeHttpRequestCreator<InitRequest> phonePeHttpRequestCreator = initConfig.getApiVersion().accept(phonePeHttpRequestCreatorFactory);


        TransactionInitiatorFactory transactionInitiatorFactory = TransactionInitiatorFactory.builder()
                                                                                             .merchantConfig(merchantConfig)
                                                                                             .initConfig(initConfig)
                                                                                             .transactionClient(transactionClient)
                                                                                             .checksumGenerator(initChecksumGenerator)
                                                                                             .phonePeHttpRequestCreator(phonePeHttpRequestCreator)
                                                                                             .build();
        this.statusChecker  =    TransactionStatusChecker.builder()
                                                         .merchantConfig(merchantConfig)
                                                         .statusConfig(statusConfig)
                                                         .transactionClient(transactionClient)
                                                         .checksumGenerator(statusChecksumGenerator)
                                                         .build();

        this.callbackHandler =  CallbackHandler.builder()
                                               .merchantConfig(merchantConfig)
                                               .transactionStatusChecker(this.statusChecker)
                                               .build();

        this.transactionInitiator = initConfig.getInitType().accept(transactionInitiatorFactory);
    }

    public InitResponse initTransaction(final String transactionId,
                                        final Long amount) throws PhonePeClientException {
        return initTransaction(transactionId, "", amount);
    }

    public InitResponse initTransaction(final String transactionId,
                                        final String userId,
                                        final Long amount) throws PhonePeClientException {
        return initTransaction(transactionId, userId, amount, null,null);
    }

    public InitResponse initTransaction(final String transactionId,
                                        final Long amount,
                                        final Set<String> offerTags) throws PhonePeClientException{
        return initTransaction(transactionId, "", amount, offerTags);
    }

    public InitResponse initTransaction(final String transactionId,
                                        final String userId,
                                        final Long amount,
                                        final Set<String> offerTags) throws PhonePeClientException{
        return initTransaction(transactionId, userId, amount, offerTags,null);
    }

    public InitResponse initTransaction(final String transactionId,
                                        final Long amount,
                                        final List<AllowedAccountConstraint> allowedAccountConstraints) throws PhonePeClientException{
        return initTransaction(transactionId, "", amount, allowedAccountConstraints);
    }

    public InitResponse initTransaction(final String transactionId,
                                        final String userId,
                                        final Long amount,
                                        final List<AllowedAccountConstraint> allowedAccountConstraints) throws PhonePeClientException{
        return initTransaction(transactionId, userId, amount, null,allowedAccountConstraints);
    }

    public InitResponse initTransaction(final String transactionId,
                                        final String userId,
                                        final Long amount,
                                        final Set<String> offerTags,
                                        final List<AllowedAccountConstraint> allowedAccountConstraints) throws PhonePeClientException{
        return initTransaction(transactionId, amount, "",userId,"","","",
                               "","","", offerTags, allowedAccountConstraints);
    }

    public InitResponse initTransaction(final String transactionId,
                                        final Long amount,
                                        final String orderId,
                                        final String userId,
                                        final String subMerchant,
                                        final String subMerchantId,
                                        final String mobileNumber,
                                        final String emailId,
                                        final String shortName,
                                        final String message,
                                        final Set<String> offerTags,
                                        final List<AllowedAccountConstraint> allowedAccountConstraints) throws PhonePeClientException {
        return initTransaction(transactionId, amount, orderId, userId,subMerchant, subMerchantId, mobileNumber,
                               emailId, shortName, message, offerTags, allowedAccountConstraints, "",
                               "", "", "");
    }


    public InitResponse initTransaction(final String transactionId,
                                        final Long amount,
                                        final String orderId,
                                        final String userId,
                                        final String subMerchant,
                                        final String subMerchantId,
                                        final String mobileNumber,
                                        final String emailId,
                                        final String shortName,
                                        final String message,
                                        final Set<String> offerTags,
                                        final List<AllowedAccountConstraint> allowedAccountConstraints,
                                        final String redirectMode,
                                        final String redirectURL,
                                        final String callbackURL,
                                        final String callMode) throws PhonePeClientException{
        return initTransaction(transactionId, amount, orderId, userId, subMerchant, subMerchantId, mobileNumber, emailId, shortName, message,
                               offerTags, allowedAccountConstraints, redirectMode,redirectURL,callMode,callbackURL,1);
    }

    public InitResponse initTransaction(final String transactionId,
                                        final Long amount,
                                        final String orderId,
                                        final String userId,
                                        final String subMerchant,
                                        final String subMerchantId,
                                        final String mobileNumber,
                                        final String emailId,
                                        final String shortName,
                                        final String message,
                                        final Set<String> offerTags,
                                        final List<AllowedAccountConstraint> allowedAccountConstraints,
                                        final String redirectMode,
                                        final String redirectURL,
                                        final String callMode,
                                        final String callbackURL,
                                        final int keyIndex) throws PhonePeClientException {
        return this.transactionInitiator.initiateTransaction(transactionId, amount, orderId, userId, subMerchant, subMerchantId, mobileNumber,
                                                             emailId, shortName, message, offerTags, allowedAccountConstraints,
                                                             redirectMode, redirectURL, callMode, callbackURL, keyIndex);
    }

    public StatusResponse handleCallback(final String responseReceived, final String checksum,
                                         final long transactionAmount) throws PhonePeClientException{
        return this.callbackHandler.handleCallback(responseReceived, checksum, transactionAmount);
    }

    public StatusResponse checkTransactionStatus(final String transactionId) throws PhonePeClientException{
        return checkTransactionStatus(transactionId, 1);
    }

    public StatusResponse checkTransactionStatus(final String transactionId, final int keyIndex) throws PhonePeClientException{
        return this.statusChecker.checkTransactionStatus(transactionId, keyIndex);
    }

}