package com.phonepe.sdk.javasdk.http;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.base.Preconditions;
import com.netflix.hystrix.HystrixCommand;
import com.netflix.hystrix.HystrixCommandGroupKey;
import com.netflix.hystrix.HystrixCommandProperties;
import com.netflix.hystrix.HystrixThreadPoolProperties;
import com.phonepe.sdk.javasdk.config.models.Endpoint;
import com.phonepe.sdk.javasdk.config.models.HystrixConfig;
import com.phonepe.sdk.javasdk.exception.PhonePeClientException;
import com.phonepe.sdk.javasdk.http.models.ExtractedResponse;
import com.phonepe.sdk.javasdk.http.models.HttpHeaderPair;
import com.phonepe.sdk.javasdk.http.models.PhonePeHttpRequest;
import com.phonepe.sdk.javasdk.http.utils.HttpUtils;
import lombok.Singular;
import lombok.extern.slf4j.Slf4j;
import okhttp3.*;
import org.apache.commons.lang3.ObjectUtils;

import java.nio.charset.Charset;
import java.util.List;

import static com.phonepe.sdk.javasdk.http.utils.HttpUtils.checkNotNull;


@Slf4j
public abstract class BaseHttpCommand<T,J> {

    /* create and initialize this okhttp client */
    OkHttpClient client;

    /* this is used for logging and as a hystrix grouping key */
    String commandName;

    /* endpoint */
    Endpoint endpoint;

    /* final url to be hit */
    String url;

    /* list of header keyValue pairs */
    @Singular("header")
    List<HttpHeaderPair> headers = null;

    /* post data for http request */
    PhonePeHttpRequest<J> httpRequest;

    /* object mapper */
    ObjectMapper mapper;

    /* response type class */
    Class<T> responseType;

    /* response type reference class */
    TypeReference<T> responseTypeReference;

    Consumer<Exception, T> exceptionConsumer;

    Consumer<ExtractedResponse, T> nonSuccessResponseConsumer;

    HystrixConfig hystrixConfig;

    BaseHttpCommand(OkHttpClient client, HystrixConfig hystrixConfig, String commandName,
                    Endpoint endpoint, String url,
                    List<HttpHeaderPair> headers, PhonePeHttpRequest<J> httpRequest,
                    ObjectMapper mapper, Class<T> responseType,
                    Consumer<Exception, T> exceptionConsumer,
                    Consumer<ExtractedResponse, T> nonSuccessResponseConsumer,
                    TypeReference<T> responseTypeReference) {
        this.client = client;
        this.hystrixConfig = hystrixConfig;
        this.commandName = commandName;
        this.endpoint = endpoint;
        this.url = url;
        this.headers = headers;
        this.httpRequest = httpRequest;
        this.mapper = mapper;
        this.responseType = responseType;
        this.exceptionConsumer = exceptionConsumer;
        this.nonSuccessResponseConsumer = nonSuccessResponseConsumer;
        this.responseTypeReference = responseTypeReference;
    }

    RequestBody getRequestBody() throws JsonProcessingException {
        return RequestBody.create(MediaType.parse(HttpUtils.APPLICATION_JSON), mapper.writeValueAsBytes(this.httpRequest.getData()));
    }

    public abstract Request getRequest(HttpUrl httpUrl) throws JsonProcessingException;

    public void buildHeaders(Request.Builder requestBuilder){
        if (ObjectUtils.isNotEmpty(headers)) {
            for (HttpHeaderPair httpHeader : headers) {
                requestBuilder.header(httpHeader.getName(), httpHeader.getValue());
            }
        }
    }

    public T execute() throws PhonePeClientException {
        preconditions();
        Request request = null;
        try {
            final HttpUrl httpUrl = endpoint.getUrl(url);
            request = getRequest(httpUrl);
            log.info("Service {} request: {}", commandName, request);
            final Response response = client.newCall(request).execute();

            /* OkHttpUtils.body ensures that the response is closed */
            final byte[] responseBody = HttpUtils.body(response);
            if (!response.isSuccessful()) {
                if (nonSuccessResponseConsumer != null) {
                    return nonSuccessResponseConsumer.consume(extract(response, responseBody));
                }
                log.error(String.format("Service %s call failed statusCode:%d request:%s response:%s",
                                        commandName, response.code(), request, new String(responseBody)));
                throw new PhonePeClientException(PhonePeClientException.ErrorCode.HTTP_CLIENT_ERROR , "Service " + commandName + " call failure: " + response.code());
            }
      if (ObjectUtils.isNotEmpty(responseBody)) {
        if (responseTypeReference != null) {
          return mapper.readValue(responseBody, responseTypeReference);
        }
        if (byte[].class.equals(responseType)) {
          return (T) responseBody;
        }
        if (String.class.equals(responseType)) {
          return (T) new String(responseBody, Charset.defaultCharset());
        }
        return mapper.readValue(responseBody, responseType);
            }
        return null;
        } catch (Exception e) {
            if (exceptionConsumer != null) {
                return exceptionConsumer.consume(e);
            }
            log.error(String.format("Error executing %s request:%s error:%s", commandName, request, e.getMessage()), e);
            throw new PhonePeClientException(PhonePeClientException.ErrorCode.EXECUTION_ERROR,e.getMessage(), e);
        }
    }

    public String getCommand() {
        return commandName;
    }

    public void preconditions() throws PhonePeClientException{
        Preconditions.checkNotNull(client, "client cant be null");
        Preconditions.checkNotNull(endpoint, "endpoint cant be null");
        Preconditions.checkNotNull(url, "url cant be null");
        Preconditions.checkNotNull(mapper, "object mapper cant be null");
        checkNotNull(responseType, responseTypeReference, "both responseType and responseTypeReference cant be null");
    }

    private ExtractedResponse extract(Response response, byte[] responseBody) {
        return ExtractedResponse
                .builder().body(responseBody)
                .code(response.code())
                .headers(response.headers())
                .message(response.message())
                .protocol(response.protocol())
                .handshake(response.handshake())
                .receivedResponseAtMillis(response.receivedResponseAtMillis())
                .sentRequestAtMillis(response.sentRequestAtMillis())
                .build();
    }

    public T executeTracked(Class clazz) throws PhonePeClientException{
        try {
            return  hystrixExecutor(new HandlerAdapter<T>() {
                @Override
                public T run() throws Exception{
                    return execute();
                }
            }, commandKey(clazz.getSimpleName(), getCommand())).execute();
        } catch (Exception e) {
            if (this.exceptionConsumer != null) {
                return this.exceptionConsumer.consume(e);
            }
            throw new PhonePeClientException(PhonePeClientException.ErrorCode.EXECUTION_ERROR,e.getMessage(), e);
        }
    }

    public HystrixCommand<T> hystrixExecutor(final HandlerAdapter<T> function, final String key) {
        return new HystrixCommand<T>(getSetter(hystrixConfig,key)) {
            protected T run() throws Exception{
                Object var1;
                var1 = function.run();
                return (T) var1;
            }
        };
    }

    private String commandKey(String group, String command) {
        return String.format("%s.%s", group, command);
    }

    private HystrixCommand.Setter getSetter(final HystrixConfig hystrixConfig, final String key){
        return HystrixCommand.Setter
                .withGroupKey(HystrixCommandGroupKey.Factory.asKey(key))
                .andCommandKey(com.netflix.hystrix.HystrixCommandKey.Factory.asKey(key))
                .andThreadPoolKey(com.netflix.hystrix.HystrixThreadPoolKey.Factory.asKey(key))
                .andCommandPropertiesDefaults(HystrixCommandProperties
                                                      .Setter()
                                                      .withExecutionIsolationStrategy(HystrixCommandProperties.ExecutionIsolationStrategy.THREAD)
                                                      .withExecutionIsolationSemaphoreMaxConcurrentRequests(hystrixConfig.getThreadPool().getConcurrency())
                                                      .withFallbackIsolationSemaphoreMaxConcurrentRequests(hystrixConfig.getThreadPool().getConcurrency())
                                                      .withFallbackEnabled(false).withCircuitBreakerErrorThresholdPercentage(hystrixConfig.getCircuitBreaker()
                                                                                                                                    .getErrorThreshold())
                                                      .withCircuitBreakerRequestVolumeThreshold(hystrixConfig.getCircuitBreaker().getAcceptableFailuresInWindow())
                                                      .withCircuitBreakerSleepWindowInMilliseconds(hystrixConfig.getCircuitBreaker().getWaitTimeBeforeRetry())
                                                      .withExecutionTimeoutInMilliseconds(hystrixConfig.getThreadPool().getTimeout())
                                                      .withMetricsHealthSnapshotIntervalInMilliseconds(hystrixConfig.getMetrics().getHealthCheckInterval())
                                                      .withMetricsRollingPercentileBucketSize(hystrixConfig.getMetrics().getPercentileBucketSize())
                                                      .withMetricsRollingPercentileWindowInMilliseconds(hystrixConfig.getMetrics().getPercentileTimeInMillis()))
                .andThreadPoolPropertiesDefaults(HystrixThreadPoolProperties
                                                         .Setter()
                                                         .withCoreSize(hystrixConfig.getThreadPool().getConcurrency())
                                                         .withMaxQueueSize(hystrixConfig.getThreadPool().getMaxRequestQueueSize())
                                                         .withQueueSizeRejectionThreshold(hystrixConfig.getThreadPool().getDynamicRequestQueueSize())
                                                         .withMetricsRollingStatisticalWindowBuckets(hystrixConfig.getMetrics().getNumBucketSize())
                                                         .withMetricsRollingStatisticalWindowInMilliseconds(hystrixConfig.getMetrics().getStatsTimeInMillis()));
    }
}

