package com.phonepe.sdk.javasdk;

import com.google.common.base.Preconditions;
import com.phonepe.sdk.javasdk.config.ConfigurationFactory;
import com.phonepe.sdk.javasdk.config.models.DefaultConfig;
import com.phonepe.sdk.javasdk.config.models.Endpoint;
import com.phonepe.sdk.javasdk.config.models.HttpClientConfig;
import com.phonepe.sdk.javasdk.config.models.InitConfig;
import com.phonepe.sdk.javasdk.config.models.MerchantConfig;
import com.phonepe.sdk.javasdk.config.models.PhonePeConfig;
import com.phonepe.sdk.javasdk.config.models.SDKConfig;
import com.phonepe.sdk.javasdk.config.models.StatusConfig;
import com.phonepe.sdk.javasdk.exception.PhonePeClientException;
import com.phonepe.sdk.javasdk.exception.PhonePeConfigurationException;
import com.phonepe.sdk.javasdk.http.PhonePeHttpClientFactory;
import com.phonepe.sdk.javasdk.transaction.checksum.ChecksumGeneratorFactory;
import com.phonepe.sdk.javasdk.transaction.checksum.ChecksumGenerator;
import com.phonepe.sdk.javasdk.transaction.client.TransactionClient;
import com.phonepe.sdk.javasdk.transaction.init.TransactionInitiatorFactory;
import com.phonepe.sdk.javasdk.transaction.init.models.InitRequest;
import com.phonepe.sdk.javasdk.transaction.models.InitParams;
import com.phonepe.sdk.javasdk.transaction.request.PhonePeHttpRequestCreator;
import com.phonepe.sdk.javasdk.transaction.request.PhonePeHttpRequestCreatorFactory;
import com.phonepe.sdk.javasdk.transaction.status.TransactionStatusChecker;
import com.phonepe.sdk.javasdk.transaction.callback.CallbackHandler;
import com.phonepe.sdk.javasdk.transaction.init.models.InitResponse;
import com.phonepe.sdk.javasdk.transaction.init.TransactionInitiator;
import com.phonepe.sdk.javasdk.transaction.status.models.StatusResponse;
import com.phonepe.sdk.javasdk.utils.KeyUtils;
import lombok.Builder;
import lombok.extern.slf4j.Slf4j;
import okhttp3.OkHttpClient;


/**
 * Facade that wraps the {@link com.phonepe.sdk.javasdk.transaction.init.TransactionInitiator} and
 * {@link com.phonepe.sdk.javasdk.transaction.callback.CallbackHandler} and
 * {@link com.phonepe.sdk.javasdk.transaction.status.TransactionStatusChecker} for the
 * user.
 */
@Slf4j
public class PhonePeClient {

    private TransactionInitiator transactionInitiator;
    private CallbackHandler callbackHandler;
    private TransactionStatusChecker statusChecker;
    private int defaultKeyIndex;

    @Builder
    public PhonePeClient(final SDKConfig sdkConfig) throws PhonePeConfigurationException{
        Preconditions.checkNotNull(sdkConfig);
        init(sdkConfig);
    }

    private void init(final SDKConfig sdkConfig) throws PhonePeConfigurationException {
        final PhonePeConfig phonePeConfig = sdkConfig.getPhonePeConfig();
        final DefaultConfig defaultConfig = ConfigurationFactory.<DefaultConfig>builder()
                                                                .clazz(DefaultConfig.class)
                                                                .build()
                                                                .buildConfig("default_properties.yml");
        final HttpClientConfig httpClientConfig = phonePeConfig.getHttpClientConfig();
        final Endpoint endpoint = phonePeConfig.getEndpoint();
        final StatusConfig statusConfig = phonePeConfig.getStatusConfig();
        final MerchantConfig merchantConfig = sdkConfig.getMerchantConfig();
        final InitConfig initConfig = phonePeConfig.getInitConfig();
        final ChecksumGeneratorFactory checksumGeneratorFactory = new ChecksumGeneratorFactory();
        final OkHttpClient client = PhonePeHttpClientFactory.buildOkHttpClient(httpClientConfig,endpoint.getEnvironment().isSecured());
        final TransactionClient transactionClient = new TransactionClient(endpoint,client);
        final ChecksumGenerator statusChecksumGenerator = statusConfig.getApiVersion().accept(checksumGeneratorFactory);
        final ChecksumGenerator initChecksumGenerator = initConfig.getApiVersion().accept(checksumGeneratorFactory);

        final PhonePeHttpRequestCreatorFactory<InitRequest> phonePeHttpRequestCreatorFactory = new PhonePeHttpRequestCreatorFactory<InitRequest>();
        final PhonePeHttpRequestCreator<InitRequest> phonePeHttpRequestCreator = initConfig.getApiVersion().accept(phonePeHttpRequestCreatorFactory);


        final TransactionInitiatorFactory transactionInitiatorFactory = TransactionInitiatorFactory.builder()
                                                                                                   .merchantConfig(merchantConfig)
                                                                                                   .initConfig(initConfig)
                                                                                                   .transactionClient(transactionClient)
                                                                                                   .checksumGenerator(initChecksumGenerator)
                                                                                                   .phonePeHttpRequestCreator(phonePeHttpRequestCreator)
                                                                                                   .build();
        this.statusChecker  =    TransactionStatusChecker.builder()
                                                         .merchantConfig(merchantConfig)
                                                         .statusConfig(statusConfig)
                                                         .transactionClient(transactionClient)
                                                         .checksumGenerator(statusChecksumGenerator)
                                                         .build();

        this.callbackHandler =  CallbackHandler.builder()
                                               .merchantConfig(merchantConfig)
                                               .transactionStatusChecker(this.statusChecker)
                                               .build();

        this.transactionInitiator = initConfig.getInitType().accept(transactionInitiatorFactory);
        this.defaultKeyIndex = KeyUtils.getDefaultKeyIndex(merchantConfig.getApiKeys());
    }

    public InitResponse initTransaction(final InitParams initParam) throws PhonePeClientException{
        return initTransaction(initParam, defaultKeyIndex);
    }

    public InitResponse initTransaction(final InitParams initParams, final int keyIndex) throws PhonePeClientException{
        return this.transactionInitiator.initiateTransaction(initParams, keyIndex);
    }

    public StatusResponse handleCallback(final String responseReceived, final String checksum,
                                        final long transactionAmount) throws PhonePeClientException{
        return this.callbackHandler.handleCallback(responseReceived, checksum, transactionAmount);
    }

    public StatusResponse checkTransactionStatus(final String transactionId) throws PhonePeClientException{
        return checkTransactionStatus(transactionId,defaultKeyIndex);
    }

    public StatusResponse checkTransactionStatus(final String transactionId, final int keyIndex) throws PhonePeClientException{
        return this.statusChecker.checkTransactionStatus(transactionId, keyIndex);
    }

}