package com.phonepe.sdk.javasdk.http;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.base.Preconditions;
import com.phonepe.sdk.javasdk.config.models.Endpoint;
import com.phonepe.sdk.javasdk.exception.PhonePeClientException;
import com.phonepe.sdk.javasdk.http.models.ExtractedResponse;
import com.phonepe.sdk.javasdk.http.models.HttpHeaderPair;
import com.phonepe.sdk.javasdk.http.models.PhonePeHttpRequest;
import com.phonepe.sdk.javasdk.http.utils.HttpUtils;
import lombok.Singular;
import lombok.extern.slf4j.Slf4j;
import okhttp3.*;
import org.apache.commons.lang3.ObjectUtils;

import java.nio.charset.Charset;
import java.util.List;

import static com.phonepe.sdk.javasdk.http.utils.HttpUtils.checkNotNull;


@Slf4j
public abstract class BaseHttpCommand<T,J> {

    /* create and initialize this okhttp client */
    OkHttpClient client;

    /* endpoint */
    Endpoint endpoint;

    /* final url to be hit */
    String url;

    /* list of header keyValue pairs */
    @Singular("header")
    List<HttpHeaderPair> headers = null;

    /* post data for http request */
    PhonePeHttpRequest<J> httpRequest;

    /* object mapper */
    ObjectMapper mapper;

    /* response type class */
    Class<T> responseType;

    /* response type reference class */
    TypeReference<T> responseTypeReference;

    Consumer<Exception, T> exceptionConsumer;

    Consumer<ExtractedResponse, T> nonSuccessResponseConsumer;

    BaseHttpCommand(OkHttpClient client, Endpoint endpoint, String url,
                    List<HttpHeaderPair> headers, PhonePeHttpRequest<J> httpRequest,
                    ObjectMapper mapper, Class<T> responseType,
                    Consumer<Exception, T> exceptionConsumer,
                    Consumer<ExtractedResponse, T> nonSuccessResponseConsumer,
                    TypeReference<T> responseTypeReference) {
        this.client = client;
        this.endpoint = endpoint;
        this.url = url;
        this.headers = headers;
        this.httpRequest = httpRequest;
        this.mapper = mapper;
        this.responseType = responseType;
        this.exceptionConsumer = exceptionConsumer;
        this.nonSuccessResponseConsumer = nonSuccessResponseConsumer;
        this.responseTypeReference = responseTypeReference;
    }

    RequestBody getRequestBody() throws JsonProcessingException {
        return RequestBody.create(MediaType.parse(HttpUtils.APPLICATION_JSON), mapper.writeValueAsBytes(this.httpRequest.getData()));
    }

    public abstract Request getRequest(HttpUrl httpUrl) throws JsonProcessingException;

    public void buildHeaders(Request.Builder requestBuilder){
        if (ObjectUtils.anyNotNull(headers)) {
            for (HttpHeaderPair httpHeader : headers) {
                requestBuilder.header(httpHeader.getName(), httpHeader.getValue());
            }
        }
    }

    public T execute() throws PhonePeClientException {
        preconditions();
        Request request = null;
        try {
            final HttpUrl httpUrl = endpoint.getUrl(url);
            request = getRequest(httpUrl);
            log.info("Service request: {}" , request);
            final Response response = client.newCall(request).execute();

            /* OkHttpUtils.body ensures that the response is closed */
            final byte[] responseBody = HttpUtils.body(response);
            if (!response.isSuccessful()) {
                if (nonSuccessResponseConsumer != null) {
                    return nonSuccessResponseConsumer.consume(extract(response, responseBody));
                }
                log.error(String.format("Service call failed statusCode:%d request:%s response:%s",
                                        response.code(), request, new String(responseBody)));
                throw new PhonePeClientException(PhonePeClientException.ErrorCode.HTTP_CLIENT_ERROR , "Service call failure: " + response.code());
            }
      if (ObjectUtils.anyNotNull(responseBody)) {
        if (responseTypeReference != null) {
          return mapper.readValue(responseBody, responseTypeReference);
        }
        if (byte[].class.equals(responseType)) {
          return (T) responseBody;
        }
        if (String.class.equals(responseType)) {
          return (T) new String(responseBody, Charset.defaultCharset());
        }
        return mapper.readValue(responseBody, responseType);
            }
        return null;
        } catch (Exception e) {
            if (exceptionConsumer != null) {
                return exceptionConsumer.consume(e);
            }
            log.error(String.format("Error executing request:%s error:%s", request, e.getMessage()), e);
            throw new PhonePeClientException(PhonePeClientException.ErrorCode.EXECUTION_ERROR,e.getMessage(), e);
        }
    }

    public void preconditions() throws PhonePeClientException{
        Preconditions.checkNotNull(client, "client cant be null");
        Preconditions.checkNotNull(endpoint, "endpoint cant be null");
        Preconditions.checkNotNull(url, "url cant be null");
        Preconditions.checkNotNull(mapper, "object mapper cant be null");
        checkNotNull(responseType, responseTypeReference, "both responseType and responseTypeReference cant be null");
    }

    private ExtractedResponse extract(Response response, byte[] responseBody) {
        return ExtractedResponse
                .builder().body(responseBody)
                .code(response.code())
                .headers(response.headers())
                .message(response.message())
                .protocol(response.protocol())
                .handshake(response.handshake())
                .receivedResponseAtMillis(response.receivedResponseAtMillis())
                .sentRequestAtMillis(response.sentRequestAtMillis())
                .build();
    }
}

