package com.phonepe.sdk.javasdk.http;

import com.google.common.base.Preconditions;
import com.phonepe.sdk.javasdk.config.models.HttpClientConfig;
import com.phonepe.sdk.javasdk.http.models.PhonePeHttpResponse;
import com.phonepe.sdk.javasdk.http.utils.HttpUtils;
import com.phonepe.sdk.javasdk.transaction.init.models.InitResponse;
import com.phonepe.sdk.javasdk.transaction.init.models.enums.RedirectType;
import com.phonepe.sdk.javasdk.utils.MapperUtils;
import lombok.extern.slf4j.Slf4j;
import okhttp3.ConnectionPool;
import okhttp3.ConnectionSpec;
import okhttp3.Dispatcher;
import okhttp3.Interceptor;
import okhttp3.MediaType;
import okhttp3.OkHttpClient;
import okhttp3.Request;
import okhttp3.Response;
import okhttp3.ResponseBody;
import okhttp3.TlsVersion;
import okhttp3.internal.tls.OkHostnameVerifier;

import java.io.IOException;
import java.util.Collections;
import java.util.concurrent.TimeUnit;

@Slf4j
public class PhonePeHttpClientFactory {

  private PhonePeHttpClientFactory() {
    throw new IllegalStateException("Factory class");
  }

  public static OkHttpClient buildOkHttpClient(final HttpClientConfig httpClientConfig,
                                               final boolean isSecured) {

    Preconditions.checkNotNull(httpClientConfig);
    return createClient(httpClientConfig,isSecured);
  }

  private static OkHttpClient createClient(final HttpClientConfig httpClientConfig,
                                           final boolean isSecured) {

    int connections = httpClientConfig.getConnections();
    int idleTimeOutSeconds = httpClientConfig.getIdleTimeOutSeconds();
    int connTimeout = httpClientConfig.getConnectTimeoutMs();
    int opTimeout = httpClientConfig.getOpTimeoutMs();

    Dispatcher dispatcher = new Dispatcher();
    dispatcher.setMaxRequests(connections);
    dispatcher.setMaxRequestsPerHost(connections);

    ConnectionSpec connectionSpec = isSecured ?
                                    new ConnectionSpec.Builder(ConnectionSpec.MODERN_TLS)
                                                  .allEnabledTlsVersions()
                                                  .allEnabledCipherSuites()
                                                  .build() : null;

    OkHttpClient.Builder clientBuilder = new OkHttpClient.Builder()
            .addNetworkInterceptor(debitRedirectHandler())
            .followRedirects(false)
            .followSslRedirects(false)
            .retryOnConnectionFailure(true)
            .connectionPool(new ConnectionPool(connections, idleTimeOutSeconds, TimeUnit.SECONDS))
            .connectTimeout(connTimeout, TimeUnit.MILLISECONDS)
            .readTimeout(opTimeout, TimeUnit.MILLISECONDS)
            .writeTimeout(opTimeout, TimeUnit.MILLISECONDS)
            .dispatcher(dispatcher);

    if(connectionSpec != null){
      clientBuilder.connectionSpecs(Collections.singletonList(connectionSpec));
    }
    clientBuilder.hostnameVerifier(OkHostnameVerifier.INSTANCE);
    return clientBuilder.build();
  }

  private static Interceptor debitRedirectHandler(){
    return new Interceptor() {
      @Override
      public Response intercept(Chain chain) throws IOException {
        Request request = chain.request();
        Response response = chain.proceed(request);
        String requestPath = request.url().url().getPath();
        if(response.isRedirect() && (requestPath.contains(HttpUtils.V1_DEBIT_URL) || requestPath.contains(HttpUtils.V3_DEBIT_URL))){

          InitResponse initResponse = InitResponse.builder()
                      .redirectType(RedirectType.WEB)
                      .redirectURL(response.header(HttpUtils.LOCATION_HEADER))
                      .build();
          PhonePeHttpResponse<InitResponse> phonePeHttpResponse = PhonePeHttpResponse.<InitResponse>builder()
                                                                                     .success(true)
                                                                                     .message("SUCCESS")
                                                                                     .code("200")
                                                                                     .data(initResponse)
                                                                                     .build();

          return response.newBuilder()
                  .body(ResponseBody.create(MediaType.parse(HttpUtils.APPLICATION_JSON),
                                            MapperUtils.getMapper().writeValueAsBytes(phonePeHttpResponse)))
                  .code(200)
                  .build();
        }
        return response;
      }
    };
  }
}
