package com.phonepe.sdk.javasdk.transaction.callback;

import com.fasterxml.jackson.core.type.TypeReference;
import com.google.common.base.Preconditions;
import com.phonepe.sdk.javasdk.config.models.MerchantConfig;
import com.phonepe.sdk.javasdk.exception.PhonePeClientException;
import com.phonepe.sdk.javasdk.http.models.PhonePeHttpRequest;
import com.phonepe.sdk.javasdk.http.models.PhonePeHttpResponse;
import com.phonepe.sdk.javasdk.transaction.models.Base64Request;
import com.phonepe.sdk.javasdk.transaction.status.TransactionStatusChecker;
import com.phonepe.sdk.javasdk.transaction.status.models.StatusResponse;
import com.phonepe.sdk.javasdk.transaction.status.models.TransactionStatusResponse;
import com.phonepe.sdk.javasdk.utils.Base64Utils;
import com.phonepe.sdk.javasdk.utils.KeyUtils;
import com.phonepe.sdk.javasdk.utils.MapperUtils;
import lombok.Builder;
import lombok.extern.slf4j.Slf4j;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

@Slf4j
public class CallbackHandler {

    private TransactionStatusChecker transactionStatusChecker;
    private MerchantConfig merchantConfig;

    @Builder
    public CallbackHandler(final TransactionStatusChecker transactionStatusChecker,
                           final MerchantConfig merchantConfig) {
        this.transactionStatusChecker = transactionStatusChecker;
        this.merchantConfig = merchantConfig;
    }

    public StatusResponse handleCallback(final String responseReceived, final String checksum, final long transactionAmount) throws PhonePeClientException {
        Preconditions.checkNotNull(responseReceived);
        Preconditions.checkNotNull(checksum);
        try{
            final String responseBody = Base64Utils.decodeBase64(responseReceived);
            final PhonePeHttpResponse<TransactionStatusResponse> transactionStatusResponse = MapperUtils.getMapper()
                                                                                                        .readValue(responseBody, new TypeReference<PhonePeHttpResponse<TransactionStatusResponse>>() {});
            final int keyIndex = KeyUtils.getKeyIndexFromChecksum(checksum);
            final TransactionStatusResponse callbackResponseData = transactionStatusResponse.getData();
            final String merchantId = callbackResponseData.getMerchantId();
            final Long callbackAmount = callbackResponseData.getAmount();
            final List<String> params = getParamsList(callbackResponseData);
            validateChecksum(responseReceived, checksum,params, keyIndex);
            validateTransactionAmount(callbackAmount, transactionAmount);
            validateMerchantId(merchantId);
            final String transactionId = transactionStatusResponse.getData().getTransactionId();
            final StatusResponse statusResponse = this.transactionStatusChecker.checkTransactionStatus(transactionId, keyIndex);
            validateTransactionAmount(callbackAmount, statusResponse.getAmount());
            return statusResponse;
        } catch (PhonePeClientException e){
            log.error("Exception while handling callback: {}", e.getMessage());
            throw e;
        } catch (Exception ex){
            log.error("Error in calling PhonePe init transaction API", ex);
            Map<String, Object> objectMap = new HashMap<String, Object>();
            objectMap.put("MESSAGE", ex.getMessage());
            throw new PhonePeClientException(PhonePeClientException.ErrorCode.HTTP_CLIENT_ERROR,
                                             "Error executing http client: for initiating transaction"
                                             + ": " + ex.getMessage(), objectMap, ex);
        }
    }

    private boolean validateMerchantId(final String merchantId) throws PhonePeClientException{
        if(this.merchantConfig.getMerchantId().equals(merchantId)){
            return true;
        }
        throw new PhonePeClientException(PhonePeClientException.ErrorCode.VALIDATION_ERROR,
                                         "Invalid merchantId in callback response");
    }

    private boolean validateChecksum(final String responseBody, final String checksum, final List<String> params, final int keyIndex) throws PhonePeClientException{
        final String apiKey = KeyUtils.getAPIKeyFromIndex(this.merchantConfig.getApiKeys(),keyIndex);
        final Base64Request base64Request = Base64Request.builder()
                                                         .request(responseBody)
                                                         .build();
        final PhonePeHttpRequest<Base64Request> phonePeHttpRequest = PhonePeHttpRequest.<Base64Request>builder()
                                                                                       .data(base64Request)
                                                                                       .build();
        final String generatedChecksum = this.transactionStatusChecker.getChecksumGenerator().getChecksumValue(phonePeHttpRequest,"",params,apiKey,keyIndex);
        if(checksum.equalsIgnoreCase(generatedChecksum)){
            return true;
        }
        throw new PhonePeClientException(PhonePeClientException.ErrorCode.VALIDATION_ERROR,
                                         "Invalid checksum in callback response");
    }

    private boolean validateTransactionAmount(final Long callbackAmount, final Long statusResponseAmount)throws PhonePeClientException{
        if(callbackAmount.equals(statusResponseAmount)){
            return true;
        }
        throw new PhonePeClientException(PhonePeClientException.ErrorCode.VALIDATION_ERROR,
                                         "Transaction amount mismatch in callback response and status check response or in provided amount");
    }

    private List<String> getParamsList(TransactionStatusResponse transactionStatusResponse){
        final List<String> params = new ArrayList<String>();
        params.add(transactionStatusResponse.getMerchantId());
        params.add(transactionStatusResponse.getTransactionId());
        params.add(transactionStatusResponse.getAmount().toString());
        return params;
    }

}